<?php
/**
 * API Client - handles communication through proxy (hides Claude API key)
 *
 * @package FormRankLS\Core
 */

namespace FormRankLS\Core;

if (!defined('ABSPATH')) {
    exit;
}

class API_Client {

    /**
     * Proxy URL - Cloudflare Worker endpoint
     * Can be overridden via FORMRANK_PROXY_URL constant in wp-config.php
     */
    private $proxy_url = 'https://formrank-proxy.eliecervera.workers.dev';

    /**
     * License key for authentication
     */
    private $license_key;

    /**
     * Site URL for license validation
     */
    private $site_url;

    /**
     * Current usage stats
     */
    private $usage_stats = null;

    public function __construct() {
        $this->license_key = $this->get_license_key();
        $this->site_url = home_url();

        // Allow override of proxy URL via constant or filter
        if (defined('FORMRANK_PROXY_URL')) {
            $this->proxy_url = FORMRANK_PROXY_URL;
        }
        $this->proxy_url = apply_filters('formrank_proxy_url', $this->proxy_url);
    }

    /**
     * Send a scoring request through the proxy
     *
     * @param string $prompt The prompt to send
     * @param int $max_tokens Maximum tokens in response (ignored - controlled by proxy)
     * @return string The response text
     * @throws \Exception On API errors
     */
    public function complete(string $prompt, int $max_tokens = 500): string {
        // Use license key if available, otherwise use free tier mode
        $request_key = $this->license_key;
        if (empty($request_key)) {
            $request_key = 'free_' . md5($this->site_url);
        }

        $response = wp_remote_post($this->proxy_url, [
            'timeout' => 30,
            'headers' => [
                'Content-Type' => 'application/json'
            ],
            'body' => wp_json_encode([
                'action' => 'score',
                'license_key' => $request_key,
                'site_url' => $this->site_url,
                'prompt' => $prompt
            ])
        ]);

        if (is_wp_error($response)) {
            throw new \Exception('Connection failed: ' . esc_html($response->get_error_message()));
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        // Handle errors
        if ($status_code === 401) {
            throw new \Exception('Invalid license. Please check your license key in settings.');
        }

        if ($status_code === 429) {
            $upgrade_url = $body['upgrade_url'] ?? 'https://formrankwp.com/pricing/';
            throw new \Exception(
                sprintf(
                    'Monthly limit reached (%d/%d leads). <a href="%s" target="_blank">Upgrade your plan</a> for more.',
                    absint($body['usage'] ?? 0),
                    absint($body['limit'] ?? 0),
                    esc_url($upgrade_url)
                )
            );
        }

        if ($status_code !== 200 || !isset($body['success'])) {
            $error_msg = $body['error'] ?? 'Unknown error';
            throw new \Exception('Scoring failed: ' . esc_html($error_msg));
        }

        // Update cached usage stats
        if (isset($body['usage']) && isset($body['limit'])) {
            $this->usage_stats = [
                'usage' => $body['usage'],
                'limit' => $body['limit'],
                'remaining' => $body['limit'] - $body['usage']
            ];
            set_transient('formrank_usage_stats', $this->usage_stats, HOUR_IN_SECONDS);
        }

        return $body['response'];
    }

    /**
     * Send an enrichment request through the proxy
     *
     * @param string $name  Lead's full name
     * @param string $email Lead's email address
     * @return array Full decoded response body from Worker
     * @throws \Exception On API errors
     */
    public function enrich(string $name, string $email): array {
        // Use license key if available, otherwise use free tier mode
        $request_key = $this->license_key;
        if (empty($request_key)) {
            $request_key = 'free_' . md5($this->site_url);
        }

        $response = wp_remote_post($this->proxy_url, [
            'timeout' => 90,
            'headers' => [
                'Content-Type' => 'application/json'
            ],
            'body' => wp_json_encode([
                'action'      => 'enrich',
                'license_key' => $request_key,
                'site_url'    => $this->site_url,
                'name'        => $name,
                'email'       => $email,
            ])
        ]);

        if (is_wp_error($response)) {
            throw new \Exception('Connection failed: ' . esc_html($response->get_error_message()));
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body        = json_decode(wp_remote_retrieve_body($response), true);

        // Handle errors
        if ($status_code === 401) {
            throw new \Exception(
                __('Invalid license. Please check your license key in settings.', 'formrank-lead-scoring')
            );
        }

        if ($status_code === 403) {
            $upgrade_url = $body['upgrade_url'] ?? 'https://formrankwp.com/pricing/';
            throw new \Exception(
                sprintf(
                    /* translators: %s: upgrade URL */
                    __('Lead enrichment requires a PRO plan. <a href="%s" target="_blank">Upgrade now</a>.', 'formrank-lead-scoring'),
                    esc_url($upgrade_url)
                )
            );
        }

        if ($status_code === 429) {
            $upgrade_url = $body['upgrade_url'] ?? 'https://formrankwp.com/pricing/';
            throw new \Exception(
                sprintf(
                    /* translators: 1: current usage count, 2: monthly limit, 3: upgrade URL */
                    __('Monthly enrichment limit reached (%1$d/%2$d). <a href="%3$s" target="_blank">Upgrade your plan</a> for more.', 'formrank-lead-scoring'),
                    absint($body['usage'] ?? 0),
                    absint($body['limit'] ?? 0),
                    esc_url($upgrade_url)
                )
            );
        }

        if ($status_code !== 200 || !isset($body['success'])) {
            $error_msg = $body['error'] ?? 'Unknown error';
            throw new \Exception('Enrichment failed: ' . esc_html($error_msg));
        }

        return $body;
    }

    /**
     * Get license key from database or Freemius
     */
    private function get_license_key(): string {
        // First check if Freemius is active and has a license
        if (function_exists('\FormRankLS\fr_fs') && \FormRankLS\fr_fs()->is_registered()) {
            $license = \FormRankLS\fr_fs()->_get_license();
            if ($license && !empty($license->id)) {
                // Send numeric license ID - the proxy uses this in the
                // Freemius API URL path which expects a numeric ID,
                // not the secret_key string.
                return (string) $license->id;
            }
        }

        // Fallback to stored license key
        return get_option('formrank_license_key', '');
    }

    /**
     * Validate license with proxy
     *
     * @return array Validation result
     */
    public function validate_license(): array {
        if (empty($this->license_key)) {
            // Free tier - no license needed for basic access
            return [
                'valid' => true,
                'plan' => 'free',
                'usage' => 0,
                'limit' => 15,
                'remaining' => 15
            ];
        }

        $response = wp_remote_post($this->proxy_url, [
            'timeout' => 15,
            'headers' => [
                'Content-Type' => 'application/json'
            ],
            'body' => wp_json_encode([
                'action' => 'validate',
                'license_key' => $this->license_key,
                'site_url' => $this->site_url
            ])
        ]);

        if (is_wp_error($response)) {
            return [
                'valid' => false,
                'error' => 'Connection failed: ' . $response->get_error_message()
            ];
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (!empty($body['valid'])) {
            // Cache the usage stats
            $this->usage_stats = [
                'usage' => $body['usage'] ?? 0,
                'limit' => $body['limit'] ?? 50,
                'remaining' => $body['remaining'] ?? 50,
                'plan' => $body['plan'] ?? 'free'
            ];
            set_transient('formrank_usage_stats', $this->usage_stats, HOUR_IN_SECONDS);
        }

        return $body;
    }

    /**
     * Get current usage statistics
     *
     * @param bool $force_refresh Force refresh from server
     * @return array Usage stats
     */
    public function get_usage_stats(bool $force_refresh = false): array {
        // Return cached if available and not forcing refresh
        if (!$force_refresh) {
            $cached = get_transient('formrank_usage_stats');
            if ($cached !== false) {
                return $cached;
            }
        }

        if (empty($this->license_key)) {
            return [
                'usage' => 0,
                'limit' => 15,
                'remaining' => 15,
                'plan' => 'free'
            ];
        }

        $response = wp_remote_post($this->proxy_url, [
            'timeout' => 15,
            'headers' => [
                'Content-Type' => 'application/json'
            ],
            'body' => wp_json_encode([
                'action' => 'usage',
                'license_key' => $this->license_key,
                'site_url' => $this->site_url
            ])
        ]);

        if (is_wp_error($response)) {
            return [
                'usage' => 0,
                'limit' => 0,
                'remaining' => 0,
                'error' => $response->get_error_message()
            ];
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        // Cache for 1 hour
        set_transient('formrank_usage_stats', $body, HOUR_IN_SECONDS);

        return $body;
    }

    /**
     * Test connection to proxy
     *
     * @return array Test result
     */
    public function test_connection(): array {
        $result = $this->validate_license();

        if (!empty($result['valid'])) {
            return [
                'success' => true,
                'message' => sprintf(
                    'Connected! Plan: %s | Usage: %d/%d leads this month',
                    ucfirst($result['plan'] ?? 'free'),
                    $result['usage'] ?? 0,
                    $result['limit'] ?? 50
                ),
                'plan' => $result['plan'] ?? 'free'
            ];
        }

        return [
            'success' => false,
            'message' => $result['error'] ?? 'Connection failed'
        ];
    }

    /**
     * Check if license is configured
     */
    public function has_license(): bool {
        return !empty($this->license_key);
    }

    /**
     * Check if using free plan
     */
    public function is_free_plan(): bool {
        $stats = $this->get_usage_stats();
        return ($stats['plan'] ?? 'free') === 'free';
    }

    /**
     * Get remaining leads this month
     */
    public function get_remaining_leads(): int {
        $stats = $this->get_usage_stats();
        return max(0, ($stats['limit'] ?? 0) - ($stats['usage'] ?? 0));
    }

    /**
     * Save license key (for manual entry fallback)
     */
    public function save_license_key(string $license_key): bool {
        if (empty($license_key)) {
            delete_option('formrank_license_key');
            delete_transient('formrank_usage_stats');
            return true;
        }

        // Validate format (Freemius license keys)
        if (!$this->validate_license_key_format($license_key)) {
            return false;
        }

        update_option('formrank_license_key', $license_key);
        $this->license_key = $license_key;

        // Validate with server
        $result = $this->validate_license();
        return !empty($result['valid']);
    }

    /**
     * Validate license key format
     */
    public function validate_license_key_format(string $key): bool {
        // Freemius numeric license ID
        if (preg_match('/^\d+$/', $key) === 1) {
            return true;
        }
        // Freemius license keys: 32 chars hex, UUID format, or sk_ prefixed
        return preg_match('/^[a-f0-9-]{32,36}$/i', $key) === 1 ||
               preg_match('/^sk_[a-zA-Z0-9]+$/', $key) === 1;
    }

    /**
     * Get available models (informational only - model is set server-side)
     */
    public static function get_available_models(): array {
        return [
            'auto' => 'Automatic (Optimized for cost & quality)'
        ];
    }

    /**
     * Get current model (always auto for proxy)
     */
    public function get_model(): string {
        return 'auto';
    }

    // =========================================================================
    // DEPRECATED METHODS - Kept for backwards compatibility
    // =========================================================================

    /**
     * @deprecated Use has_license() instead
     */
    public function has_api_key(): bool {
        return $this->has_license();
    }

    /**
     * @deprecated API key is now managed server-side
     */
    public function save_api_key(string $api_key): bool {
        // No-op - API key is managed on the server
        return false;
    }

    /**
     * @deprecated API key format no longer applicable
     */
    public function validate_api_key_format(string $key): bool {
        return $this->validate_license_key_format($key);
    }
}
