<?php
/**
 * Privacy Handler - GDPR compliance for personal data export and erasure
 *
 * @package FormRankLS
 * @since 1.1.0
 */

namespace FormRankLS;

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Privacy {

    /**
     * Initialize privacy hooks
     */
    public function __construct() {
        // Register data exporter
        add_filter('wp_privacy_personal_data_exporters', [$this, 'register_exporter']);

        // Register data eraser
        add_filter('wp_privacy_personal_data_erasers', [$this, 'register_eraser']);

        // Add privacy policy suggested text
        add_action('admin_init', [$this, 'add_privacy_policy_content']);
    }

    /**
     * Register the personal data exporter
     *
     * @param array $exporters Registered exporters
     * @return array Modified exporters
     */
    public function register_exporter(array $exporters): array {
        $exporters['formrank-lead-scoring'] = [
            'exporter_friendly_name' => __('FormRank Lead Data', 'formrank-lead-scoring'),
            'callback' => [$this, 'export_personal_data'],
        ];

        return $exporters;
    }

    /**
     * Register the personal data eraser
     *
     * @param array $erasers Registered erasers
     * @return array Modified erasers
     */
    public function register_eraser(array $erasers): array {
        $erasers['formrank-lead-scoring'] = [
            'eraser_friendly_name' => __('FormRank Lead Data', 'formrank-lead-scoring'),
            'callback' => [$this, 'erase_personal_data'],
        ];

        return $erasers;
    }

    /**
     * Export personal data for a given email address
     *
     * @param string $email_address The email address to export data for
     * @param int $page Page number for pagination
     * @return array Export data
     */
    public function export_personal_data(string $email_address, int $page = 1): array {
        global $wpdb;

        $table = $wpdb->prefix . 'formrank_leads';
        $per_page = 50;
        $offset = ($page - 1) * $per_page;

        // Get leads for this email address
        $leads = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$table} WHERE email = %s LIMIT %d OFFSET %d",
                $email_address,
                $per_page,
                $offset
            ),
            ARRAY_A
        );

        $export_items = [];

        // Batch-load enrichments for all leads on this page (avoids N+1 queries).
        $enrichments_by_lead = [];
        $enrichments_table   = $wpdb->prefix . 'formrank_enrichments';

        if (!empty($leads)) {
            // Check that the enrichments table exists before querying it.
            $table_exists = $wpdb->get_var(
                $wpdb->prepare('SHOW TABLES LIKE %s', $enrichments_table)
            );

            if ($table_exists) {
                $lead_ids         = array_map('absint', array_column($leads, 'id'));
                $ids_placeholder  = implode(',', $lead_ids);

                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- IDs are absint-sanitized.
                $enrichments = $wpdb->get_results(
                    "SELECT * FROM {$enrichments_table} WHERE lead_id IN ({$ids_placeholder})",
                    ARRAY_A
                );

                foreach ($enrichments as $enrichment) {
                    $enrichments_by_lead[ (int) $enrichment['lead_id'] ] = $enrichment;
                }
            }
        }

        foreach ($leads as $lead) {
            $data = [
                [
                    'name' => __('Email Address', 'formrank-lead-scoring'),
                    'value' => $lead['email'] ?: '',
                ],
                [
                    'name' => __('Name', 'formrank-lead-scoring'),
                    'value' => $lead['name'] ?: '',
                ],
                [
                    'name' => __('Company', 'formrank-lead-scoring'),
                    'value' => $lead['company'] ?: '',
                ],
                [
                    'name' => __('Phone', 'formrank-lead-scoring'),
                    'value' => $lead['phone'] ?: '',
                ],
                [
                    'name' => __('Lead Score', 'formrank-lead-scoring'),
                    'value' => $lead['score'] ? $lead['score'] . ' (' . $lead['score_label'] . ')' : '',
                ],
                [
                    'name' => __('Score Reasoning', 'formrank-lead-scoring'),
                    'value' => $lead['score_reasoning'] ?: '',
                ],
                [
                    'name' => __('Form Source', 'formrank-lead-scoring'),
                    'value' => ucfirst($lead['form_plugin']) . ' #' . $lead['form_id'],
                ],
                [
                    'name' => __('Source URL', 'formrank-lead-scoring'),
                    'value' => $lead['source_url'] ?: '',
                ],
                [
                    'name' => __('IP Address', 'formrank-lead-scoring'),
                    'value' => $lead['ip_address'] ?: '',
                ],
                [
                    'name' => __('User Agent', 'formrank-lead-scoring'),
                    'value' => $lead['user_agent'] ?: '',
                ],
                [
                    'name' => __('Status', 'formrank-lead-scoring'),
                    'value' => ucfirst($lead['status']),
                ],
                [
                    'name' => __('Submission Date', 'formrank-lead-scoring'),
                    'value' => $lead['created_at'],
                ],
            ];

            // Add form data fields
            $form_data = json_decode($lead['form_data'], true);
            if (is_array($form_data)) {
                foreach ($form_data as $field_name => $field_value) {
                    if (is_array($field_value)) {
                        $field_value = implode(', ', $field_value);
                    }
                    $data[] = [
                        /* translators: %s: form field name */
                    'name' => sprintf(__('Form Field: %s', 'formrank-lead-scoring'), ucwords(str_replace(['_', '-'], ' ', $field_name))),
                        'value' => $field_value,
                    ];
                }
            }

            // Add notes if present
            if (!empty($lead['notes'])) {
                $data[] = [
                    'name' => __('Admin Notes', 'formrank-lead-scoring'),
                    'value' => $lead['notes'],
                ];
            }

            // Add enrichment fields if this lead has been enriched (batch-loaded above).
            $enrichment = $enrichments_by_lead[ (int) $lead['id'] ] ?? null;
            if ($enrichment) {
                $enrichment_fields = [
                    'job_title'        => __('Job Title (Enriched)', 'formrank-lead-scoring'),
                    'company_name'     => __('Company Name (Enriched)', 'formrank-lead-scoring'),
                    'seniority_level'  => __('Seniority Level (Enriched)', 'formrank-lead-scoring'),
                    'linkedin_url'     => __('LinkedIn URL (Enriched)', 'formrank-lead-scoring'),
                    'company_industry' => __('Company Industry (Enriched)', 'formrank-lead-scoring'),
                    'company_size'     => __('Company Size (Enriched)', 'formrank-lead-scoring'),
                ];

                foreach ($enrichment_fields as $col => $label) {
                    if (!empty($enrichment[ $col ])) {
                        $data[] = [
                            'name'  => $label,
                            'value' => $enrichment[ $col ],
                        ];
                    }
                }
            }

            $export_items[] = [
                'group_id' => 'formrank-leads',
                'group_label' => __('Lead Submissions', 'formrank-lead-scoring'),
                'item_id' => 'lead-' . $lead['id'],
                'data' => $data,
            ];
        }

        // Check if there are more leads to export
        $total_leads = (int) $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM {$table} WHERE email = %s",
                $email_address
            )
        );

        $done = ($offset + $per_page) >= $total_leads;

        return [
            'data' => $export_items,
            'done' => $done,
        ];
    }

    /**
     * Erase personal data for a given email address
     *
     * @param string $email_address The email address to erase data for
     * @param int $page Page number for pagination
     * @return array Erasure result
     */
    public function erase_personal_data(string $email_address, int $page = 1): array {
        global $wpdb;

        $table = $wpdb->prefix . 'formrank_leads';
        $per_page = 50;

        // Count total leads for this email
        $total_leads = (int) $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM {$table} WHERE email = %s",
                $email_address
            )
        );

        if ($total_leads === 0) {
            return [
                'items_removed' => 0,
                'items_retained' => false,
                'messages' => [],
                'done' => true,
            ];
        }

        // CRITICAL: Collect lead IDs BEFORE deleting leads so we can cascade to enrichments.
        // Fetch all lead IDs for this email (no LIMIT — we need every ID before deletion).
        $lead_ids = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT id FROM {$table} WHERE email = %s",
                $email_address
            )
        );

        $items_removed = 0;

        // Delete enrichment rows that belong to these leads.
        // Guard with SHOW TABLES in case the enrichments table does not yet exist
        // (e.g., plugin installed before v1.3.0 migration ran).
        if (!empty($lead_ids)) {
            $enrichments_table = $wpdb->prefix . 'formrank_enrichments';
            $table_exists      = $wpdb->get_var(
                $wpdb->prepare('SHOW TABLES LIKE %s', $enrichments_table)
            );

            if ($table_exists) {
                $ids_placeholder = implode(',', array_map('absint', $lead_ids));
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- IDs are absint-sanitized.
                $enrichments_deleted = $wpdb->query(
                    "DELETE FROM {$enrichments_table} WHERE lead_id IN ({$ids_placeholder})"
                );

                if ($enrichments_deleted !== false) {
                    $items_removed += (int) $enrichments_deleted;
                }
            }
        }

        // Delete leads in batches to avoid timeout
        $deleted = $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$table} WHERE email = %s LIMIT %d",
                $email_address,
                $per_page
            )
        );

        $items_removed += $deleted !== false ? (int) $deleted : 0;

        // Check if there are more leads to delete
        $remaining = (int) $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM {$table} WHERE email = %s",
                $email_address
            )
        );

        $done = $remaining === 0;

        $messages = [];
        if ($items_removed > 0) {
            $messages[] = sprintf(
                /* translators: %d: number of leads removed */
                _n(
                    'Removed %d lead submission from FormRank.',
                    'Removed %d lead submissions from FormRank.',
                    $items_removed,
                    'formrank-lead-scoring'
                ),
                $items_removed
            );
        }

        return [
            'items_removed' => $items_removed,
            'items_retained' => false,
            'messages' => $messages,
            'done' => $done,
        ];
    }

    /**
     * Add privacy policy suggested text
     */
    public function add_privacy_policy_content(): void {
        if (!function_exists('wp_add_privacy_policy_content')) {
            return;
        }

        $content = $this->get_privacy_policy_text();

        wp_add_privacy_policy_content(
            'FormRank Lead Scoring',
            wp_kses_post($content)
        );
    }

    /**
     * Get the privacy policy suggested text
     *
     * @return string Privacy policy HTML content
     */
    private function get_privacy_policy_text(): string {
        $content = '<h2>' . esc_html__('Lead Data Collection', 'formrank-lead-scoring') . '</h2>';

        $content .= '<p>' . esc_html__('When you submit a form on our website, we collect the information you provide, which may include:', 'formrank-lead-scoring') . '</p>';

        $content .= '<ul>';
        $content .= '<li>' . esc_html__('Your name and contact information (email address, phone number)', 'formrank-lead-scoring') . '</li>';
        $content .= '<li>' . esc_html__('Your company or organization name', 'formrank-lead-scoring') . '</li>';
        $content .= '<li>' . esc_html__('The content of your message or inquiry', 'formrank-lead-scoring') . '</li>';
        $content .= '<li>' . esc_html__('Technical information such as your IP address and browser type', 'formrank-lead-scoring') . '</li>';
        $content .= '<li>' . esc_html__('The page URL where you submitted the form', 'formrank-lead-scoring') . '</li>';
        $content .= '</ul>';

        $content .= '<h2>' . esc_html__('How We Use Your Data', 'formrank-lead-scoring') . '</h2>';

        $content .= '<p>' . esc_html__('We use the information collected through forms to:', 'formrank-lead-scoring') . '</p>';

        $content .= '<ul>';
        $content .= '<li>' . esc_html__('Respond to your inquiries and provide requested information', 'formrank-lead-scoring') . '</li>';
        $content .= '<li>' . esc_html__('Assess the relevance of your inquiry to our products or services (lead scoring)', 'formrank-lead-scoring') . '</li>';
        $content .= '<li>' . esc_html__('Prioritize and manage communications with potential customers', 'formrank-lead-scoring') . '</li>';
        $content .= '<li>' . esc_html__('Send notifications to our team about high-priority inquiries', 'formrank-lead-scoring') . '</li>';
        $content .= '</ul>';

        $content .= '<h2>' . esc_html__('Lead Scoring', 'formrank-lead-scoring') . '</h2>';

        $content .= '<p>' . esc_html__('We use automated lead scoring technology to analyze form submissions. This process:', 'formrank-lead-scoring') . '</p>';

        $content .= '<ul>';
        $content .= '<li>' . esc_html__('Evaluates the content of your submission to determine relevance to our business', 'formrank-lead-scoring') . '</li>';
        $content .= '<li>' . esc_html__('Assigns a priority score to help our team respond efficiently', 'formrank-lead-scoring') . '</li>';
        $content .= '<li>' . esc_html__('Does not make automated decisions that significantly affect you', 'formrank-lead-scoring') . '</li>';
        $content .= '<li>' . esc_html__('All scoring is processed locally on your WordPress server by default. If AI scoring is enabled, form data may be sent to external AI services for analysis.', 'formrank-lead-scoring') . '</li>';
        $content .= '</ul>';

        $content .= '<h2>' . esc_html__('Lead Enrichment (PRO Feature)', 'formrank-lead-scoring') . '</h2>';

        $content .= '<p>' . esc_html__('When a PRO user triggers lead enrichment, the lead\'s name and email address are sent to Anthropic\'s Claude AI API via a secure FormRank Cloudflare Worker proxy. The AI performs web-based research to identify job title, company, LinkedIn URL, and related professional information. No enrichment data is stored by Anthropic beyond the time needed to process the request.', 'formrank-lead-scoring') . '</p>';

        $content .= '<p>' . sprintf(
            /* translators: %s: Anthropic privacy policy URL */
            esc_html__('For more information, see the %sAnthropic Privacy Policy%s.', 'formrank-lead-scoring'),
            '<a href="https://www.anthropic.com/privacy" target="_blank" rel="noopener noreferrer">',
            '</a>'
        ) . '</p>';

        $content .= '<h2>' . esc_html__('Data Storage and Security', 'formrank-lead-scoring') . '</h2>';

        $content .= '<p>' . esc_html__('Your form submission data is stored securely in our website database. We implement appropriate technical and organizational measures to protect your personal data against unauthorized access, alteration, disclosure, or destruction.', 'formrank-lead-scoring') . '</p>';

        $content .= '<h2>' . esc_html__('Your Rights', 'formrank-lead-scoring') . '</h2>';

        $content .= '<p>' . esc_html__('Under applicable data protection laws, you have the right to:', 'formrank-lead-scoring') . '</p>';

        $content .= '<ul>';
        $content .= '<li>' . esc_html__('Request access to your personal data', 'formrank-lead-scoring') . '</li>';
        $content .= '<li>' . esc_html__('Request correction or deletion of your personal data', 'formrank-lead-scoring') . '</li>';
        $content .= '<li>' . esc_html__('Request that we export your personal data in a portable format', 'formrank-lead-scoring') . '</li>';
        $content .= '<li>' . esc_html__('Object to the processing of your personal data', 'formrank-lead-scoring') . '</li>';
        $content .= '</ul>';

        $content .= '<p>' . esc_html__('To exercise these rights, please contact us using the contact information provided on our website, or use the data request tools available in your account (if applicable).', 'formrank-lead-scoring') . '</p>';

        return $content;
    }
}
