<?php

/**
 * Freemius SDK Integration
 *
 * Handles licensing, payments, and updates through Freemius.
 *
 * @package FormRankLS
 */
namespace FormRankLS;

// Prevent direct access
if ( !defined( 'ABSPATH' ) ) {
    exit;
}
/**
 * Initialize Freemius SDK
 *
 * @return \Freemius
 */
function fr_fs() {
    global $fr_fs;
    if ( !isset( $fr_fs ) ) {
        // Include Freemius SDK
        require_once FORMRANK_LS_PATH . 'vendor/freemius/start.php';
        $fr_fs = fs_dynamic_init( [
            'id'               => '23789',
            'slug'             => 'formrank-lead-scoring',
            'type'             => 'plugin',
            'public_key'       => 'pk_fdbeb5162b0d4a3d9a3d441e121a0',
            'is_premium'       => false,
            'premium_suffix'   => 'Pro',
            'has_addons'       => false,
            'has_paid_plans'   => true,
            'is_org_compliant' => true,
            'menu'             => [
                'slug'       => 'formrank-lead-scoring',
                'first-path' => 'admin.php?page=formrank-lead-scoring',
                'account'    => true,
                'contact'    => true,
                'support'    => true,
            ],
            'is_live'          => true,
        ] );
    }
    return $fr_fs;
}

// Initialize Freemius
fr_fs();
// Hook to handle license activation
fr_fs()->add_action( 'after_license_change', function ( $license_key, $license_state ) {
    // Clear cached usage stats when license changes
    delete_transient( 'formrank_usage_stats' );
    // Store license info
    if ( $license_state === 'activated' ) {
        update_option( 'formrank_license_activated', current_time( 'mysql' ) );
        // Reset AI usage counter so paid users start fresh
        update_option( 'formrank_ai_usage', [] );
        update_option( 'formrank_previous_plan_status', 'pro' );
    } else {
        delete_option( 'formrank_license_activated' );
        update_option( 'formrank_previous_plan_status', 'free' );
    }
} );
// Hook for uninstall cleanup (replaces uninstall.php — Freemius requires this approach)
fr_fs()->add_action( 'after_uninstall', function () {
    global $wpdb;
    // Drop custom tables
    // phpcs:disable WordPress.DB.DirectDatabaseQuery
    $leads_table = esc_sql( $wpdb->prefix . 'formrank_leads' );
    $wpdb->query( "DROP TABLE IF EXISTS `{$leads_table}`" );
    $mappings_table = esc_sql( $wpdb->prefix . 'formrank_form_mappings' );
    $wpdb->query( "DROP TABLE IF EXISTS `{$mappings_table}`" );
    // Delete all plugin options
    $options = [
        'formrank_settings',
        'formrank_api_key',
        'formrank_usage_stats',
        'formrank_ai_usage',
        'formrank_ls_db_version',
        'formrank_license_key',
        'formrank_license_activated',
        'formrank_previous_plan_status',
        'formrank_onboarding_status',
        'formrank_import_permissions',
        'formrank_import_log',
        'formrank_feedback_log'
    ];
    foreach ( $options as $option ) {
        delete_option( $option );
    }
    // Delete any remaining formrank_ options
    $wpdb->query( "DELETE FROM {$wpdb->options} WHERE option_name LIKE 'formrank\\_%'" );
    // Delete transients
    delete_transient( 'formrank_ls_activated' );
    delete_transient( 'formrank_usage_stats' );
    $wpdb->query( "DELETE FROM {$wpdb->options}\n         WHERE option_name LIKE '_transient_formrank\\_%'\n         OR option_name LIKE '_transient_timeout_formrank\\_%'" );
    // phpcs:enable WordPress.DB.DirectDatabaseQuery
    // Clear scheduled cron events
    wp_clear_scheduled_hook( 'formrank_cleanup_old_cache' );
    wp_clear_scheduled_hook( 'formrank_data_retention_cleanup' );
    wp_clear_scheduled_hook( 'formrank_webhook_retry' );
} );
/**
 * Check if user has active premium license
 *
 * @return bool
 */
function formrank_is_premium() {
    return fr_fs()->is_paying();
}

/**
 * Check if user is on trial
 *
 * @return bool
 */
function formrank_is_trial() {
    return fr_fs()->is_trial();
}

/**
 * Get current plan name
 *
 * @return string
 */
function formrank_get_plan() {
    if ( !fr_fs()->is_registered() ) {
        return 'free';
    }
    $license = fr_fs()->_get_license();
    if ( !$license ) {
        return 'free';
    }
    // Map Freemius plan to our plan names
    // Get plan info from license or fall back to plan_id
    $plan_id = $license->plan_id ?? 0;
    // Check plan name directly if available (Freemius provides this)
    if ( !empty( $license->plan ) && is_object( $license->plan ) ) {
        $plan_name = strtolower( $license->plan->name ?? '' );
        if ( strpos( $plan_name, 'agency' ) !== false || strpos( $plan_name, 'business' ) !== false ) {
            return 'agency';
        }
        if ( strpos( $plan_name, 'pro' ) !== false ) {
            return 'pro';
        }
    }
    // Freemius Plan ID mapping
    $plan_map = [
        0     => 'free',
        39724 => 'free',
        39725 => 'pro',
        39726 => 'agency',
    ];
    return $plan_map[$plan_id] ?? 'free';
}

/**
 * Get upgrade URL
 *
 * @param string $plan Target plan
 * @return string
 */
function formrank_get_upgrade_url(  $plan = 'pro'  ) {
    return fr_fs()->get_upgrade_url();
}

/**
 * Add usage stats to account page
 */
fr_fs()->add_filter( 'account/page_hook', function ( $hook ) {
    add_action( $hook, function () {
        $api_client = new \FormRankLS\Core\API_Client();
        $stats = $api_client->get_usage_stats();
        if ( !empty( $stats ) && !isset( $stats['error'] ) ) {
            $usage_percent = ( $stats['limit'] > 0 ? round( $stats['usage'] / $stats['limit'] * 100 ) : 0 );
            echo '<div class="formrank-usage-widget" style="margin: 20px 0; padding: 20px; background: #f9f9f9; border-radius: 8px;">';
            echo '<h3 style="margin-top: 0;">Lead Scoring Usage</h3>';
            echo '<div style="margin-bottom: 10px;">';
            echo '<div style="background: #e0e0e0; border-radius: 4px; height: 20px; overflow: hidden;">';
            echo '<div style="background: ' . (( $usage_percent > 80 ? '#e74c3c' : '#3498db' )) . '; height: 100%; width: ' . esc_attr( min( 100, $usage_percent ) ) . '%;"></div>';
            echo '</div>';
            echo '</div>';
            echo '<p><strong>' . esc_html( $stats['usage'] ) . '</strong> / ' . esc_html( $stats['limit'] ) . ' leads scored this month</p>';
            echo '<p><strong>' . esc_html( $stats['remaining'] ) . '</strong> remaining</p>';
            if ( isset( $stats['resets_at'] ) ) {
                echo '<p style="color: #666; font-size: 12px;">Resets on ' . esc_html( $stats['resets_at'] ) . '</p>';
            }
            echo '</div>';
        }
    } );
    return $hook;
} );
/**
 * Customize Freemius opt-in message
 */
fr_fs()->add_filter(
    'connect_message',
    function (
        $message,
        $user_first_name,
        $plugin_title,
        $user_login,
        $site_link,
        $freemius_link
    ) {
        return sprintf( 
            /* translators: %1$s: user first name */
            __( 'Hey %1$s', 'formrank-lead-scoring' ) . ',<br>' . __( 'Never miss another hot lead! %2$s uses AI to automatically score and qualify your form submissions. To get started, please opt-in to help us improve the plugin.', 'formrank-lead-scoring' ),
            esc_html( $user_first_name ),
            '<b>' . esc_html( $plugin_title ) . '</b>'
         );
    },
    10,
    6
);
/**
 * Custom deactivation survey reasons
 *
 * Tailored to understand why users aren't activating the plugin.
 */
fr_fs()->add_filter( 'uninstall_reasons', function ( $reasons ) {
    return [
        (object) [
            'id'                => 'setup-too-complicated',
            'text'              => __( "I couldn't figure out how to set it up", 'formrank-lead-scoring' ),
            'input_type'        => 'textfield',
            'input_placeholder' => __( 'What part was confusing?', 'formrank-lead-scoring' ),
        ],
        (object) [
            'id'                => 'no-forms-yet',
            'text'              => __( "I don't have enough form submissions to use it yet", 'formrank-lead-scoring' ),
            'input_type'        => '',
            'input_placeholder' => '',
        ],
        (object) [
            'id'                => 'not-what-i-expected',
            'text'              => __( "It wasn't what I expected", 'formrank-lead-scoring' ),
            'input_type'        => 'textfield',
            'input_placeholder' => __( 'What were you hoping it would do?', 'formrank-lead-scoring' ),
        ],
        (object) [
            'id'                => 'my-form-plugin-not-supported',
            'text'              => __( "My form plugin isn't supported", 'formrank-lead-scoring' ),
            'input_type'        => 'textfield',
            'input_placeholder' => __( 'Which form plugin do you use?', 'formrank-lead-scoring' ),
        ],
        (object) [
            'id'                => 'found-alternative',
            'text'              => __( 'I found a better alternative', 'formrank-lead-scoring' ),
            'input_type'        => 'textfield',
            'input_placeholder' => __( 'Which one?', 'formrank-lead-scoring' ),
        ],
        (object) [
            'id'                => 'temporary-deactivation',
            'text'              => __( "It's a temporary deactivation - I'm troubleshooting", 'formrank-lead-scoring' ),
            'input_type'        => '',
            'input_placeholder' => '',
        ],
        (object) [
            'id'                => 'other',
            'text'              => __( 'Other', 'formrank-lead-scoring' ),
            'input_type'        => 'textfield',
            'input_placeholder' => __( 'Please share the reason...', 'formrank-lead-scoring' ),
        ]
    ];
} );
/**
 * Add custom permissions message
 */
fr_fs()->add_filter( 'permission_list', function ( $permissions ) {
    // Customize which permissions are shown/requested
    return $permissions;
} );