<?php
/**
 * Main plugin class
 *
 * @package FormRankLS
 */

namespace FormRankLS;

if (!defined('ABSPATH')) {
    exit;
}

class FormRank_Lead_Scoring {

    protected $loader;
    protected $plugin_name;
    protected $version;

    public function __construct() {
        $this->version = FORMRANK_LS_VERSION;
        $this->plugin_name = 'formrank-lead-scoring';

        $this->load_dependencies();
        $this->define_admin_hooks();
        $this->define_integration_hooks();
        $this->define_cron_hooks();
    }

    private function load_dependencies() {
        // Core classes
        require_once FORMRANK_LS_PATH . 'includes/class-loader.php';
        require_once FORMRANK_LS_PATH . 'includes/core/class-scoring-engine.php';
        require_once FORMRANK_LS_PATH . 'includes/core/class-local-scoring-engine.php';
        require_once FORMRANK_LS_PATH . 'includes/core/class-data-importer.php';
        require_once FORMRANK_LS_PATH . 'includes/core/class-api-client.php';
        require_once FORMRANK_LS_PATH . 'includes/core/class-enrichment-engine.php';
        require_once FORMRANK_LS_PATH . 'includes/core/class-cache-manager.php';
        require_once FORMRANK_LS_PATH . 'includes/core/class-webhook-handler.php';

        // Integration classes
        require_once FORMRANK_LS_PATH . 'includes/integrations/class-integration-base.php';
        require_once FORMRANK_LS_PATH . 'includes/integrations/class-wpforms.php';
        require_once FORMRANK_LS_PATH . 'includes/integrations/class-gravity-forms.php';
        require_once FORMRANK_LS_PATH . 'includes/integrations/class-cf7.php';
        require_once FORMRANK_LS_PATH . 'includes/integrations/class-fluent-forms.php';
        require_once FORMRANK_LS_PATH . 'includes/integrations/class-formidable.php';

        // Admin classes
        require_once FORMRANK_LS_PATH . 'includes/admin/class-admin.php';
        require_once FORMRANK_LS_PATH . 'includes/admin/class-settings.php';
        require_once FORMRANK_LS_PATH . 'includes/admin/class-dashboard.php';
        require_once FORMRANK_LS_PATH . 'includes/admin/class-lead-list.php';
        require_once FORMRANK_LS_PATH . 'includes/admin/class-lead-detail.php';
        require_once FORMRANK_LS_PATH . 'includes/admin/class-data-import.php';
        require_once FORMRANK_LS_PATH . 'includes/admin/class-onboarding.php';
        require_once FORMRANK_LS_PATH . 'includes/admin/class-setup-wizard.php';
        require_once FORMRANK_LS_PATH . 'includes/admin/class-upgrade-notice.php';

        // Privacy/GDPR compliance
        require_once FORMRANK_LS_PATH . 'includes/class-privacy.php';

        // Data retention
        require_once FORMRANK_LS_PATH . 'includes/class-data-retention.php';

        $this->loader = new Loader();
    }

    private function define_admin_hooks() {
        $admin = new Admin\Admin($this->plugin_name, $this->version);

        $this->loader->add_action('admin_menu', $admin, 'add_admin_menu');
        $this->loader->add_action('admin_enqueue_scripts', $admin, 'enqueue_styles');
        $this->loader->add_action('admin_enqueue_scripts', $admin, 'enqueue_scripts');
        $this->loader->add_action('admin_init', $admin, 'register_settings');

        // These must run on admin_init (before any output) so redirects work
        $this->loader->add_action('admin_init', $admin, 'handle_csv_export');
        $this->loader->add_action('admin_init', $admin, 'handle_json_export');
        $this->loader->add_action('admin_init', $admin, 'handle_archive_action');
        $this->loader->add_action('admin_init', $admin, 'handle_bulk_actions');

        // Enrichment notice (shown after first enrichment; dismissible)
        $this->loader->add_action('admin_notices', $admin, 'show_enrichment_notice');

        // AJAX handlers
        $this->loader->add_action('wp_ajax_formrank_test_api', $admin, 'ajax_test_api');
        $this->loader->add_action('wp_ajax_formrank_rescore_lead', $admin, 'ajax_rescore_lead');
        $this->loader->add_action('wp_ajax_formrank_update_status', $admin, 'ajax_update_status');
        $this->loader->add_action('wp_ajax_formrank_save_notes', $admin, 'ajax_save_notes');
        $this->loader->add_action('wp_ajax_formrank_get_stats', $admin, 'ajax_get_stats');
        $this->loader->add_action('wp_ajax_formrank_get_ai_stats', $admin, 'ajax_get_ai_stats');
        $this->loader->add_action('wp_ajax_formrank_get_conversion_analytics', $admin, 'ajax_get_conversion_analytics');
        $this->loader->add_action('wp_ajax_formrank_activate_license', $admin, 'ajax_activate_license');
        $this->loader->add_action('wp_ajax_formrank_deactivate_license', $admin, 'ajax_deactivate_license');
        $this->loader->add_action('wp_ajax_formrank_dismiss_enrichment_notice', $admin, 'ajax_dismiss_enrichment_notice');

        // Initialize Data Import admin page
        new Admin\Data_Import();

        // Initialize Onboarding wizard
        new Admin\Onboarding();

        // Initialize Setup Wizard
        new Admin\Setup_Wizard();

        // Initialize Privacy/GDPR handler
        new Privacy();

        // Initialize Data Retention handler
        new Data_Retention();

        // REST API endpoints
        $this->loader->add_action('rest_api_init', $this, 'register_rest_routes');
    }

    private function define_integration_hooks() {
        // WPForms integration
        if ($this->is_plugin_active('wpforms-lite/wpforms.php') || $this->is_plugin_active('wpforms/wpforms.php')) {
            $wpforms = new Integrations\WPForms();
            $this->loader->add_action('wpforms_process_complete', $wpforms, 'capture_submission', 10, 4);
        }

        // Gravity Forms integration
        if ($this->is_plugin_active('gravityforms/gravityforms.php')) {
            $gravity = new Integrations\Gravity_Forms();
            $this->loader->add_action('gform_after_submission', $gravity, 'capture_submission', 10, 2);
        }

        // Contact Form 7 integration
        // Use wpcf7_before_send_mail instead of wpcf7_mail_sent so submissions
        // are captured even when mail sending fails or is disabled
        if ($this->is_plugin_active('contact-form-7/wp-contact-form-7.php')) {
            $cf7 = new Integrations\CF7();
            $this->loader->add_action('wpcf7_before_send_mail', $cf7, 'capture_submission', 10, 1);
        }

        // Fluent Forms integration
        if ($this->is_plugin_active('fluentform/fluentform.php') || defined('FLUENTFORM')) {
            $fluent = new Integrations\Fluent_Forms();
            $this->loader->add_action('fluentform/submission_inserted', $fluent, 'capture_submission', 10, 3);
        }

        // Formidable Forms integration
        if ($this->is_plugin_active('formidable/formidable.php')) {
            $formidable = new Integrations\Formidable();
            $this->loader->add_action('frm_after_create_entry', $formidable, 'capture_submission', 10, 2);
        }
    }

    private function define_cron_hooks() {
        // Register cron event for async lead scoring
        $this->loader->add_action('formrank_score_lead', new Integrations\Integration_Base(), 'score_lead', 10, 1);

        // Register cron event for webhook retries
        $this->loader->add_action('formrank_webhook_retry', new Core\Webhook_Handler(), 'process_retry', 10, 1);
    }

    private function is_plugin_active($plugin) {
        if (!function_exists('is_plugin_active')) {
            include_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        return is_plugin_active($plugin);
    }

    public function run() {
        $this->loader->run();
    }

    public function get_plugin_name() {
        return $this->plugin_name;
    }

    public function get_version() {
        return $this->version;
    }

    /**
     * Register REST API routes
     */
    public function register_rest_routes(): void {
        register_rest_route('formrank/v1', '/enrich', [
            'methods'             => \WP_REST_Server::CREATABLE,
            'callback'            => [$this, 'rest_enrich_lead'],
            'permission_callback' => function () {
                return current_user_can('manage_options');
            },
            'args' => [
                'lead_id' => [
                    'required'          => true,
                    'type'              => 'integer',
                    'validate_callback' => function ($param) {
                        return is_numeric($param) && absint($param) > 0;
                    },
                    'sanitize_callback' => 'absint',
                ],
            ],
        ]);
    }

    /**
     * REST API callback: Enrich a lead
     *
     * @param \WP_REST_Request $request REST request object
     * @return \WP_REST_Response REST response
     */
    public function rest_enrich_lead(\WP_REST_Request $request): \WP_REST_Response {
        $lead_id = $request->get_param('lead_id');

        try {
            $engine = new Core\Enrichment_Engine();
            $result = $engine->enrich_lead($lead_id);

            // Cache enrichment usage stats in transient for list-page usage counter
            set_transient('formrank_enrich_stats', [
                'usage'     => $result['usage'],
                'limit'     => $result['limit'],
                'remaining' => $result['remaining'],
            ], DAY_IN_SECONDS);

            return new \WP_REST_Response([
                'success'     => true,
                'data'        => $result['enrichment'],
                'usage'       => $result['usage'],
                'limit'       => $result['limit'],
                'remaining'   => $result['remaining'],
                'enriched_at' => $result['enriched_at'],
            ], 200);

        } catch (\Exception $e) {
            $error_message = $e->getMessage();

            // Determine appropriate error code for the response
            $error_code = 'enrichment_failed';
            $status_code = 400;

            if (strpos($error_message, 'not found') !== false && strpos($error_message, 'Lead') !== false) {
                $error_code = 'lead_not_found';
                $status_code = 404;
            } elseif (strpos($error_message, 'PRO') !== false || strpos($error_message, 'Upgrade') !== false) {
                $error_code = 'upgrade_required';
                $status_code = 403;
            } elseif (strpos($error_message, 'limit reached') !== false) {
                $error_code = 'quota_exceeded';
                $status_code = 429;
            } elseif (strpos($error_message, 'wait 30 seconds') !== false) {
                $error_code = 'cooldown_active';
                $status_code = 429;
            } elseif (strpos($error_message, 'Person not found') !== false) {
                $error_code = 'person_not_found';
                $status_code = 404;
            }

            return new \WP_REST_Response([
                'code'    => $error_code,
                'message' => $error_message,
                'data'    => ['status' => $status_code],
            ], $status_code);
        }
    }
}
