<?php
/**
 * Data Retention Handler - GDPR-compliant automatic data cleanup
 *
 * @package FormRankLS
 * @since 1.1.0
 */

namespace FormRankLS;

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Data_Retention {

    /**
     * Cron hook name
     */
    const CRON_HOOK = 'formrank_data_retention_cleanup';

    /**
     * Initialize data retention hooks
     */
    public function __construct() {
        // Register cron action
        add_action(self::CRON_HOOK, [$this, 'run_cleanup']);

        // Schedule cron on settings save
        add_action('update_option_formrank_settings', [$this, 'schedule_cron'], 10, 2);

        // Ensure cron is scheduled on plugin load
        add_action('init', [$this, 'maybe_schedule_cron']);
    }

    /**
     * Maybe schedule cron if not already scheduled
     */
    public function maybe_schedule_cron(): void {
        $settings = get_option('formrank_settings', []);
        $enabled = !empty($settings['retention']['enabled']);
        $days = $settings['retention']['days'] ?? 0;

        if ($enabled && $days > 0) {
            if (!wp_next_scheduled(self::CRON_HOOK)) {
                wp_schedule_event(time(), 'daily', self::CRON_HOOK);
            }
        } else {
            // Unschedule if disabled
            $timestamp = wp_next_scheduled(self::CRON_HOOK);
            if ($timestamp) {
                wp_unschedule_event($timestamp, self::CRON_HOOK);
            }
        }
    }

    /**
     * Schedule or unschedule cron when settings are saved
     *
     * @param mixed $old_value Old settings value
     * @param mixed $new_value New settings value
     */
    public function schedule_cron($old_value, $new_value): void {
        $enabled = !empty($new_value['retention']['enabled']);
        $days = $new_value['retention']['days'] ?? 0;

        // Clear existing schedule
        $timestamp = wp_next_scheduled(self::CRON_HOOK);
        if ($timestamp) {
            wp_unschedule_event($timestamp, self::CRON_HOOK);
        }

        // Schedule new if enabled
        if ($enabled && $days > 0) {
            wp_schedule_event(time() + HOUR_IN_SECONDS, 'daily', self::CRON_HOOK);
        }
    }

    /**
     * Run the cleanup process
     *
     * @return array Cleanup results
     */
    public function run_cleanup(): array {
        global $wpdb;

        $settings = get_option('formrank_settings', []);
        $enabled = !empty($settings['retention']['enabled']);
        $days = absint($settings['retention']['days'] ?? 0);
        $exclude_converted = !empty($settings['retention']['exclude_converted']);

        $results = [
            'success' => false,
            'deleted' => 0,
            'message' => '',
        ];

        // Check if retention is enabled and days is set
        if (!$enabled || $days <= 0) {
            $results['message'] = __('Data retention is disabled.', 'formrank-lead-scoring');
            return $results;
        }

        $table = $wpdb->prefix . 'formrank_leads';
        $cutoff_date = gmdate('Y-m-d H:i:s', strtotime("-{$days} days"));

        // Build WHERE clause
        $where = $wpdb->prepare("created_at < %s", $cutoff_date);

        if ($exclude_converted) {
            $where .= " AND status != 'converted'";
        }

        // Count leads to be deleted (for logging)
        $count_to_delete = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table} WHERE {$where}");

        if ($count_to_delete === 0) {
            $results['success'] = true;
            $results['message'] = __('No leads to delete.', 'formrank-lead-scoring');
            return $results;
        }

        // Delete in batches to prevent timeout
        $batch_size = 100;
        $total_deleted = 0;

        do {
            $deleted = $wpdb->query(
                "DELETE FROM {$table} WHERE {$where} LIMIT {$batch_size}"
            );

            if ($deleted === false) {
                $results['message'] = __('Database error during cleanup.', 'formrank-lead-scoring');
                $results['deleted'] = $total_deleted;
                return $results;
            }

            $total_deleted += $deleted;

            // Prevent infinite loop
            if ($total_deleted >= $count_to_delete) {
                break;
            }

        } while ($deleted > 0);

        $results['success'] = true;
        $results['deleted'] = $total_deleted;
        $results['message'] = sprintf(
            /* translators: %d: number of leads deleted */
            _n(
                '%d lead deleted during retention cleanup.',
                '%d leads deleted during retention cleanup.',
                $total_deleted,
                'formrank-lead-scoring'
            ),
            $total_deleted
        );

        // Log the cleanup
        error_log('FormRank: ' . $results['message']);

        return $results;
    }

    /**
     * Get retention status information
     *
     * @return array Status information
     */
    public function get_status(): array {
        global $wpdb;

        $settings = get_option('formrank_settings', []);
        $enabled = !empty($settings['retention']['enabled']);
        $days = $settings['retention']['days'] ?? 0;
        $exclude_converted = !empty($settings['retention']['exclude_converted']);

        $table = $wpdb->prefix . 'formrank_leads';

        $status = [
            'enabled' => $enabled,
            'days' => $days,
            'exclude_converted' => $exclude_converted,
            'next_run' => null,
            'leads_pending_deletion' => 0,
        ];

        // Get next scheduled run
        $next_run = wp_next_scheduled(self::CRON_HOOK);
        if ($next_run) {
            $status['next_run'] = wp_date(get_option('date_format') . ' ' . get_option('time_format'), $next_run);
        }

        // Count leads pending deletion
        if ($enabled && $days > 0) {
            $cutoff_date = gmdate('Y-m-d H:i:s', strtotime("-{$days} days"));
            $where = $wpdb->prepare("created_at < %s", $cutoff_date);

            if ($exclude_converted) {
                $where .= " AND status != 'converted'";
            }

            $status['leads_pending_deletion'] = (int) $wpdb->get_var(
                "SELECT COUNT(*) FROM {$table} WHERE {$where}"
            );
        }

        return $status;
    }

    /**
     * Manually trigger cleanup (for admin use)
     *
     * @return array Cleanup results
     */
    public function manual_cleanup(): array {
        return $this->run_cleanup();
    }
}
