<?php
/**
 * Plugin activation handler
 *
 * @package FormRankLS
 */

namespace FormRankLS;

if (!defined('ABSPATH')) {
    exit;
}

class Activator {

    /**
     * Current database schema version
     */
    const DB_VERSION = '1.3.0';

    /**
     * Migration methods mapped to version numbers
     */
    private static $migrations = [
        '1.1.0' => 'migrate_to_1_1_0',
        '1.2.0' => 'migrate_to_1_2_0',
        '1.3.0' => 'migrate_to_1_3_0',
    ];

    public static function activate() {
        self::create_tables();
        self::run_migrations();
        self::set_default_options();
        self::schedule_cron_events();

        // Set activation flag for welcome notice
        set_transient('formrank_ls_activated', true, 60);

        // Set wizard redirect flag for new installs only
        if (!get_option('formrank_onboarding_status')) {
            set_transient('formrank_ls_setup_redirect', true, 120);
        }

        // Clear any caches
        wp_cache_flush();
    }

    /**
     * Run database migrations in order
     */
    private static function run_migrations() {
        $current_version = get_option('formrank_ls_db_version', '1.0.0');

        // Skip if already at current version
        if (version_compare($current_version, self::DB_VERSION, '>=')) {
            return;
        }

        // Log migration start
        error_log(sprintf('FormRank: Starting database migration from %s to %s', $current_version, self::DB_VERSION));

        foreach (self::$migrations as $version => $method) {
            // Only run migrations for versions newer than current
            if (version_compare($current_version, $version, '<')) {
                if (method_exists(__CLASS__, $method)) {
                    try {
                        self::$method();
                        // Update version after successful migration
                        update_option('formrank_ls_db_version', $version);
                        error_log(sprintf('FormRank: Migration to %s completed', $version));
                    } catch (\Exception $e) {
                        error_log(sprintf('FormRank: Migration to %s failed: %s', $version, $e->getMessage()));
                        // Stop processing further migrations on failure
                        return;
                    }
                }
            }
        }

        // Update to final version
        update_option('formrank_ls_db_version', self::DB_VERSION);
        error_log('FormRank: All migrations completed successfully');
    }

    /**
     * Migration: Add scoring_method column (v1.1.0)
     */
    private static function migrate_to_1_1_0() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'formrank_leads';

        // Add scoring_method column if it doesn't exist
        $column_exists = $wpdb->get_results("SHOW COLUMNS FROM {$table_name} LIKE 'scoring_method'");

        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE {$table_name} ADD COLUMN scoring_method VARCHAR(20) DEFAULT 'local' AFTER score_factors");
            $wpdb->query("ALTER TABLE {$table_name} ADD INDEX scoring_method (scoring_method)");

            // Set existing leads to 'local' scoring method
            $wpdb->query("UPDATE {$table_name} SET scoring_method = 'local' WHERE scoring_method IS NULL");
        }
    }

    /**
     * Migration: Add conversion tracking columns (v1.2.0)
     */
    private static function migrate_to_1_2_0() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'formrank_leads';

        // Add converted_at column for conversion tracking if not exists
        $column_exists = $wpdb->get_results("SHOW COLUMNS FROM {$table_name} LIKE 'converted_at'");

        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE {$table_name} ADD COLUMN converted_at DATETIME DEFAULT NULL AFTER scored_at");
            $wpdb->query("ALTER TABLE {$table_name} ADD INDEX converted_at (converted_at)");

            // Backfill converted_at for existing converted leads
            $wpdb->query("UPDATE {$table_name} SET converted_at = updated_at WHERE status = 'converted' AND converted_at IS NULL");
        }

        // Add score_at_conversion column to track what score was when converted
        $column_exists = $wpdb->get_results("SHOW COLUMNS FROM {$table_name} LIKE 'score_at_conversion'");

        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE {$table_name} ADD COLUMN score_at_conversion TINYINT UNSIGNED DEFAULT NULL AFTER converted_at");

            // Backfill score_at_conversion for existing converted leads
            $wpdb->query("UPDATE {$table_name} SET score_at_conversion = score WHERE status = 'converted' AND score_at_conversion IS NULL");
        }
    }

    /**
     * Migration: Add enrichments table (v1.3.0)
     */
    private static function migrate_to_1_3_0(): void {
        self::create_enrichments_table();
    }

    private static function create_tables() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'formrank_leads';
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$table_name} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            form_plugin VARCHAR(50) NOT NULL,
            form_id BIGINT(20) UNSIGNED NOT NULL,
            entry_id BIGINT(20) UNSIGNED DEFAULT NULL,
            email VARCHAR(255) DEFAULT NULL,
            name VARCHAR(255) DEFAULT NULL,
            company VARCHAR(255) DEFAULT NULL,
            phone VARCHAR(50) DEFAULT NULL,
            form_data LONGTEXT NOT NULL,
            score TINYINT UNSIGNED DEFAULT NULL,
            score_label VARCHAR(20) DEFAULT NULL,
            score_reasoning TEXT DEFAULT NULL,
            score_factors LONGTEXT DEFAULT NULL,
            scoring_method VARCHAR(20) DEFAULT 'local',
            source_url VARCHAR(2048) DEFAULT NULL,
            user_agent VARCHAR(512) DEFAULT NULL,
            ip_address VARCHAR(45) DEFAULT NULL,
            status VARCHAR(20) DEFAULT 'new',
            tags VARCHAR(255) DEFAULT NULL,
            notes TEXT DEFAULT NULL,
            created_at DATETIME NOT NULL,
            updated_at DATETIME NOT NULL,
            scored_at DATETIME DEFAULT NULL,
            converted_at DATETIME DEFAULT NULL,
            score_at_conversion TINYINT UNSIGNED DEFAULT NULL,
            PRIMARY KEY (id),
            KEY form_plugin_form_id (form_plugin, form_id),
            KEY score (score),
            KEY status (status),
            KEY created_at (created_at),
            KEY email (email(191)),
            KEY scoring_method (scoring_method),
            KEY converted_at (converted_at)
        ) {$charset_collate};";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql);

        self::create_enrichments_table();

        // Store database version
        update_option('formrank_ls_db_version', self::DB_VERSION);
    }

    /**
     * Create the enrichments table for AI-enriched lead data
     */
    private static function create_enrichments_table(): void {
        global $wpdb;

        $table_name = $wpdb->prefix . 'formrank_enrichments';
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$table_name} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            lead_id BIGINT(20) UNSIGNED NOT NULL,
            job_title VARCHAR(255) DEFAULT NULL,
            company_name VARCHAR(255) DEFAULT NULL,
            seniority_level VARCHAR(50) DEFAULT NULL,
            linkedin_url VARCHAR(2048) DEFAULT NULL,
            company_industry VARCHAR(255) DEFAULT NULL,
            company_size VARCHAR(100) DEFAULT NULL,
            confidence VARCHAR(10) NOT NULL DEFAULT 'low',
            email_type VARCHAR(10) NOT NULL DEFAULT 'personal',
            enrichment_data LONGTEXT DEFAULT NULL,
            enriched_at DATETIME NOT NULL,
            updated_at DATETIME NOT NULL,
            PRIMARY KEY  (id),
            UNIQUE KEY lead_id (lead_id),
            KEY confidence (confidence)
        ) {$charset_collate};";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql);
    }

    private static function set_default_options() {
        $default_settings = [
            'enabled_forms' => [
                'wpforms' => [],
                'gravity' => [],
                'cf7' => []
            ],
            'scoring' => [
                'business_context' => '',
                'ideal_customer' => '',
                'score_threshold_hot' => 80,
                'score_threshold_warm' => 60,
                'score_threshold_neutral' => 40,
                'score_threshold_cool' => 20
            ],
            'notifications' => [
                'email_enabled' => false,
                'email_recipients' => get_option('admin_email'),
                'hot_lead_threshold' => 80,
                'slack_webhook' => ''
            ],
            'webhooks' => [],
            'advanced' => [
                'api_model' => 'claude-3-haiku-20240307',
                'cache_duration' => 3600,
                'rate_limit' => 100
            ]
        ];

        // Only set if not exists
        if (!get_option('formrank_settings')) {
            add_option('formrank_settings', $default_settings);
        }

        // Initialize usage stats
        if (!get_option('formrank_usage_stats')) {
            add_option('formrank_usage_stats', []);
        }

        // Initialize AI rescore usage tracking
        if (!get_option('formrank_ai_usage')) {
            add_option('formrank_ai_usage', []);
        }
    }

    private static function schedule_cron_events() {
        if (!wp_next_scheduled('formrank_cleanup_old_cache')) {
            wp_schedule_event(time(), 'daily', 'formrank_cleanup_old_cache');
        }
    }
}
