<?php
/**
 * Upgrade Notice Handler
 *
 * Displays upgrade banners and CTAs for free users.
 *
 * @package FormRankLS\Admin
 */

namespace FormRankLS\Admin;

if (!defined('ABSPATH')) {
    exit;
}

class Upgrade_Notice {

    /**
     * Check if user is on free plan
     *
     * @return bool
     */
    public static function is_free_user() {
        if (function_exists('FormRankLS\fr_fs')) {
            $fs = \FormRankLS\fr_fs();
            return !$fs->is_paying();
        }
        return true;
    }

    /**
     * Get upgrade URL
     *
     * @return string
     */
    public static function get_upgrade_url() {
        // Try Freemius upgrade URL first
        if (function_exists('FormRankLS\fr_fs')) {
            $fs = \FormRankLS\fr_fs();

            // Use Freemius upgrade URL (works for both registered and anonymous users)
            return $fs->get_upgrade_url();
        }

        // Fallback to pricing page on website
        return 'https://formrankwp.com/pricing/';
    }

    /**
     * Get current plan name
     *
     * @return string
     */
    public static function get_plan_name() {
        if (function_exists('FormRankLS\fr_fs')) {
            $fs = \FormRankLS\fr_fs();
            if ($fs->is_paying()) {
                $plan = $fs->get_plan_name();
                return ucfirst($plan ?: 'Pro');
            }
        }
        return 'Free';
    }

    /**
     * Render the top marketing banner for dashboard/lead pages
     *
     * @param string $context Context where banner is shown (dashboard, leads)
     */
    public static function render_top_banner($context = 'dashboard') {
        if (!self::is_free_user()) {
            return;
        }

        $upgrade_url = self::get_upgrade_url();

        $banners = [
            'dashboard' => [
                'title' => __('Unlock Unlimited AI Lead Scoring', 'formrank-lead-scoring'),
                'description' => __('Upgrade to Pro for unlimited AI rescores, advanced Claude AI scoring, Slack notifications, and priority support.', 'formrank-lead-scoring'),
                'cta' => __('Upgrade to Pro', 'formrank-lead-scoring'),
                'features' => [
                    __('Unlimited AI rescores', 'formrank-lead-scoring'),
                    __('Advanced Claude AI scoring', 'formrank-lead-scoring'),
                    __('Slack notifications', 'formrank-lead-scoring'),
                    __('Priority email support', 'formrank-lead-scoring'),
                ],
            ],
            'leads' => [
                'title' => __('Get More from Your Leads', 'formrank-lead-scoring'),
                'description' => __('Upgrade to Pro for unlimited AI scoring, advanced webhooks, and connect with your favorite CRM tools.', 'formrank-lead-scoring'),
                'cta' => __('Upgrade Now', 'formrank-lead-scoring'),
                'features' => [
                    __('Unlimited AI rescores', 'formrank-lead-scoring'),
                    __('CRM integrations', 'formrank-lead-scoring'),
                    __('Custom webhooks', 'formrank-lead-scoring'),
                    __('Export to CSV', 'formrank-lead-scoring'),
                ],
            ],
        ];

        $banner = $banners[$context] ?? $banners['dashboard'];
        ?>
        <div class="formrank-upgrade-banner">
            <div class="formrank-upgrade-banner-content">
                <div class="formrank-upgrade-banner-text">
                    <div class="formrank-upgrade-badge">
                        <span class="dashicons dashicons-star-filled"></span>
                        <?php esc_html_e('PRO', 'formrank-lead-scoring'); ?>
                    </div>
                    <h3><?php echo esc_html($banner['title']); ?></h3>
                    <p><?php echo esc_html($banner['description']); ?></p>
                    <ul class="formrank-upgrade-features">
                        <?php foreach ($banner['features'] as $feature) : ?>
                            <li>
                                <span class="dashicons dashicons-yes-alt"></span>
                                <?php echo esc_html($feature); ?>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                </div>
                <div class="formrank-upgrade-banner-action">
                    <div class="formrank-upgrade-price">
                        <span class="formrank-price-amount">$149</span>
                        <span class="formrank-price-period">/year</span>
                    </div>
                    <a href="<?php echo esc_url($upgrade_url); ?>" class="button button-primary formrank-upgrade-btn">
                        <?php echo esc_html($banner['cta']); ?>
                        <span class="dashicons dashicons-arrow-right-alt"></span>
                    </a>
                    <p class="formrank-upgrade-guarantee">
                        <span class="dashicons dashicons-shield"></span>
                        <?php esc_html_e('14-day money-back guarantee', 'formrank-lead-scoring'); ?>
                    </p>
                </div>
            </div>
            <button type="button" class="formrank-banner-dismiss" data-banner="top-<?php echo esc_attr($context); ?>">
                <span class="dashicons dashicons-no-alt"></span>
            </button>
        </div>
        <?php
    }

    /**
     * Render compact inline upgrade CTA
     *
     * @param string $feature Feature being promoted
     */
    public static function render_inline_cta($feature = 'general') {
        if (!self::is_free_user()) {
            return;
        }

        $upgrade_url = self::get_upgrade_url();

        $messages = [
            'general' => __('Upgrade to Pro for unlimited AI scoring', 'formrank-lead-scoring'),
            'ai_rescore' => __('Unlimited AI rescores with Pro', 'formrank-lead-scoring'),
        ];

        $message = $messages[$feature] ?? $messages['general'];
        ?>
        <div class="formrank-inline-cta">
            <span class="dashicons dashicons-lock"></span>
            <span><?php echo esc_html($message); ?></span>
            <a href="<?php echo esc_url($upgrade_url); ?>" class="formrank-inline-cta-link">
                <?php esc_html_e('Upgrade', 'formrank-lead-scoring'); ?>
                <span class="dashicons dashicons-arrow-right-alt2"></span>
            </a>
        </div>
        <?php
    }

    /**
     * Render settings page upgrade section
     */
    public static function render_settings_upgrade_section() {
        if (!self::is_free_user()) {
            self::render_pro_status();
            return;
        }

        $upgrade_url = self::get_upgrade_url();
        ?>
        <div class="formrank-settings-upgrade-card">
            <div class="formrank-settings-upgrade-header">
                <span class="dashicons dashicons-superhero-alt"></span>
                <h3><?php esc_html_e('Upgrade to FormRank Pro', 'formrank-lead-scoring'); ?></h3>
            </div>

            <div class="formrank-plan-comparison">
                <div class="formrank-plan formrank-plan-free">
                    <h4><?php esc_html_e('Free Plan', 'formrank-lead-scoring'); ?></h4>
                    <div class="formrank-plan-price">
                        <span class="price">$0</span>
                    </div>
                    <ul>
                        <li><span class="dashicons dashicons-yes"></span> <?php esc_html_e('Unlimited lead capture', 'formrank-lead-scoring'); ?></li>
                        <li><span class="dashicons dashicons-yes"></span> <?php esc_html_e('All form integrations', 'formrank-lead-scoring'); ?></li>
                        <li><span class="dashicons dashicons-yes"></span> <?php esc_html_e('Local scoring engine', 'formrank-lead-scoring'); ?></li>
                        <li><span class="dashicons dashicons-yes"></span> <?php esc_html_e('Email notifications', 'formrank-lead-scoring'); ?></li>
                        <li><span class="dashicons dashicons-yes"></span> <?php esc_html_e('15 AI rescores/month', 'formrank-lead-scoring'); ?></li>
                        <li class="disabled"><span class="dashicons dashicons-no"></span> <?php esc_html_e('Priority support', 'formrank-lead-scoring'); ?></li>
                    </ul>
                    <div class="formrank-plan-badge current">
                        <?php esc_html_e('Current Plan', 'formrank-lead-scoring'); ?>
                    </div>
                </div>

                <div class="formrank-plan formrank-plan-pro">
                    <div class="formrank-popular-badge"><?php esc_html_e('Most Popular', 'formrank-lead-scoring'); ?></div>
                    <h4><?php esc_html_e('Pro Plan', 'formrank-lead-scoring'); ?></h4>
                    <div class="formrank-plan-price">
                        <span class="price">$149</span>
                        <span class="period">/year</span>
                    </div>
                    <ul>
                        <li><span class="dashicons dashicons-yes"></span> <strong><?php esc_html_e('Unlimited AI rescores', 'formrank-lead-scoring'); ?></strong></li>
                        <li><span class="dashicons dashicons-yes"></span> <?php esc_html_e('Advanced Claude AI scoring', 'formrank-lead-scoring'); ?></li>
                        <li><span class="dashicons dashicons-yes"></span> <?php esc_html_e('Slack notifications', 'formrank-lead-scoring'); ?></li>
                        <li><span class="dashicons dashicons-yes"></span> <?php esc_html_e('Priority email support', 'formrank-lead-scoring'); ?></li>
                    </ul>
                    <a href="<?php echo esc_url($upgrade_url); ?>" class="button button-primary formrank-upgrade-plan-btn">
                        <?php esc_html_e('Upgrade to Pro', 'formrank-lead-scoring'); ?>
                    </a>
                </div>

                <div class="formrank-plan formrank-plan-agency">
                    <h4><?php esc_html_e('Agency Plan', 'formrank-lead-scoring'); ?></h4>
                    <div class="formrank-plan-price">
                        <span class="price">$299</span>
                        <span class="period">/year</span>
                    </div>
                    <ul>
                        <li><span class="dashicons dashicons-yes"></span> <strong><?php esc_html_e('Everything in Pro', 'formrank-lead-scoring'); ?></strong></li>
                        <li><span class="dashicons dashicons-yes"></span> <?php esc_html_e('10 site licenses', 'formrank-lead-scoring'); ?></li>
                        <li><span class="dashicons dashicons-yes"></span> <?php esc_html_e('White-label option', 'formrank-lead-scoring'); ?></li>
                        <li><span class="dashicons dashicons-yes"></span> <?php esc_html_e('Client management', 'formrank-lead-scoring'); ?></li>
                        <li><span class="dashicons dashicons-yes"></span> <?php esc_html_e('Priority support', 'formrank-lead-scoring'); ?></li>
                    </ul>
                    <a href="<?php echo esc_url($upgrade_url); ?>" class="button formrank-upgrade-plan-btn-secondary">
                        <?php esc_html_e('Get Agency', 'formrank-lead-scoring'); ?>
                    </a>
                </div>
            </div>

            <div class="formrank-upgrade-footer">
                <p>
                    <span class="dashicons dashicons-shield"></span>
                    <?php esc_html_e('14-day money-back guarantee. Cancel anytime.', 'formrank-lead-scoring'); ?>
                </p>
            </div>
        </div>
        <?php
    }

    /**
     * Render Pro status for paying users
     */
    private static function render_pro_status() {
        $plan_name = self::get_plan_name();
        ?>
        <div class="formrank-pro-status-card">
            <div class="formrank-pro-status-header">
                <span class="dashicons dashicons-awards"></span>
                <h3>
                    <?php
                    printf(
                        /* translators: %s: Plan name */
                        esc_html__('You\'re on the %s Plan', 'formrank-lead-scoring'),
                        esc_html($plan_name)
                    );
                    ?>
                </h3>
            </div>
            <p><?php esc_html_e('Thank you for being a Pro customer! You have access to all features.', 'formrank-lead-scoring'); ?></p>
            <ul class="formrank-pro-features">
                <li><span class="dashicons dashicons-yes-alt"></span> <?php esc_html_e('Unlimited leads', 'formrank-lead-scoring'); ?></li>
                <li><span class="dashicons dashicons-yes-alt"></span> <?php esc_html_e('All form integrations', 'formrank-lead-scoring'); ?></li>
                <li><span class="dashicons dashicons-yes-alt"></span> <?php esc_html_e('Webhooks & API access', 'formrank-lead-scoring'); ?></li>
                <li><span class="dashicons dashicons-yes-alt"></span> <?php esc_html_e('Priority support', 'formrank-lead-scoring'); ?></li>
            </ul>
            <?php if (function_exists('FormRankLS\fr_fs')) : ?>
                <p>
                    <a href="<?php echo esc_url(\FormRankLS\fr_fs()->get_account_url()); ?>" class="button">
                        <?php esc_html_e('Manage Subscription', 'formrank-lead-scoring'); ?>
                    </a>
                </p>
            <?php endif; ?>
        </div>
        <?php
    }
}
