<?php
/**
 * Setup Wizard - Full-page onboarding experience for new installs
 *
 * Inserts industry-specific demo leads so the user can explore
 * the dashboard immediately after activation.
 *
 * @package FormRankLS\Admin
 */

namespace FormRankLS\Admin;

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Setup_Wizard {

    /**
     * Transient name used to trigger the post-activation redirect.
     */
    const REDIRECT_TRANSIENT = 'formrank_ls_setup_redirect';

    /**
     * Register hooks.
     */
    public function __construct() {
        add_action( 'admin_menu', [ $this, 'register_page' ] );
        add_action( 'admin_init', [ $this, 'maybe_redirect' ], 1 );
        add_action( 'admin_init', [ $this, 'maybe_render_wizard' ], 2 );
        add_action( 'wp_ajax_formrank_setup_wizard_insert', [ $this, 'ajax_insert_demo_data' ] );
        add_action( 'wp_ajax_formrank_remove_demo_data', [ $this, 'ajax_remove_demo_data' ] );
    }

    /**
     * Whether the wizard should be shown.
     *
     * Returns false when the onboarding status already records
     * completed=true or skipped=true.
     *
     * @return bool
     */
    public static function should_show(): bool {
        $status = get_option( 'formrank_onboarding_status', [] );

        if ( ! empty( $status['completed'] ) || ! empty( $status['skipped'] ) ) {
            return false;
        }

        return true;
    }

    /**
     * Redirect to the wizard on first activation.
     *
     * Fires at admin_init priority 1 so it runs before any output.
     */
    public function maybe_redirect(): void {
        // Check for the redirect transient set during activation.
        if ( ! get_transient( self::REDIRECT_TRANSIENT ) ) {
            return;
        }

        // Delete transient so we only redirect once.
        delete_transient( self::REDIRECT_TRANSIENT );

        // Do not redirect during AJAX, network admin, or bulk activation.
        if ( wp_doing_ajax() || is_network_admin() || isset( $_GET['activate-multi'] ) ) {
            return;
        }

        // Only redirect if the wizard should still be displayed.
        if ( ! self::should_show() ) {
            return;
        }

        wp_safe_redirect( admin_url( 'admin.php?page=formrank-setup-wizard' ) );
        exit;
    }

    /**
     * Register a hidden admin page for the wizard.
     */
    public function register_page(): void {
        add_submenu_page(
            null, // null parent = hidden page
            __( 'FormRank Setup Wizard', 'formrank-lead-scoring' ),
            __( 'FormRank Setup', 'formrank-lead-scoring' ),
            'manage_options',
            'formrank-setup-wizard',
            [ $this, 'render' ]
        );
    }

    /**
     * Intercept wizard page request on admin_init (before WP outputs admin chrome).
     *
     * This renders the full-page wizard and exits, so no WP sidebar/toolbar appears.
     */
    public function maybe_render_wizard(): void {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( ! isset( $_GET['page'] ) || 'formrank-setup-wizard' !== $_GET['page'] ) {
            return;
        }

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'formrank-lead-scoring' ) );
        }

        // If onboarding is already completed, redirect to dashboard.
        if ( ! self::should_show() ) {
            wp_safe_redirect( admin_url( 'admin.php?page=formrank-lead-scoring-dashboard' ) );
            exit;
        }

        // Enqueue wizard-specific assets.
        wp_enqueue_style(
            'formrank-setup-wizard',
            FORMRANK_LS_URL . 'assets/css/setup-wizard.css',
            [],
            FORMRANK_LS_VERSION
        );

        wp_enqueue_style( 'dashicons' );

        wp_enqueue_script(
            'jquery',
            includes_url( 'js/jquery/jquery.min.js' ),
            [],
            null,
            true
        );

        wp_enqueue_script(
            'formrank-setup-wizard',
            FORMRANK_LS_URL . 'assets/js/setup-wizard.js',
            [ 'jquery' ],
            FORMRANK_LS_VERSION,
            true
        );

        wp_localize_script( 'formrank-setup-wizard', 'formrankWizard', [
            'ajax_url'      => admin_url( 'admin-ajax.php' ),
            'nonce'         => wp_create_nonce( 'formrank_setup_wizard' ),
            'dashboard_url' => admin_url( 'admin.php?page=formrank-lead-scoring-dashboard' ),
            'skip_url'      => admin_url( 'admin.php?page=formrank-lead-scoring-dashboard' ),
        ] );

        // Output full-page wizard and exit before WP renders admin chrome.
        include FORMRANK_LS_PATH . 'templates/admin/setup-wizard.php';
        exit;
    }

    /**
     * Placeholder render callback for add_submenu_page (never reached due to maybe_render_wizard).
     */
    public function render(): void {
        // Handled by maybe_render_wizard() on admin_init.
    }

    /**
     * AJAX: Insert demo leads for the chosen industry.
     */
    public function ajax_insert_demo_data(): void {
        check_ajax_referer( 'formrank_setup_wizard', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [ 'message' => 'Permission denied.' ] );
        }

        $industry = isset( $_POST['industry'] ) ? sanitize_key( $_POST['industry'] ) : '';

        // Special value: user chose to skip the wizard entirely.
        if ( '__skip__' === $industry ) {
            update_option( 'formrank_onboarding_status', [
                'completed'    => true,
                'skipped'      => false,
                'completed_at' => current_time( 'mysql' ),
                'scenario'     => 'skipped',
                'steps'        => [],
            ] );

            wp_send_json_success( [
                'message'      => 'Wizard skipped.',
                'redirect_url' => admin_url( 'admin.php?page=formrank-lead-scoring-dashboard' ),
            ] );
        }

        // Load the demo-leads data file.
        require_once FORMRANK_LS_PATH . 'includes/data/demo-leads.php';

        // Validate the chosen industry key.
        $industries = formrank_get_demo_industries();

        if ( ! isset( $industries[ $industry ] ) ) {
            wp_send_json_error( [ 'message' => 'Invalid industry.' ] );
        }

        $leads = formrank_get_demo_leads( $industry );
        $count = $this->insert_demo_leads( $leads );

        // Mark onboarding as completed via wizard.
        update_option( 'formrank_onboarding_status', [
            'completed'    => true,
            'skipped'      => false,
            'completed_at' => current_time( 'mysql' ),
            'scenario'     => 'wizard',
            'steps'        => [],
        ] );

        wp_send_json_success( [
            'message'      => sprintf( '%d demo leads inserted.', $count ),
            'count'        => $count,
            'redirect_url' => admin_url( 'admin.php?page=formrank-lead-scoring-dashboard' ),
        ] );
    }

    /**
     * Insert an array of demo leads into the database.
     *
     * Spreads created_at timestamps across the last 7 days so the
     * dashboard charts look realistic.
     *
     * @param array $leads List of lead arrays from demo-leads.php.
     * @return int Number of rows successfully inserted.
     */
    private function insert_demo_leads( array $leads ): int {
        global $wpdb;

        $table   = $wpdb->prefix . 'formrank_leads';
        $now     = current_time( 'mysql' );
        $count   = count( $leads );
        $inserted = 0;

        foreach ( $leads as $index => $lead ) {
            // Spread leads across the last 7 days based on their index.
            $days_ago   = $count > 1 ? ( $index / ( $count - 1 ) ) * 7 : 0;
            $created_at = gmdate( 'Y-m-d H:i:s', strtotime( $now ) - (int) ( $days_ago * DAY_IN_SECONDS ) );

            $result = $wpdb->insert(
                $table,
                [
                    'form_plugin'    => 'demo',
                    'form_id'        => 0,
                    'entry_id'       => null,
                    'email'          => $lead['email'] ?? null,
                    'name'           => $lead['name'] ?? null,
                    'company'        => $lead['company'] ?? null,
                    'phone'          => $lead['phone'] ?? null,
                    'form_data'      => wp_json_encode( $lead['form_data'] ?? [] ),
                    'score'          => $lead['score'] ?? null,
                    'score_label'    => $lead['score_label'] ?? null,
                    'score_reasoning' => $lead['score_reasoning'] ?? null,
                    'score_factors'  => wp_json_encode( $lead['score_factors'] ?? [] ),
                    'scoring_method' => 'demo',
                    'source_url'     => home_url( '/contact/' ),
                    'status'         => $lead['status'] ?? 'new',
                    'created_at'     => $created_at,
                    'updated_at'     => $now,
                    'scored_at'      => $created_at,
                ],
                [
                    '%s', '%d', '%d', '%s', '%s', '%s', '%s',
                    '%s', '%d', '%s', '%s', '%s', '%s', '%s',
                    '%s', '%s', '%s', '%s',
                ]
            );

            if ( false !== $result ) {
                $inserted++;
            }
        }

        return $inserted;
    }

    /**
     * AJAX: Remove all demo leads from the database.
     *
     * Called from the dashboard banner, so it uses the main admin nonce.
     */
    public function ajax_remove_demo_data(): void {
        check_ajax_referer( 'formrank_ajax_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [ 'message' => 'Permission denied.' ] );
        }

        global $wpdb;
        $table   = $wpdb->prefix . 'formrank_leads';
        $deleted = (int) $wpdb->query(
            $wpdb->prepare( "DELETE FROM {$table} WHERE scoring_method = %s", 'demo' )
        );

        wp_send_json_success( [
            'message' => sprintf( '%d demo leads removed.', $deleted ),
            'count'   => $deleted,
        ] );
    }

    /**
     * Get the number of demo leads currently in the database.
     *
     * @return int
     */
    public static function get_demo_lead_count(): int {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        return (int) $wpdb->get_var(
            $wpdb->prepare( "SELECT COUNT(*) FROM {$table} WHERE scoring_method = %s", 'demo' )
        );
    }
}
