<?php
/**
 * Settings Handler - Local AI Scoring (No External API Required)
 *
 * @package FormRankLS\Admin
 */

namespace FormRankLS\Admin;

if (!defined('ABSPATH')) {
    exit;
}

use FormRankLS\Core\Local_Scoring_Engine;
use FormRankLS\Integrations\WPForms;
use FormRankLS\Integrations\Gravity_Forms;
use FormRankLS\Integrations\CF7;
use FormRankLS\Integrations\Fluent_Forms;
use FormRankLS\Integrations\Formidable;

class Settings {

    /**
     * Local scoring engine
     */
    private $scoring_engine;

    public function __construct() {
        $this->scoring_engine = new Local_Scoring_Engine();
    }

    /**
     * Register settings
     */
    public function register() {
        // Register settings group
        register_setting('formrank_settings_group', 'formrank_settings', [
            'sanitize_callback' => [$this, 'sanitize_settings']
        ]);

        // AI Learning Status Section (NEW - replaces API section)
        add_settings_section(
            'formrank_learning_section',
            __('AI Learning Status', 'formrank-lead-scoring'),
            [$this, 'render_learning_section'],
            'formrank-lead-scoring-settings'
        );

        // Scoring Settings Section
        add_settings_section(
            'formrank_scoring_section',
            __('Scoring Configuration', 'formrank-lead-scoring'),
            [$this, 'render_scoring_section'],
            'formrank-lead-scoring-settings'
        );

        add_settings_field(
            'business_context',
            __('Business Context', 'formrank-lead-scoring'),
            [$this, 'render_business_context_field'],
            'formrank-lead-scoring-settings',
            'formrank_scoring_section'
        );

        add_settings_field(
            'ideal_customer',
            __('Ideal Customer Profile', 'formrank-lead-scoring'),
            [$this, 'render_ideal_customer_field'],
            'formrank-lead-scoring-settings',
            'formrank_scoring_section'
        );

        add_settings_field(
            'score_thresholds',
            __('Score Thresholds', 'formrank-lead-scoring'),
            [$this, 'render_thresholds_field'],
            'formrank-lead-scoring-settings',
            'formrank_scoring_section'
        );

        add_settings_field(
            'scoring_weights',
            __('Scoring Weights', 'formrank-lead-scoring'),
            [$this, 'render_weights_field'],
            'formrank-lead-scoring-settings',
            'formrank_scoring_section'
        );

        // Forms Settings Section
        add_settings_section(
            'formrank_forms_section',
            __('Form Integrations', 'formrank-lead-scoring'),
            [$this, 'render_forms_section'],
            'formrank-lead-scoring-settings'
        );

        add_settings_field(
            'enabled_forms',
            __('Forms to Score', 'formrank-lead-scoring'),
            [$this, 'render_forms_field'],
            'formrank-lead-scoring-settings',
            'formrank_forms_section'
        );

        // Notifications Section
        add_settings_section(
            'formrank_notifications_section',
            __('Notifications', 'formrank-lead-scoring'),
            [$this, 'render_notifications_section'],
            'formrank-lead-scoring-settings'
        );

        add_settings_field(
            'email_notifications',
            __('Email Notifications', 'formrank-lead-scoring'),
            [$this, 'render_email_notifications_field'],
            'formrank-lead-scoring-settings',
            'formrank_notifications_section'
        );

        add_settings_field(
            'slack_webhook',
            __('Slack Webhook', 'formrank-lead-scoring'),
            [$this, 'render_slack_webhook_field'],
            'formrank-lead-scoring-settings',
            'formrank_notifications_section'
        );

        // Data Retention Section
        add_settings_section(
            'formrank_retention_section',
            __('Data Retention', 'formrank-lead-scoring'),
            [$this, 'render_retention_section'],
            'formrank-lead-scoring-settings'
        );

        add_settings_field(
            'data_retention',
            __('Auto-Delete Old Leads', 'formrank-lead-scoring'),
            [$this, 'render_retention_field'],
            'formrank-lead-scoring-settings',
            'formrank_retention_section'
        );

    }

    /**
     * Sanitize settings before save
     */
    public function sanitize_settings($input) {
        $sanitized = [];

        // Scoring settings
        $sanitized['scoring'] = [
            'business_context' => sanitize_textarea_field($input['scoring']['business_context'] ?? ''),
            'ideal_customer' => sanitize_textarea_field($input['scoring']['ideal_customer'] ?? ''),
            'score_threshold_hot' => absint($input['scoring']['score_threshold_hot'] ?? 80),
            'score_threshold_warm' => absint($input['scoring']['score_threshold_warm'] ?? 60),
            'score_threshold_neutral' => absint($input['scoring']['score_threshold_neutral'] ?? 40),
            'score_threshold_cool' => absint($input['scoring']['score_threshold_cool'] ?? 20),
            'fit_weight' => max(0, min(100, absint($input['scoring']['fit_weight'] ?? 40))),
            'engagement_weight' => max(0, min(100, absint($input['scoring']['engagement_weight'] ?? 60))),
        ];

        // Enabled forms
        $sanitized['enabled_forms'] = [
            'wpforms' => array_map('absint', $input['enabled_forms']['wpforms'] ?? []),
            'gravity' => array_map('absint', $input['enabled_forms']['gravity'] ?? []),
            'cf7' => array_map('absint', $input['enabled_forms']['cf7'] ?? []),
            'fluent' => array_map('absint', $input['enabled_forms']['fluent'] ?? []),
            'formidable' => array_map('absint', $input['enabled_forms']['formidable'] ?? []),
        ];

        // Notifications - validate email recipients
        $email_recipients = $input['notifications']['email_recipients'] ?? '';
        $validated_emails = $this->validate_email_recipients($email_recipients);

        // Hot lead threshold must be between 1 and 100
        $hot_threshold = absint($input['notifications']['hot_lead_threshold'] ?? 80);
        $hot_threshold = max(1, min(100, $hot_threshold));

        $sanitized['notifications'] = [
            'email_enabled' => !empty($input['notifications']['email_enabled']),
            'email_recipients' => $validated_emails,
            'hot_lead_threshold' => $hot_threshold,
            'slack_webhook' => $this->validate_slack_webhook($input['notifications']['slack_webhook'] ?? '')
        ];

        // Advanced settings (keep for cache, rate limit)
        $sanitized['advanced'] = [
            'cache_duration' => absint($input['advanced']['cache_duration'] ?? 3600),
            'rate_limit' => absint($input['advanced']['rate_limit'] ?? 100)
        ];

        // Data retention settings
        $valid_retention_periods = [0, 30, 60, 90, 180, 365];
        $retention_days = absint($input['retention']['days'] ?? 0);
        $sanitized['retention'] = [
            'enabled' => !empty($input['retention']['enabled']),
            'days' => in_array($retention_days, $valid_retention_periods) ? $retention_days : 0,
            'exclude_converted' => !empty($input['retention']['exclude_converted']),
        ];

        // Webhooks (from existing settings)
        $existing = get_option('formrank_settings', []);
        $sanitized['webhooks'] = $existing['webhooks'] ?? [];

        return $sanitized;
    }

    /**
     * Validate and sanitize email recipients (comma-separated list)
     *
     * @param string $recipients Comma-separated email addresses
     * @return string Validated email addresses
     */
    private function validate_email_recipients(string $recipients): string {
        if (empty($recipients)) {
            return '';
        }

        // Split by comma, newline, or semicolon
        $emails = preg_split('/[,;\s]+/', $recipients);
        $valid_emails = [];

        foreach ($emails as $email) {
            $email = trim($email);

            if (empty($email)) {
                continue;
            }

            // Validate email format
            if (is_email($email)) {
                $valid_emails[] = sanitize_email($email);
            } else {
                // Log invalid email for debugging
                error_log(sprintf('FormRank: Invalid email address in notification settings: %s', $email));
            }
        }

        // Return unique valid emails
        return implode(', ', array_unique($valid_emails));
    }

    /**
     * Validate Slack webhook URL
     *
     * @param string $url Slack webhook URL
     * @return string Validated URL or empty string
     */
    private function validate_slack_webhook(string $url): string {
        if (empty($url)) {
            return '';
        }

        $url = esc_url_raw($url);

        // Slack webhook URLs should start with https://hooks.slack.com/
        if (strpos($url, 'https://hooks.slack.com/') === 0) {
            return $url;
        }

        // Also allow custom webhook services that follow similar patterns
        if (preg_match('#^https://[a-zA-Z0-9-]+\.(slack|discordapp|discord)\.com/#', $url)) {
            return $url;
        }

        // Allow any HTTPS URL for generic webhooks (Zapier, Make, n8n, custom endpoints)
        // Developers can restrict allowed domains via the formrank_allowed_webhook_domains filter
        $allowed_domains = apply_filters('formrank_allowed_webhook_domains', []);
        if (!empty($allowed_domains)) {
            $parsed = wp_parse_url($url);
            $host = $parsed['host'] ?? '';
            foreach ($allowed_domains as $domain) {
                if ($host === $domain || str_ends_with($host, '.' . $domain)) {
                    return $url;
                }
            }
            return '';
        }

        if (strpos($url, 'https://') === 0) {
            return $url;
        }

        // Invalid URL
        error_log(sprintf('FormRank: Invalid Slack webhook URL: %s', $url));
        return '';
    }

    /**
     * Render License section
     */
    public function render_license_section() {
        $plan = \FormRankLS\formrank_get_plan();
        $is_paying = \FormRankLS\fr_fs()->is_paying();

        $plan_labels = [
            'free'   => __('Free', 'formrank-lead-scoring'),
            'pro'    => __('Pro', 'formrank-lead-scoring'),
            'agency' => __('Agency', 'formrank-lead-scoring'),
        ];

        $plan_colors = [
            'free'   => '#6b7280',
            'pro'    => '#8b5cf6',
            'agency' => '#f59e0b',
        ];

        $plan_label = $plan_labels[$plan] ?? $plan_labels['free'];
        $plan_color = $plan_colors[$plan] ?? $plan_colors['free'];

        ?>
        <div class="formrank-license-card" style="background: #fff; border: 1px solid #c3c4c7; border-radius: 4px; padding: 20px; margin-bottom: 10px;">
            <div style="display: flex; align-items: center; gap: 12px; margin-bottom: 16px;">
                <span style="font-weight: 600; font-size: 14px;"><?php esc_html_e('Current Plan:', 'formrank-lead-scoring'); ?></span>
                <span style="background: <?php echo esc_attr($plan_color); ?>; color: #fff; padding: 3px 12px; border-radius: 12px; font-size: 12px; font-weight: 600; text-transform: uppercase;">
                    <?php echo esc_html($plan_label); ?>
                </span>
            </div>

            <?php if ($is_paying) : ?>
                <p style="color: #059669; margin-bottom: 12px;">
                    <span class="dashicons dashicons-yes-alt" style="color: #059669;"></span>
                    <?php esc_html_e('License active. All PRO features are unlocked.', 'formrank-lead-scoring'); ?>
                </p>
                <button type="button" class="button" id="formrank-deactivate-license">
                    <?php esc_html_e('Deactivate License', 'formrank-lead-scoring'); ?>
                </button>
            <?php else : ?>
                <p style="color: #6b7280; margin-bottom: 12px;">
                    <?php esc_html_e('Enter your license key to unlock PRO features like AI lead enrichment, advanced scoring, and Slack notifications.', 'formrank-lead-scoring'); ?>
                </p>
                <div style="display: flex; gap: 8px; align-items: flex-start;">
                    <input type="text"
                           id="formrank-license-key"
                           class="regular-text"
                           placeholder="<?php esc_attr_e('sk_xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx', 'formrank-lead-scoring'); ?>"
                           style="flex: 1;">
                    <button type="button" class="button button-primary" id="formrank-activate-license">
                        <?php esc_html_e('Activate License', 'formrank-lead-scoring'); ?>
                    </button>
                </div>
                <p id="formrank-license-message" style="margin-top: 8px; display: none;"></p>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Render AI Learning Status section
     */
    public function render_learning_section() {
        $stats = $this->scoring_engine->get_learning_stats();
        $mode = $stats['scoring_mode'];

        $mode_labels = [
            'cold_start' => __('Cold Start', 'formrank-lead-scoring'),
            'enhanced_rules' => __('Enhanced Rules', 'formrank-lead-scoring'),
            'learning' => __('Learning Mode', 'formrank-lead-scoring'),
        ];

        $mode_descriptions = [
            'cold_start' => __('Using rule-based BANT scoring. Import existing data or capture more leads to unlock enhanced scoring.', 'formrank-lead-scoring'),
            'enhanced_rules' => __('Using enhanced rules with pattern detection. Mark leads as "converted" to enable full learning mode.', 'formrank-lead-scoring'),
            'learning' => __('AI is actively learning from your conversion data to improve scoring accuracy!', 'formrank-lead-scoring'),
        ];

        $mode_colors = [
            'cold_start' => '#dc3545',
            'enhanced_rules' => '#ffc107',
            'learning' => '#28a745',
        ];

        ?>
        <div class="formrank-learning-status-card">
            <div class="status-header">
                <span class="status-badge" style="background: <?php echo esc_attr($mode_colors[$mode]); ?>">
                    <?php echo esc_html($mode_labels[$mode]); ?>
                </span>
                <span class="confidence-badge">
                    <?php
                    $confidence = $mode === 'cold_start' ? 60 : ($mode === 'enhanced_rules' ? 70 : min(95, 70 + ($stats['total_conversions'] / 5)));
                    /* translators: %d: confidence percentage */
                    printf(esc_html__('Confidence: %d%%', 'formrank-lead-scoring'), absint($confidence));
                    ?>
                </span>
            </div>

            <p class="status-description"><?php echo esc_html($mode_descriptions[$mode]); ?></p>

            <div class="stats-grid">
                <div class="stat-item">
                    <span class="stat-value"><?php echo esc_html(number_format($stats['total_leads'])); ?></span>
                    <span class="stat-label"><?php esc_html_e('Total Leads', 'formrank-lead-scoring'); ?></span>
                </div>
                <div class="stat-item">
                    <span class="stat-value"><?php echo esc_html(number_format($stats['total_conversions'])); ?></span>
                    <span class="stat-label"><?php esc_html_e('Conversions', 'formrank-lead-scoring'); ?></span>
                </div>
                <div class="stat-item">
                    <span class="stat-value"><?php echo esc_html($stats['conversion_rate']); ?>%</span>
                    <span class="stat-label"><?php esc_html_e('Conversion Rate', 'formrank-lead-scoring'); ?></span>
                </div>
                <div class="stat-item">
                    <span class="stat-value"><?php echo esc_html($stats['avg_converted_score'] ?: '-'); ?></span>
                    <span class="stat-label"><?php esc_html_e('Avg. Converted Score', 'formrank-lead-scoring'); ?></span>
                </div>
            </div>

            <?php if ($mode === 'cold_start') : ?>
                <div class="progress-section">
                    <p><strong><?php esc_html_e('Progress to Enhanced Scoring:', 'formrank-lead-scoring'); ?></strong></p>
                    <div class="progress-bar">
                        <div class="progress-fill" style="width: <?php echo esc_attr(min(100, ($stats['total_leads'] / 15) * 100)); ?>%"></div>
                    </div>
                    <p class="progress-text">
                        <?php
                        printf(
                            /* translators: %d: number of leads */
                            esc_html__('%d / 15 leads', 'formrank-lead-scoring'),
                            absint($stats['total_leads'])
                        );
                        ?>
                    </p>
                </div>
            <?php elseif ($mode === 'enhanced_rules') : ?>
                <div class="progress-section">
                    <p><strong><?php esc_html_e('Progress to Learning Mode:', 'formrank-lead-scoring'); ?></strong></p>
                    <div class="progress-bar">
                        <div class="progress-fill" style="width: <?php echo esc_attr(min(100, ($stats['total_conversions'] / 10) * 100)); ?>%"></div>
                    </div>
                    <p class="progress-text">
                        <?php
                        printf(
                            /* translators: %d: number of conversions */
                            esc_html__('%d / 10 conversions', 'formrank-lead-scoring'),
                            absint($stats['total_conversions'])
                        );
                        ?>
                    </p>
                </div>
            <?php endif; ?>

            <div class="info-box">
                <strong><?php esc_html_e('No API Key Required!', 'formrank-lead-scoring'); ?></strong>
                <p><?php esc_html_e('FormRank uses an intelligent local scoring engine that learns from YOUR data. All processing happens on your server - no data is sent to external services.', 'formrank-lead-scoring'); ?></p>
            </div>
        </div>
        <?php
    }

    /**
     * Render scoring section description
     */
    public function render_scoring_section() {
        echo '<p>' . esc_html__('Configure how leads are scored. These settings help the AI understand your business better.', 'formrank-lead-scoring') . '</p>';
    }

    /**
     * Render business context field
     */
    public function render_business_context_field() {
        $settings = get_option('formrank_settings', []);
        $value = $settings['scoring']['business_context'] ?? '';

        ?>
        <textarea name="formrank_settings[scoring][business_context]"
                  rows="3"
                  class="large-text"
                  placeholder="<?php esc_attr_e('e.g., B2B SaaS company selling CRM software to mid-market businesses', 'formrank-lead-scoring'); ?>"
        ><?php echo esc_textarea($value); ?></textarea>
        <p class="description">
            <?php esc_html_e('Describe your business, products/services, and target market. This helps contextualize lead scoring.', 'formrank-lead-scoring'); ?>
        </p>
        <?php
    }

    /**
     * Render ideal customer field
     */
    public function render_ideal_customer_field() {
        $settings = get_option('formrank_settings', []);
        $value = $settings['scoring']['ideal_customer'] ?? '';

        ?>
        <textarea name="formrank_settings[scoring][ideal_customer]"
                  rows="3"
                  class="large-text"
                  placeholder="<?php esc_attr_e('e.g., Companies with 10-500 employees in tech industry, decision makers with VP or Director titles', 'formrank-lead-scoring'); ?>"
        ><?php echo esc_textarea($value); ?></textarea>
        <p class="description">
            <?php esc_html_e('Describe your ideal customer profile - industry, company size, roles, budget range, etc.', 'formrank-lead-scoring'); ?>
        </p>
        <?php
    }

    /**
     * Render score thresholds field
     */
    public function render_thresholds_field() {
        $settings = get_option('formrank_settings', []);
        $thresholds = [
            'hot' => $settings['scoring']['score_threshold_hot'] ?? 80,
            'warm' => $settings['scoring']['score_threshold_warm'] ?? 60,
            'neutral' => $settings['scoring']['score_threshold_neutral'] ?? 40,
            'cool' => $settings['scoring']['score_threshold_cool'] ?? 20
        ];

        ?>
        <div class="formrank-thresholds">
            <label>
                <span style="color: #ef4444; font-weight: 600;">Hot</span> (minimum score):
                <input type="number" name="formrank_settings[scoring][score_threshold_hot]"
                       value="<?php echo esc_attr($thresholds['hot']); ?>" min="1" max="100" style="width: 60px;">
            </label>
            <label>
                <span style="color: #f59e0b; font-weight: 600;">Warm</span>:
                <input type="number" name="formrank_settings[scoring][score_threshold_warm]"
                       value="<?php echo esc_attr($thresholds['warm']); ?>" min="1" max="100" style="width: 60px;">
            </label>
            <label>
                <span style="color: #10b981; font-weight: 600;">Neutral</span>:
                <input type="number" name="formrank_settings[scoring][score_threshold_neutral]"
                       value="<?php echo esc_attr($thresholds['neutral']); ?>" min="1" max="100" style="width: 60px;">
            </label>
            <label>
                <span style="color: #3b82f6; font-weight: 600;">Cool</span>:
                <input type="number" name="formrank_settings[scoring][score_threshold_cool]"
                       value="<?php echo esc_attr($thresholds['cool']); ?>" min="1" max="100" style="width: 60px;">
            </label>
        </div>
        <p class="description">
            <?php esc_html_e('Leads with scores below "Cool" threshold are labeled as "Cold".', 'formrank-lead-scoring'); ?>
        </p>
        <?php
    }

    /**
     * Render scoring weights field
     */
    public function render_weights_field() {
        $settings = get_option('formrank_settings', []);
        $fit_weight = $settings['scoring']['fit_weight'] ?? 40;
        $engagement_weight = $settings['scoring']['engagement_weight'] ?? 60;

        ?>
        <div class="formrank-weights">
            <div class="weight-item">
                <label>
                    <strong><?php esc_html_e('Fit Score Weight', 'formrank-lead-scoring'); ?></strong>
                    <input type="range" name="formrank_settings[scoring][fit_weight]"
                           value="<?php echo esc_attr($fit_weight); ?>" min="0" max="100"
                           class="weight-slider" data-target="fit-value">
                    <span class="weight-value" id="fit-value"><?php echo esc_html($fit_weight); ?>%</span>
                </label>
                <p class="description"><?php esc_html_e('Based on BANT: Budget, Authority, Need, Timeline', 'formrank-lead-scoring'); ?></p>
            </div>
            <div class="weight-item">
                <label>
                    <strong><?php esc_html_e('Engagement Score Weight', 'formrank-lead-scoring'); ?></strong>
                    <input type="range" name="formrank_settings[scoring][engagement_weight]"
                           value="<?php echo esc_attr($engagement_weight); ?>" min="0" max="100"
                           class="weight-slider" data-target="engagement-value">
                    <span class="weight-value" id="engagement-value"><?php echo esc_html($engagement_weight); ?>%</span>
                </label>
                <p class="description"><?php esc_html_e('Based on message detail, intent signals, email type', 'formrank-lead-scoring'); ?></p>
            </div>
        </div>
        <p class="description" style="margin-top: 15px;">
            <strong><?php esc_html_e('Tip:', 'formrank-lead-scoring'); ?></strong>
            <?php esc_html_e('Research shows engagement signals are often more predictive than demographics. The default 40/60 split is recommended.', 'formrank-lead-scoring'); ?>
        </p>

        <?php
    }

    /**
     * Render forms section description
     */
    public function render_forms_section() {
        echo '<p>' . esc_html__('Select which forms should be scored. Leave all unchecked to score all forms automatically.', 'formrank-lead-scoring') . '</p>';
    }

    /**
     * Render forms selection field
     */
    public function render_forms_field() {
        $settings = get_option('formrank_settings', []);
        $enabled_forms = $settings['enabled_forms'] ?? [];
        $has_forms = false;

        // WPForms
        $wpforms = WPForms::get_available_forms();
        if (!empty($wpforms)) {
            $has_forms = true;
            echo '<h4>' . esc_html__('WPForms', 'formrank-lead-scoring') . '</h4>';
            echo '<div class="formrank-form-list">';
            foreach ($wpforms as $form) {
                $checked = in_array($form['id'], $enabled_forms['wpforms'] ?? []) ? 'checked' : '';
                printf(
                    '<label><input type="checkbox" name="formrank_settings[enabled_forms][wpforms][]" value="%d" %s> %s</label><br>',
                    esc_attr($form['id']),
                    esc_attr($checked),
                    esc_html($form['title'])
                );
            }
            echo '</div>';
        }

        // Gravity Forms
        $gravity = Gravity_Forms::get_available_forms();
        if (!empty($gravity)) {
            $has_forms = true;
            echo '<h4>' . esc_html__('Gravity Forms', 'formrank-lead-scoring') . '</h4>';
            echo '<div class="formrank-form-list">';
            foreach ($gravity as $form) {
                $checked = in_array($form['id'], $enabled_forms['gravity'] ?? []) ? 'checked' : '';
                printf(
                    '<label><input type="checkbox" name="formrank_settings[enabled_forms][gravity][]" value="%d" %s> %s</label><br>',
                    esc_attr($form['id']),
                    esc_attr($checked),
                    esc_html($form['title'])
                );
            }
            echo '</div>';
        }

        // Contact Form 7
        $cf7 = CF7::get_available_forms();
        if (!empty($cf7)) {
            $has_forms = true;
            echo '<h4>' . esc_html__('Contact Form 7', 'formrank-lead-scoring') . '</h4>';
            echo '<div class="formrank-form-list">';
            foreach ($cf7 as $form) {
                $checked = in_array($form['id'], $enabled_forms['cf7'] ?? []) ? 'checked' : '';
                printf(
                    '<label><input type="checkbox" name="formrank_settings[enabled_forms][cf7][]" value="%d" %s> %s</label><br>',
                    esc_attr($form['id']),
                    esc_attr($checked),
                    esc_html($form['title'])
                );
            }
            echo '</div>';
        }

        // Fluent Forms
        $fluent = Fluent_Forms::get_available_forms();
        if (!empty($fluent)) {
            $has_forms = true;
            echo '<h4>' . esc_html__('Fluent Forms', 'formrank-lead-scoring') . '</h4>';
            echo '<div class="formrank-form-list">';
            foreach ($fluent as $form) {
                $checked = in_array($form['id'], $enabled_forms['fluent'] ?? []) ? 'checked' : '';
                printf(
                    '<label><input type="checkbox" name="formrank_settings[enabled_forms][fluent][]" value="%d" %s> %s</label><br>',
                    esc_attr($form['id']),
                    esc_attr($checked),
                    esc_html($form['title'])
                );
            }
            echo '</div>';
        }

        // Formidable Forms
        $formidable = Formidable::get_available_forms();
        if (!empty($formidable)) {
            $has_forms = true;
            echo '<h4>' . esc_html__('Formidable Forms', 'formrank-lead-scoring') . '</h4>';
            echo '<div class="formrank-form-list">';
            foreach ($formidable as $form) {
                $checked = in_array($form['id'], $enabled_forms['formidable'] ?? []) ? 'checked' : '';
                printf(
                    '<label><input type="checkbox" name="formrank_settings[enabled_forms][formidable][]" value="%d" %s> %s</label><br>',
                    esc_attr($form['id']),
                    esc_attr($checked),
                    esc_html($form['title'])
                );
            }
            echo '</div>';
        }

        if (!$has_forms) {
            echo '<div class="notice notice-warning inline" style="margin: 10px 0;">';
            echo '<p>' . esc_html__('No supported form plugins detected. Please install one of the following:', 'formrank-lead-scoring') . '</p>';
            echo '<ul style="list-style: disc; margin-left: 20px;">';
            echo '<li>WPForms</li>';
            echo '<li>Gravity Forms</li>';
            echo '<li>Contact Form 7</li>';
            echo '<li>Fluent Forms</li>';
            echo '<li>Formidable Forms</li>';
            echo '</ul>';
            echo '</div>';
        }
    }

    /**
     * Render notifications section description
     */
    public function render_notifications_section() {
        echo '<p>' . esc_html__('Get notified when hot leads are captured.', 'formrank-lead-scoring') . '</p>';
    }

    /**
     * Render email notifications field
     */
    public function render_email_notifications_field() {
        $settings = get_option('formrank_settings', []);
        $enabled = !empty($settings['notifications']['email_enabled']);
        $recipients = $settings['notifications']['email_recipients'] ?? get_option('admin_email');
        $threshold = $settings['notifications']['hot_lead_threshold'] ?? 80;

        ?>
        <label>
            <input type="checkbox"
                   name="formrank_settings[notifications][email_enabled]"
                   value="1"
                   <?php checked($enabled); ?>>
            <?php esc_html_e('Enable email notifications for hot leads', 'formrank-lead-scoring'); ?>
        </label>
        <br><br>
        <label>
            <?php esc_html_e('Recipients:', 'formrank-lead-scoring'); ?>
            <input type="text"
                   name="formrank_settings[notifications][email_recipients]"
                   value="<?php echo esc_attr($recipients); ?>"
                   class="regular-text"
                   placeholder="<?php esc_attr_e('email@example.com', 'formrank-lead-scoring'); ?>">
        </label>
        <p class="description"><?php esc_html_e('Separate multiple emails with commas.', 'formrank-lead-scoring'); ?></p>
        <br>
        <label>
            <?php esc_html_e('Notify when score is at least:', 'formrank-lead-scoring'); ?>
            <input type="number"
                   name="formrank_settings[notifications][hot_lead_threshold]"
                   value="<?php echo esc_attr($threshold); ?>"
                   min="1"
                   max="100"
                   style="width: 60px;">
        </label>
        <?php
    }

    /**
     * Render Slack webhook field
     */
    public function render_slack_webhook_field() {
        $settings = get_option('formrank_settings', []);
        $webhook = $settings['notifications']['slack_webhook'] ?? '';

        ?>
        <input type="url"
               name="formrank_settings[notifications][slack_webhook]"
               value="<?php echo esc_attr($webhook); ?>"
               class="regular-text"
               placeholder="https://hooks.slack.com/services/...">
        <p class="description">
            <?php esc_html_e('Optional: Receive Slack notifications for hot leads.', 'formrank-lead-scoring'); ?>
        </p>
        <?php
    }

    /**
     * Render data retention section description
     */
    public function render_retention_section() {
        echo '<p>' . esc_html__('Configure automatic cleanup of old lead data to comply with data protection regulations (GDPR).', 'formrank-lead-scoring') . '</p>';
    }

    /**
     * Render data retention field
     */
    public function render_retention_field() {
        $settings = get_option('formrank_settings', []);
        $enabled = !empty($settings['retention']['enabled']);
        $days = $settings['retention']['days'] ?? 0;
        $exclude_converted = !empty($settings['retention']['exclude_converted']);

        // Get count of leads that would be affected
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';
        $total_leads = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table}");

        ?>
        <div class="formrank-retention-settings">
            <label style="display: block; margin-bottom: 15px;">
                <input type="checkbox"
                       name="formrank_settings[retention][enabled]"
                       value="1"
                       <?php checked($enabled); ?>
                       id="formrank-retention-enabled">
                <?php esc_html_e('Enable automatic data retention cleanup', 'formrank-lead-scoring'); ?>
            </label>

            <div class="formrank-retention-options" style="margin-left: 25px; <?php echo !$enabled ? 'opacity: 0.5;' : ''; ?>">
                <label style="display: block; margin-bottom: 10px;">
                    <?php esc_html_e('Delete leads older than:', 'formrank-lead-scoring'); ?>
                    <select name="formrank_settings[retention][days]" id="formrank-retention-days" <?php disabled(!$enabled); ?>>
                        <option value="0" <?php selected($days, 0); ?>><?php esc_html_e('Never (keep forever)', 'formrank-lead-scoring'); ?></option>
                        <option value="30" <?php selected($days, 30); ?>><?php esc_html_e('30 days', 'formrank-lead-scoring'); ?></option>
                        <option value="60" <?php selected($days, 60); ?>><?php esc_html_e('60 days', 'formrank-lead-scoring'); ?></option>
                        <option value="90" <?php selected($days, 90); ?>><?php esc_html_e('90 days', 'formrank-lead-scoring'); ?></option>
                        <option value="180" <?php selected($days, 180); ?>><?php esc_html_e('6 months', 'formrank-lead-scoring'); ?></option>
                        <option value="365" <?php selected($days, 365); ?>><?php esc_html_e('1 year', 'formrank-lead-scoring'); ?></option>
                    </select>
                </label>

                <label style="display: block; margin-bottom: 10px;">
                    <input type="checkbox"
                           name="formrank_settings[retention][exclude_converted]"
                           value="1"
                           <?php checked($exclude_converted); ?>
                           <?php disabled(!$enabled); ?>>
                    <?php esc_html_e('Keep converted leads indefinitely (recommended)', 'formrank-lead-scoring'); ?>
                </label>

                <p class="description">
                    <?php
                    printf(
                        /* translators: %d: total number of leads */
                        esc_html__('You currently have %d leads stored.', 'formrank-lead-scoring'),
                        absint($total_leads)
                    );
                    ?>
                </p>

                <?php if ($enabled && $days > 0) : ?>
                    <?php
                    $cutoff_date = gmdate('Y-m-d H:i:s', strtotime('-' . absint($days) . ' days'));
                    // Use prepared statement for security
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                    if ($exclude_converted) {
                        $leads_to_delete = (int) $wpdb->get_var(
                            $wpdb->prepare(
                                "SELECT COUNT(*) FROM {$table} WHERE created_at < %s AND status != 'converted'",
                                $cutoff_date
                            )
                        );
                    } else {
                        $leads_to_delete = (int) $wpdb->get_var(
                            $wpdb->prepare(
                                "SELECT COUNT(*) FROM {$table} WHERE created_at < %s",
                                $cutoff_date
                            )
                        );
                    }
                    ?>
                    <?php if ($leads_to_delete > 0) : ?>
                        <div class="notice notice-warning inline" style="margin: 10px 0; padding: 10px;">
                            <p>
                                <strong><?php esc_html_e('Warning:', 'formrank-lead-scoring'); ?></strong>
                                <?php
                                printf(
                                    /* translators: %d: number of leads to be deleted */
                                    esc_html(_n(
                                        '%d lead will be deleted during the next cleanup.',
                                        '%d leads will be deleted during the next cleanup.',
                                        $leads_to_delete,
                                        'formrank-lead-scoring'
                                    )),
                                    absint($leads_to_delete)
                                );
                                ?>
                            </p>
                        </div>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
        </div>

        <?php
    }
}
