<?php
/**
 * Onboarding Wizard - Guides users through setup based on their situation
 *
 * Handles two scenarios:
 * 1. Cold Start: No existing data, start fresh with rule-based scoring
 * 2. Existing Data: Import historical entries to jumpstart learning
 *
 * @package FormRankLS\Admin
 */

namespace FormRankLS\Admin;

if (!defined('ABSPATH')) {
    exit;
}

use FormRankLS\Core\Data_Importer;
use FormRankLS\Core\Local_Scoring_Engine;

class Onboarding {

    /**
     * Option name for onboarding status
     */
    const STATUS_OPTION = 'formrank_onboarding_status';

    /**
     * Data importer instance
     */
    private $importer;

    /**
     * Scoring engine instance
     */
    private $scoring_engine;

    public function __construct() {
        $this->importer = new Data_Importer();
        $this->scoring_engine = new Local_Scoring_Engine();

        // Show onboarding notice if not completed
        add_action( 'admin_notices', [ $this, 'show_onboarding_notice' ] );

        // Register AJAX handlers
        add_action( 'wp_ajax_formrank_complete_onboarding', [ $this, 'ajax_complete_onboarding' ] );
        add_action( 'wp_ajax_formrank_skip_onboarding', [ $this, 'ajax_skip_onboarding' ] );
        add_action( 'wp_ajax_formrank_get_onboarding_data', [ $this, 'ajax_get_onboarding_data' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_onboarding_scripts' ] );
    }

    /**
     * Get onboarding status
     */
    public function get_status(): array {
        return get_option( self::STATUS_OPTION, [
            'completed'    => false,
            'skipped'      => false,
            'completed_at' => null,
            'scenario'     => null, // 'cold_start' or 'existing_data'
            'steps'        => [],
        ] );
    }

    /**
     * Check if onboarding is needed
     */
    public function needs_onboarding(): bool {
        $status = $this->get_status();
        return ! $status['completed'] && ! $status['skipped'];
    }

    /**
     * Complete onboarding
     */
    public function complete_onboarding( string $scenario, array $steps = [] ): bool {
        return update_option( self::STATUS_OPTION, [
            'completed'    => true,
            'skipped'      => false,
            'completed_at' => current_time( 'mysql' ),
            'scenario'     => $scenario,
            'steps'        => $steps,
        ] );
    }

    /**
     * Skip onboarding
     */
    public function skip_onboarding(): bool {
        return update_option( self::STATUS_OPTION, [
            'completed'    => false,
            'skipped'      => true,
            'completed_at' => current_time( 'mysql' ),
            'scenario'     => 'skipped',
            'steps'        => [],
        ] );
    }

    /**
     * Enqueue onboarding scripts
     */
    public function enqueue_onboarding_scripts() {
        if ( ! $this->needs_onboarding() ) {
            return;
        }

        wp_enqueue_script(
            'formrank-lead-scoring-onboarding',
            FORMRANK_LS_URL . 'assets/js/admin-onboarding.js',
            [ 'jquery' ],
            FORMRANK_LS_VERSION,
            true
        );

        wp_localize_script( 'formrank-lead-scoring-onboarding', 'formrankOnboarding', [
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'nonce'    => wp_create_nonce( 'formrank_onboarding_nonce' ),
        ] );
    }

    /**
     * Show onboarding notice in admin
     */
    public function show_onboarding_notice() {
        // Only show on FormRank pages or plugins page
        $screen = get_current_screen();
        if ( ! $screen ) {
            return;
        }

        // Don't show notice on the setup wizard page
        if ( $screen->id === 'admin_page_formrank-setup-wizard' ) {
            return;
        }

        $formrank_pages = [
            'toplevel_page_formrank-lead-scoring',
            'formrank-lead-scoring_page_formrank-lead-scoring-leads',
            'formrank-lead-scoring_page_formrank-lead-scoring-settings',
            'formrank-lead-scoring_page_formrank-lead-scoring-import',
            'plugins',
        ];

        if ( ! in_array( $screen->id, $formrank_pages, true ) ) {
            return;
        }

        if ( ! $this->needs_onboarding() ) {
            return;
        }

        // Get detection data
        $detected_plugins = $this->importer->detect_form_plugins();
        $has_existing_data = false;
        $total_entries = 0;

        foreach ( $detected_plugins as $plugin ) {
            $total_entries += $plugin['entry_count'];
            if ( $plugin['entry_count'] > 0 ) {
                $has_existing_data = true;
            }
        }

        ?>
        <div class="notice notice-info formrank-onboarding-notice" style="position: relative;">
            <button type="button" class="notice-dismiss formrank-dismiss-onboarding">
                <span class="screen-reader-text"><?php esc_html_e( 'Dismiss this notice.', 'formrank-lead-scoring' ); ?></span>
            </button>
            <div class="formrank-onboarding-header">
                <h2><?php esc_html_e( 'Welcome to FormRank! 🎉', 'formrank-lead-scoring' ); ?></h2>
                <p><?php esc_html_e( 'Let\'s get you set up with AI-powered lead scoring.', 'formrank-lead-scoring' ); ?></p>
            </div>

            <?php if ( $has_existing_data ) : ?>
                <div class="formrank-onboarding-scenario formrank-scenario-existing">
                    <div class="scenario-icon">📊</div>
                    <div class="scenario-content">
                        <h3><?php esc_html_e( 'We found existing form data!', 'formrank-lead-scoring' ); ?></h3>
                        <p>
                            <?php
                            printf(
                                /* translators: 1: number of entries, 2: number of plugins */
                                esc_html__( 'You have %1$s entries across %2$s form plugin(s). Importing this data will help the AI learn your patterns faster.', 'formrank-lead-scoring' ),
                                '<strong>' . number_format( $total_entries ) . '</strong>',
                                '<strong>' . count( $detected_plugins ) . '</strong>'
                            );
                            ?>
                        </p>
                        <div class="detected-plugins">
                            <?php foreach ( $detected_plugins as $plugin ) : ?>
                                <span class="plugin-badge">
                                    <?php echo esc_html( $plugin['name'] ); ?>
                                    (<?php echo esc_html( number_format( $plugin['entry_count'] ) ); ?>)
                                </span>
                            <?php endforeach; ?>
                        </div>
                        <div class="scenario-actions">
                            <a href="<?php echo esc_url( admin_url( 'admin.php?page=formrank-lead-scoring-import' ) ); ?>" class="button button-primary button-large">
                                <?php esc_html_e( 'Import Existing Data', 'formrank-lead-scoring' ); ?>
                            </a>
                            <button type="button" class="button button-secondary formrank-skip-to-fresh">
                                <?php esc_html_e( 'Start Fresh Instead', 'formrank-lead-scoring' ); ?>
                            </button>
                        </div>
                    </div>
                </div>
            <?php else : ?>
                <div class="formrank-onboarding-scenario formrank-scenario-fresh">
                    <div class="scenario-icon">🚀</div>
                    <div class="scenario-content">
                        <h3><?php esc_html_e( 'Starting fresh? No problem!', 'formrank-lead-scoring' ); ?></h3>
                        <p>
                            <?php esc_html_e( 'FormRank will use rule-based scoring until it has enough data to learn from. The more leads you capture, the smarter it gets!', 'formrank-lead-scoring' ); ?>
                        </p>
                        <div class="cold-start-info">
                            <div class="info-item">
                                <span class="info-number">50</span>
                                <span class="info-label"><?php esc_html_e( 'leads to unlock enhanced scoring', 'formrank-lead-scoring' ); ?></span>
                            </div>
                            <div class="info-item">
                                <span class="info-number">10</span>
                                <span class="info-label"><?php esc_html_e( 'conversions to enable learning mode', 'formrank-lead-scoring' ); ?></span>
                            </div>
                        </div>
                        <div class="scenario-actions">
                            <a href="<?php echo esc_url( admin_url( 'admin.php?page=formrank-lead-scoring-settings' ) ); ?>" class="button button-primary button-large">
                                <?php esc_html_e( 'Configure Settings', 'formrank-lead-scoring' ); ?>
                            </a>
                            <button type="button" class="button button-secondary formrank-dismiss-onboarding">
                                <?php esc_html_e( 'Dismiss', 'formrank-lead-scoring' ); ?>
                            </button>
                        </div>
                    </div>
                </div>
            <?php endif; ?>

            <div class="formrank-onboarding-footer">
                <p>
                    <?php esc_html_e( 'Need help?', 'formrank-lead-scoring' ); ?>
                    <a href="https://formrankwp.com/docs" target="_blank"><?php esc_html_e( 'Read the documentation', 'formrank-lead-scoring' ); ?></a>
                </p>
            </div>
        </div>

        <?php
    }

    /**
     * AJAX: Complete onboarding
     */
    public function ajax_complete_onboarding() {
        check_ajax_referer( 'formrank_onboarding_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [ 'message' => 'Permission denied' ] );
        }

        $scenario = sanitize_text_field( wp_unslash( $_POST['scenario'] ?? 'cold_start' ) );
        $steps = isset( $_POST['steps'] ) ? array_map( 'sanitize_text_field', wp_unslash( (array) $_POST['steps'] ) ) : [];

        $result = $this->complete_onboarding( $scenario, $steps );

        if ( $result ) {
            wp_send_json_success( [ 'message' => 'Onboarding completed' ] );
        } else {
            wp_send_json_error( [ 'message' => 'Failed to save onboarding status' ] );
        }
    }

    /**
     * AJAX: Skip onboarding
     */
    public function ajax_skip_onboarding() {
        check_ajax_referer( 'formrank_onboarding_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [ 'message' => 'Permission denied' ] );
        }

        $result = $this->skip_onboarding();

        if ( $result ) {
            wp_send_json_success( [ 'message' => 'Onboarding skipped' ] );
        } else {
            wp_send_json_error( [ 'message' => 'Failed to save status' ] );
        }
    }

    /**
     * AJAX: Get onboarding data
     */
    public function ajax_get_onboarding_data() {
        check_ajax_referer( 'formrank_onboarding_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [ 'message' => 'Permission denied' ] );
        }

        $detected_plugins = $this->importer->detect_form_plugins();
        $learning_stats = $this->scoring_engine->get_learning_stats();

        $total_entries = 0;
        foreach ( $detected_plugins as $plugin ) {
            $total_entries += $plugin['entry_count'];
        }

        wp_send_json_success( [
            'status'           => $this->get_status(),
            'detected_plugins' => $detected_plugins,
            'total_entries'    => $total_entries,
            'learning_stats'   => $learning_stats,
            'has_existing_data' => $total_entries > 0,
        ] );
    }
}
