<?php
/**
 * Lead List Table - displays leads using WP_List_Table
 *
 * @package FormRankLS\Admin
 */

namespace FormRankLS\Admin;

if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('WP_List_Table')) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

use FormRankLS\Admin\Upgrade_Notice;

class Lead_List extends \WP_List_Table {

    /**
     * Lead IDs that have enrichment records (batch-loaded in prepare_items to avoid N+1 queries)
     *
     * @var int[]
     */
    private $enriched_lead_ids = [];

    public function __construct() {
        parent::__construct([
            'singular' => 'lead',
            'plural' => 'leads',
            'ajax' => false
        ]);
    }

    /**
     * Define table columns
     */
    public function get_columns() {
        return [
            'cb' => '<input type="checkbox" />',
            'score' => __('Score', 'formrank-lead-scoring'),
            'name' => __('Name', 'formrank-lead-scoring'),
            'email' => __('Email', 'formrank-lead-scoring'),
            'company' => __('Company', 'formrank-lead-scoring'),
            'form' => __('Form', 'formrank-lead-scoring'),
            'status' => __('Status', 'formrank-lead-scoring'),
            'created_at' => __('Date', 'formrank-lead-scoring')
        ];
    }

    /**
     * Define sortable columns
     */
    public function get_sortable_columns() {
        return [
            'score' => ['score', true],
            'created_at' => ['created_at', true],
            'name' => ['name', false],
            'status' => ['status', false]
        ];
    }

    /**
     * Prepare items for display
     */
    public function prepare_items() {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        $per_page = 20;
        $current_page = $this->get_pagenum();

        // Build query
        $where = '1=1';
        $params = [];

        // Filter by score label (check both GET and REQUEST for flexibility)
        $label = isset($_REQUEST['label']) ? sanitize_text_field(wp_unslash($_REQUEST['label'])) : '';
        if (!empty($label)) {
            $where .= ' AND score_label = %s';
            $params[] = $label;
        }

        // Filter by status
        $status = isset($_REQUEST['status']) ? sanitize_text_field(wp_unslash($_REQUEST['status'])) : '';
        if (!empty($status)) {
            $where .= ' AND status = %s';
            $params[] = $status;
        } elseif (empty($label)) {
            // Hide archived leads from the default "All" view
            $where .= " AND status != 'archived'";
        }

        // Search - check both 's' parameter (GET form) and search input
        $search_term = isset($_REQUEST['s']) ? sanitize_text_field(wp_unslash($_REQUEST['s'])) : '';
        if (!empty($search_term)) {
            $search = '%' . $wpdb->esc_like($search_term) . '%';
            $where .= ' AND (name LIKE %s OR email LIKE %s OR company LIKE %s OR phone LIKE %s OR form_data LIKE %s)';
            $params = array_merge($params, [$search, $search, $search, $search, $search]);
        }

        // Sorting
        $orderby_options = ['score', 'created_at', 'name', 'status'];
        $raw_orderby = isset($_REQUEST['orderby']) ? sanitize_text_field(wp_unslash($_REQUEST['orderby'])) : '';
        $orderby = in_array($raw_orderby, $orderby_options, true) ? $raw_orderby : 'created_at';
        $raw_order = isset($_REQUEST['order']) ? sanitize_text_field(wp_unslash($_REQUEST['order'])) : '';
        $order = $raw_order === 'asc' ? 'ASC' : 'DESC';

        // Get total
        $total_query = "SELECT COUNT(*) FROM {$table} WHERE {$where}";
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table names are safely constructed from $wpdb->prefix.
        $total = (int) $wpdb->get_var($params ? $wpdb->prepare($total_query, $params) : $total_query);

        // Get items
        $offset = ($current_page - 1) * $per_page;
        $query = "SELECT * FROM {$table} WHERE {$where} ORDER BY {$orderby} {$order} LIMIT %d OFFSET %d";
        $query_params = array_merge($params, [$per_page, $offset]);

        // phpcs:disable WordPress.DB.PreparedSQL.NotPrepared -- Table names are safely constructed from $wpdb->prefix.
        $this->items = $wpdb->get_results(
            $wpdb->prepare($query, $query_params),
            // phpcs:enable WordPress.DB.PreparedSQL.NotPrepared
            ARRAY_A
        );

        // Batch-load enrichment status to avoid N+1 queries in column_name()
        if (!empty($this->items)) {
            $lead_ids = array_column($this->items, 'id');
            $ids_placeholder = implode(',', array_map('absint', $lead_ids));
            $enrichments_table = $wpdb->prefix . 'formrank_enrichments';
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- IDs are absint-sanitized above.
            $enriched = $wpdb->get_col("SELECT lead_id FROM {$enrichments_table} WHERE lead_id IN ({$ids_placeholder})");
            $this->enriched_lead_ids = array_map('intval', $enriched);
        }

        // Set column headers
        $this->_column_headers = [
            $this->get_columns(),
            [],
            $this->get_sortable_columns()
        ];

        // Pagination
        $this->set_pagination_args([
            'total_items' => $total,
            'per_page' => $per_page,
            'total_pages' => ceil($total / $per_page)
        ]);
    }

    /**
     * Render score column
     */
    public function column_score($item) {
        $score = $item['score'] ?? '-';
        $label = $item['score_label'] ?? 'unknown';
        $scoring_method = $item['scoring_method'] ?? 'local';

        $colors = [
            'hot' => '#ef4444',
            'warm' => '#f59e0b',
            'neutral' => '#10b981',
            'cool' => '#3b82f6',
            'cold' => '#6b7280'
        ];

        $color = $colors[$label] ?? '#6b7280';

        // Scoring method badge
        $method_badge = '';
        if ($scoring_method === 'ai') {
            $method_badge = '<span class="formrank-list-method-badge ai" title="' . esc_attr__('AI Scored', 'formrank-lead-scoring') . '"><span class="dashicons dashicons-cloud"></span></span>';
        } else {
            $method_badge = '<span class="formrank-list-method-badge local" title="' . esc_attr__('Rule-based', 'formrank-lead-scoring') . '"><span class="dashicons dashicons-laptop"></span></span>';
        }

        return sprintf(
            '<span class="formrank-score" style="display:inline-flex;align-items:center;gap:4px;">
                <strong style="color:%s;font-size:16px;">%s</strong>
                %s
            </span>',
            esc_attr($color),
            esc_html($score),
            $method_badge
        );
    }

    /**
     * Render name column with enrichment status dot and actions
     */
    public function column_name($item) {
        $name = esc_html($item['name'] ?: __('Unknown', 'formrank-lead-scoring'));
        $lead_id = $item['id'];

        // Enrichment status dot (green = enriched, gray = not enriched)
        $is_enriched = in_array((int) $item['id'], $this->enriched_lead_ids, true);
        $dot_color = $is_enriched ? '#22c55e' : '#9ca3af';
        $dot_title = $is_enriched
            ? __('Enriched', 'formrank-lead-scoring')
            : __('Not enriched', 'formrank-lead-scoring');
        $status_dot = sprintf(
            '<span class="formrank-enrichment-dot" style="display:inline-block;width:8px;height:8px;border-radius:50%%;background:%s;margin-right:6px;vertical-align:middle;" title="%s"></span>',
            esc_attr($dot_color),
            esc_attr($dot_title)
        );

        $actions = [
            'view' => sprintf(
                '<a href="%s">%s</a>',
                esc_url(admin_url("admin.php?page=formrank-lead-scoring-lead&id={$lead_id}")),
                __('View', 'formrank-lead-scoring')
            ),
        ];

        // Enrich/Re-enrich hover action (between view and rescore)
        // ROADMAP criterion 6: "clicking 'Enrich' in the list triggers enrichment without navigating away"
        // The link is AJAX-wired by plan 03-03 JS. The href is a no-JS fallback to the detail page.
        $is_pro = !Upgrade_Notice::is_free_user();
        if ($is_pro) {
            $enrich_fallback_url = admin_url("admin.php?page=formrank-lead-scoring-lead&id={$lead_id}&enrich=1");
            $actions['enrich'] = sprintf(
                '<a href="%s" class="formrank-list-enrich-btn" data-lead-id="%d">%s</a>',
                esc_url($enrich_fallback_url),
                absint($lead_id),
                $is_enriched
                    ? __('Re-enrich', 'formrank-lead-scoring')
                    : __('Enrich', 'formrank-lead-scoring')
            );
        } else {
            $actions['enrich'] = sprintf(
                '<a href="%s">%s</a>',
                esc_url(Upgrade_Notice::get_upgrade_url()),
                __('Upgrade to Enrich', 'formrank-lead-scoring')
            );
        }

        $actions['rescore'] = sprintf(
            '<a href="#" class="formrank-rescore" data-id="%d">%s</a>',
            $lead_id,
            __('Rescore', 'formrank-lead-scoring')
        );

        $actions['archive'] = sprintf(
            '<a href="%s" class="formrank-archive">%s</a>',
            esc_url(wp_nonce_url(
                admin_url("admin.php?page=formrank-lead-scoring&action=archive&id={$lead_id}"),
                'archive_lead_' . $lead_id
            )),
            __('Archive', 'formrank-lead-scoring')
        );

        return sprintf(
            '%s<strong><a href="%s">%s</a></strong>%s',
            $status_dot,
            esc_url(admin_url("admin.php?page=formrank-lead-scoring-lead&id={$lead_id}")),
            $name,
            $this->row_actions($actions)
        );
    }

    /**
     * Render email column
     */
    public function column_email($item) {
        $email_raw = $item['email'] ?? '-';

        if ($email_raw !== '-') {
            return sprintf('<a href="mailto:%s">%s</a>', esc_attr($email_raw), esc_html($email_raw));
        }

        return esc_html($email_raw);
    }

    /**
     * Render company column
     */
    public function column_company($item) {
        return esc_html($item['company'] ?: '-');
    }

    /**
     * Render form column
     */
    public function column_form($item) {
        $plugin = ucfirst($item['form_plugin'] ?? 'Unknown');
        $form_id = $item['form_id'] ?? '-';
        return esc_html("{$plugin} #{$form_id}");
    }

    /**
     * Render status column
     */
    public function column_status($item) {
        $status = $item['status'] ?? 'new';
        $labels = [
            'new' => __('New', 'formrank-lead-scoring'),
            'contacted' => __('Contacted', 'formrank-lead-scoring'),
            'qualified' => __('Qualified', 'formrank-lead-scoring'),
            'converted' => __('Converted', 'formrank-lead-scoring'),
            'archived' => __('Archived', 'formrank-lead-scoring')
        ];

        $colors = [
            'new' => '#3b82f6',
            'contacted' => '#f59e0b',
            'qualified' => '#10b981',
            'converted' => '#059669',
            'archived' => '#6b7280'
        ];

        $icons = [
            'new' => '●',
            'contacted' => '◐',
            'qualified' => '◑',
            'converted' => '✓',
            'archived' => '○'
        ];

        return sprintf(
            '<span class="formrank-status" style="color:%s;">%s %s</span>',
            esc_attr($colors[$status] ?? '#6b7280'),
            esc_html($icons[$status] ?? '●'),
            esc_html($labels[$status] ?? $status)
        );
    }

    /**
     * Render created_at column
     */
    public function column_created_at($item) {
        $date = $item['created_at'] ?? '';

        if ($date) {
            $timestamp = strtotime($date);
            return sprintf(
                '<span title="%s">%s</span>',
                esc_attr(wp_date(get_option('date_format') . ' ' . get_option('time_format'), $timestamp)),
                esc_html(human_time_diff($timestamp, current_time('timestamp')) . ' ' . __('ago', 'formrank-lead-scoring'))
            );
        }

        return '-';
    }

    /**
     * Render checkbox column
     */
    public function column_cb($item) {
        return sprintf(
            '<input type="checkbox" name="lead_ids[]" value="%d" />',
            $item['id']
        );
    }

    /**
     * Default column handler
     */
    public function column_default($item, $column_name) {
        return esc_html($item[$column_name] ?? '-');
    }

    /**
     * Get view links (All, Hot, Warm, etc.)
     */
    protected function get_views() {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        $counts = $wpdb->get_results(
            "SELECT score_label, COUNT(*) as count FROM {$table} WHERE status != 'archived' GROUP BY score_label",
            OBJECT_K
        );

        $total = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table} WHERE status != 'archived'");

        $current = isset($_REQUEST['label']) ? sanitize_text_field(wp_unslash($_REQUEST['label'])) : 'all';

        $views = [
            'all' => sprintf(
                '<a href="%s" class="%s">%s <span class="count">(%d)</span></a>',
                esc_url(admin_url('admin.php?page=formrank-lead-scoring')),
                $current === 'all' ? 'current' : '',
                __('All', 'formrank-lead-scoring'),
                $total
            )
        ];

        $labels = [
            'hot' => __('Hot', 'formrank-lead-scoring'),
            'warm' => __('Warm', 'formrank-lead-scoring'),
            'neutral' => __('Neutral', 'formrank-lead-scoring'),
            'cool' => __('Cool', 'formrank-lead-scoring'),
            'cold' => __('Cold', 'formrank-lead-scoring')
        ];

        foreach ($labels as $key => $label) {
            $count = isset($counts[$key]) ? $counts[$key]->count : 0;
            $views[$key] = sprintf(
                '<a href="%s" class="%s">%s <span class="count">(%d)</span></a>',
                esc_url(admin_url("admin.php?page=formrank-lead-scoring&label={$key}")),
                $current === $key ? 'current' : '',
                $label,
                $count
            );
        }

        // Add status filters
        $status_counts = $wpdb->get_results(
            "SELECT status, COUNT(*) as count FROM {$table} GROUP BY status",
            OBJECT_K
        );

        $current_status = isset($_REQUEST['status']) ? sanitize_text_field(wp_unslash($_REQUEST['status'])) : '';

        // Add Converted filter
        $converted_count = isset($status_counts['converted']) ? $status_counts['converted']->count : 0;
        if ($converted_count > 0 || $current_status === 'converted') {
            $views['converted'] = sprintf(
                '<a href="%s" class="%s" style="color: #059669;">✓ %s <span class="count">(%d)</span></a>',
                esc_url(admin_url('admin.php?page=formrank-lead-scoring&status=converted')),
                $current_status === 'converted' ? 'current' : '',
                __('Converted', 'formrank-lead-scoring'),
                $converted_count
            );
        }

        // Add Archived filter
        $archived_count = isset($status_counts['archived']) ? $status_counts['archived']->count : 0;
        if ($archived_count > 0 || $current_status === 'archived') {
            $views['archived'] = sprintf(
                '<a href="%s" class="%s" style="color: #6b7280;">%s <span class="count">(%d)</span></a>',
                esc_url(admin_url('admin.php?page=formrank-lead-scoring&status=archived')),
                $current_status === 'archived' ? 'current' : '',
                __('Archived', 'formrank-lead-scoring'),
                $archived_count
            );
        }

        return $views;
    }

    /**
     * Bulk actions
     */
    public function get_bulk_actions() {
        return [
            'mark_converted' => __('Mark as Converted', 'formrank-lead-scoring'),
            'ai_rescore' => __('Score with AI', 'formrank-lead-scoring'),
            'archive' => __('Archive', 'formrank-lead-scoring'),
            'rescore' => __('Rescore (Rules)', 'formrank-lead-scoring'),
            'delete' => __('Delete Permanently', 'formrank-lead-scoring'),
        ];
    }

    /**
     * Message to show when no leads found
     */
    public function no_items() {
        esc_html_e('No leads found.', 'formrank-lead-scoring');
    }
}
