<?php
/**
 * Lead Detail - single lead view
 *
 * @package FormRankLS\Admin
 */

namespace FormRankLS\Admin;

if (!defined('ABSPATH')) {
    exit;
}

class Lead_Detail {

    private $lead;

    /**
     * Get lead data by ID
     */
    public function get_lead(): ?array {
        if ($this->lead !== null) {
            return $this->lead;
        }

        $lead_id = isset($_GET['id']) ? absint(wp_unslash($_GET['id'])) : 0;

        if (!$lead_id) {
            return null;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        $this->lead = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE id = %d",
            $lead_id
        ), ARRAY_A);

        return $this->lead;
    }

    /**
     * Get formatted form data
     */
    public function get_form_data(): array {
        $lead = $this->get_lead();

        if (!$lead) {
            return [];
        }

        $form_data = json_decode($lead['form_data'], true);

        return is_array($form_data) ? $form_data : [];
    }

    /**
     * Get score factors
     */
    public function get_score_factors(): array {
        $lead = $this->get_lead();

        if (!$lead || empty($lead['score_factors'])) {
            return [];
        }

        $factors = json_decode($lead['score_factors'], true);

        return is_array($factors) ? $factors : [];
    }

    /**
     * Get score color
     */
    public function get_score_color(): string {
        $lead = $this->get_lead();
        $label = $lead['score_label'] ?? 'unknown';

        $colors = [
            'hot' => '#ef4444',
            'warm' => '#f59e0b',
            'neutral' => '#10b981',
            'cool' => '#3b82f6',
            'cold' => '#6b7280'
        ];

        return $colors[$label] ?? '#6b7280';
    }

    /**
     * Get score icon
     */
    public function get_score_icon(): string {
        $lead = $this->get_lead();
        $label = $lead['score_label'] ?? 'unknown';

        $icons = [
            'hot' => '&#128293;',    // fire
            'warm' => '&#9728;',     // sun
            'neutral' => '&#128994;', // green circle
            'cool' => '&#10052;',    // snowflake
            'cold' => '&#129482;'    // ice cube
        ];

        return $icons[$label] ?? '';
    }

    /**
     * Update lead status
     */
    public function update_status(string $status): bool {
        $lead = $this->get_lead();

        if (!$lead) {
            return false;
        }

        $valid_statuses = ['new', 'contacted', 'qualified', 'converted', 'archived'];

        if (!in_array($status, $valid_statuses)) {
            return false;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        return $wpdb->update(
            $table,
            [
                'status' => $status,
                'updated_at' => current_time('mysql')
            ],
            ['id' => $lead['id']]
        ) !== false;
    }

    /**
     * Update lead notes
     */
    public function update_notes(string $notes): bool {
        $lead = $this->get_lead();

        if (!$lead) {
            return false;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        return $wpdb->update(
            $table,
            [
                'notes' => sanitize_textarea_field($notes),
                'updated_at' => current_time('mysql')
            ],
            ['id' => $lead['id']]
        ) !== false;
    }

    /**
     * Update lead tags
     */
    public function update_tags(string $tags): bool {
        $lead = $this->get_lead();

        if (!$lead) {
            return false;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        // Sanitize tags
        $tag_array = explode(',', $tags);
        $tag_array = array_map('sanitize_text_field', $tag_array);
        $tag_array = array_map('trim', $tag_array);
        $tag_array = array_filter($tag_array);
        $clean_tags = implode(', ', $tag_array);

        return $wpdb->update(
            $table,
            [
                'tags' => $clean_tags,
                'updated_at' => current_time('mysql')
            ],
            ['id' => $lead['id']]
        ) !== false;
    }

    /**
     * Get tags as array
     */
    public function get_tags_array(): array {
        $lead = $this->get_lead();

        if (!$lead || empty($lead['tags'])) {
            return [];
        }

        $tags = explode(',', $lead['tags']);

        return array_map('trim', $tags);
    }

    /**
     * Get enrichment data for the current lead.
     *
     * @return array|null Decoded enrichment data or null if no enrichment exists.
     */
    public function get_enrichment_data(): ?array {
        $lead = $this->get_lead();

        if (!$lead) {
            return null;
        }

        $engine = new \FormRankLS\Core\Enrichment_Engine();
        $row    = $engine->get_enrichment_by_lead(absint($lead['id']));

        if (!$row || empty($row['enrichment_data'])) {
            return null;
        }

        $decoded = json_decode($row['enrichment_data'], true);

        if (!is_array($decoded)) {
            return null;
        }

        return [
            'row'        => $row,
            'person'     => $decoded['person']     ?? [],
            'company'    => $decoded['company']    ?? [],
            'confidence' => $decoded['confidence'] ?? 'not_found',
            'email_type' => $decoded['email_type'] ?? 'personal',
        ];
    }

    /**
     * Get a relative human-readable enrichment timestamp.
     *
     * @param string $enriched_at MySQL datetime string from the enrichments table.
     * @return string Human-readable "Enriched X ago" string, or empty string on failure.
     */
    public function get_enrichment_timestamp(string $enriched_at): string {
        $timestamp = strtotime($enriched_at);

        if (!$timestamp) {
            return '';
        }

        $diff = human_time_diff($timestamp, current_time('timestamp'));

        /* translators: %s: human-readable time difference, e.g. "3 days" */
        return sprintf(__('Enriched %s ago', 'formrank-lead-scoring'), $diff);
    }

    /**
     * Get enrichment availability for the current lead.
     *
     * @return array ['available' => bool, 'reason' => string]
     */
    public function get_enrichment_availability(): array {
        $lead = $this->get_lead();

        if (!$lead) {
            return [
                'available' => false,
                'reason'    => '',
            ];
        }

        $engine = new \FormRankLS\Core\Enrichment_Engine();

        return $engine->check_enrichment_availability(absint($lead['id']));
    }

    /**
     * Delete lead
     */
    public function delete_lead(): bool {
        $lead = $this->get_lead();

        if (!$lead) {
            return false;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        return $wpdb->delete($table, ['id' => $lead['id']]) !== false;
    }
}
