<?php
/**
 * Data Import Admin Page - Handles historical data import with user permission
 *
 * @package FormRankLS\Admin
 */

namespace FormRankLS\Admin;

if (!defined('ABSPATH')) {
    exit;
}

use FormRankLS\Core\Data_Importer;
use FormRankLS\Core\Local_Scoring_Engine;

class Data_Import {

    /**
     * Data importer instance
     */
    private $importer;

    /**
     * Scoring engine instance
     */
    private $scoring_engine;

    /**
     * Option name for import permissions
     */
    const PERMISSION_OPTION = 'formrank_import_permissions';

    public function __construct() {
        $this->importer = new Data_Importer();
        $this->scoring_engine = new Local_Scoring_Engine();

        // Use priority 20 to ensure parent menu 'formrank-lead-scoring' is registered first (at priority 10)
        add_action( 'admin_menu', [ $this, 'add_submenu_page' ], 20 );
        add_action( 'admin_init', [ $this, 'register_settings' ] );
        add_action( 'wp_ajax_formrank_import_entries', [ $this, 'ajax_import_entries' ] );
        add_action( 'wp_ajax_formrank_get_import_status', [ $this, 'ajax_get_import_status' ] );
        add_action( 'wp_ajax_formrank_grant_permission', [ $this, 'ajax_grant_permission' ] );
        add_action( 'wp_ajax_formrank_revoke_permission', [ $this, 'ajax_revoke_permission' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_import_scripts' ] );
    }

    /**
     * Add submenu page under FormRank Lead Scoring
     */
    public function add_submenu_page() {
        add_submenu_page(
            'formrank-lead-scoring',
            __( 'Import Data', 'formrank-lead-scoring' ),
            __( 'Import Data', 'formrank-lead-scoring' ),
            'manage_options',
            'formrank-lead-scoring-import',
            [ $this, 'render_page' ]
        );
    }

    /**
     * Register settings
     */
    public function register_settings() {
        register_setting( 'formrank_import_settings', self::PERMISSION_OPTION, [
            'type'              => 'array',
            'sanitize_callback' => [ $this, 'sanitize_permissions' ],
        ] );
    }

    /**
     * Get current permissions
     */
    public function get_permissions(): array {
        return get_option( self::PERMISSION_OPTION, [
            'granted'    => [],
            'granted_at' => [],
            'granted_by' => [],
        ] );
    }

    /**
     * Check if permission is granted for a plugin
     */
    public function has_permission( string $plugin_key ): bool {
        $permissions = $this->get_permissions();
        return in_array( $plugin_key, $permissions['granted'] ?? [], true );
    }

    /**
     * Sanitize permissions option
     */
    public function sanitize_permissions( $input ): array {
        $valid_plugins = [ 'wpforms', 'gravity', 'cf7', 'fluent', 'formidable', 'ninja' ];
        $sanitized = [
            'granted'    => [],
            'granted_at' => [],
            'granted_by' => [],
        ];

        if ( is_array( $input ) && ! empty( $input['granted'] ) && is_array( $input['granted'] ) ) {
            foreach ( $input['granted'] as $plugin_key ) {
                $plugin_key = sanitize_key( $plugin_key );
                if ( in_array( $plugin_key, $valid_plugins, true ) ) {
                    $sanitized['granted'][] = $plugin_key;
                }
            }
        }

        return $sanitized;
    }

    /**
     * Grant permission to access a plugin's data
     */
    public function grant_permission( string $plugin_key ): bool {
        $permissions = $this->get_permissions();

        if ( ! in_array( $plugin_key, $permissions['granted'], true ) ) {
            $permissions['granted'][] = $plugin_key;
            $permissions['granted_at'][ $plugin_key ] = current_time( 'mysql' );
            $permissions['granted_by'][ $plugin_key ] = get_current_user_id();
        }

        return update_option( self::PERMISSION_OPTION, $permissions );
    }

    /**
     * Revoke permission for a plugin
     */
    public function revoke_permission( string $plugin_key ): bool {
        $permissions = $this->get_permissions();

        $permissions['granted'] = array_diff( $permissions['granted'], [ $plugin_key ] );
        unset( $permissions['granted_at'][ $plugin_key ] );
        unset( $permissions['granted_by'][ $plugin_key ] );

        return update_option( self::PERMISSION_OPTION, $permissions );
    }

    /**
     * AJAX: Grant permission
     */
    public function ajax_grant_permission() {
        check_ajax_referer( 'formrank_import_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [ 'message' => 'Permission denied' ] );
        }

        $plugin_key = sanitize_text_field( wp_unslash( $_POST['plugin'] ?? '' ) );

        if ( empty( $plugin_key ) ) {
            wp_send_json_error( [ 'message' => 'Invalid plugin' ] );
        }

        $result = $this->grant_permission( $plugin_key );

        if ( $result ) {
            wp_send_json_success( [ 'message' => 'Permission granted' ] );
        } else {
            wp_send_json_error( [ 'message' => 'Failed to grant permission' ] );
        }
    }

    /**
     * AJAX: Revoke permission
     */
    public function ajax_revoke_permission() {
        check_ajax_referer( 'formrank_import_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [ 'message' => 'Permission denied' ] );
        }

        $plugin_key = sanitize_text_field( wp_unslash( $_POST['plugin'] ?? '' ) );

        if ( empty( $plugin_key ) ) {
            wp_send_json_error( [ 'message' => 'Invalid plugin' ] );
        }

        $result = $this->revoke_permission( $plugin_key );

        if ( $result ) {
            wp_send_json_success( [ 'message' => 'Permission revoked' ] );
        } else {
            wp_send_json_error( [ 'message' => 'Failed to revoke permission' ] );
        }
    }

    /**
     * AJAX: Import entries from a plugin
     */
    public function ajax_import_entries() {
        check_ajax_referer( 'formrank_import_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [ 'message' => 'Permission denied' ] );
        }

        $plugin_key = sanitize_text_field( wp_unslash( $_POST['plugin'] ?? '' ) );
        $form_ids = isset( $_POST['form_ids'] ) ? array_map( 'absint', wp_unslash( (array) $_POST['form_ids'] ) ) : [];
        $batch_size = absint( wp_unslash( $_POST['batch_size'] ?? 50 ) );
        $offset = absint( wp_unslash( $_POST['offset'] ?? 0 ) );

        // Check permission
        if ( ! $this->has_permission( $plugin_key ) ) {
            wp_send_json_error( [ 'message' => 'Permission not granted for this plugin. Please grant access first.' ] );
        }

        // Run import
        $result = $this->importer->import_from_plugin( $plugin_key, $form_ids, $batch_size, $offset );

        if ( $result['success'] ) {
            // Log the import
            $this->log_import( $plugin_key, $result );

            wp_send_json_success( $result );
        } else {
            wp_send_json_error( $result );
        }
    }

    /**
     * AJAX: Get import status
     */
    public function ajax_get_import_status() {
        check_ajax_referer( 'formrank_import_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [ 'message' => 'Permission denied' ] );
        }

        $summary = $this->importer->get_import_summary();
        $permissions = $this->get_permissions();
        $learning_stats = $this->scoring_engine->get_learning_stats();

        wp_send_json_success( [
            'summary'        => $summary,
            'permissions'    => $permissions,
            'learning_stats' => $learning_stats,
        ] );
    }

    /**
     * Log import activity
     */
    private function log_import( string $plugin_key, array $result ): void {
        $log = get_option( 'formrank_import_log', [] );

        $log[] = [
            'plugin'    => $plugin_key,
            'imported'  => $result['imported'] ?? 0,
            'skipped'   => $result['skipped'] ?? 0,
            'errors'    => count( $result['errors'] ?? [] ),
            'timestamp' => current_time( 'mysql' ),
            'user_id'   => get_current_user_id(),
        ];

        // Keep last 100 log entries
        $log = array_slice( $log, -100 );

        update_option( 'formrank_import_log', $log );
    }

    /**
     * Render the import page
     */
    public function render_page() {
        $summary = $this->importer->get_import_summary();
        $permissions = $this->get_permissions();
        $learning_stats = $this->scoring_engine->get_learning_stats();
        $import_log = get_option( 'formrank_import_log', [] );

        ?>
        <div class="wrap formrank-import-page">
            <h1><?php esc_html_e( 'Import Historical Data', 'formrank-lead-scoring' ); ?></h1>

            <!-- Learning Status Card -->
            <div class="formrank-card formrank-learning-status">
                <h2><?php esc_html_e( 'AI Learning Status', 'formrank-lead-scoring' ); ?></h2>

                <div class="formrank-stats-grid">
                    <div class="formrank-stat">
                        <span class="stat-value"><?php echo esc_html( $learning_stats['total_leads'] ); ?></span>
                        <span class="stat-label"><?php esc_html_e( 'Total Leads', 'formrank-lead-scoring' ); ?></span>
                    </div>
                    <div class="formrank-stat">
                        <span class="stat-value"><?php echo esc_html( $learning_stats['total_conversions'] ); ?></span>
                        <span class="stat-label"><?php esc_html_e( 'Conversions', 'formrank-lead-scoring' ); ?></span>
                    </div>
                    <div class="formrank-stat">
                        <span class="stat-value"><?php echo esc_html( $learning_stats['conversion_rate'] ); ?>%</span>
                        <span class="stat-label"><?php esc_html_e( 'Conversion Rate', 'formrank-lead-scoring' ); ?></span>
                    </div>
                    <div class="formrank-stat">
                        <span class="stat-value formrank-mode-<?php echo esc_attr( $learning_stats['scoring_mode'] ); ?>">
                            <?php echo esc_html( ucwords( str_replace( '_', ' ', $learning_stats['scoring_mode'] ) ) ); ?>
                        </span>
                        <span class="stat-label"><?php esc_html_e( 'Scoring Mode', 'formrank-lead-scoring' ); ?></span>
                    </div>
                </div>

                <?php if ( $learning_stats['scoring_mode'] === 'cold_start' ) : ?>
                    <div class="formrank-notice formrank-notice-info">
                        <p>
                            <strong><?php esc_html_e( 'Cold Start Mode', 'formrank-lead-scoring' ); ?></strong><br>
                            <?php
                            printf(
                                /* translators: 1: current lead count, 2: required lead count */
                                esc_html__( 'You have %1$d leads. Import at least %2$d leads to enable enhanced scoring.', 'formrank-lead-scoring' ),
                                absint( $learning_stats['total_leads'] ),
                                50
                            );
                            ?>
                        </p>
                    </div>
                <?php elseif ( $learning_stats['scoring_mode'] === 'enhanced_rules' ) : ?>
                    <div class="formrank-notice formrank-notice-warning">
                        <p>
                            <strong><?php esc_html_e( 'Enhanced Rules Mode', 'formrank-lead-scoring' ); ?></strong><br>
                            <?php
                            printf(
                                /* translators: 1: current conversion count, 2: required conversion count */
                                esc_html__( 'You have %1$d conversions. Mark at least %2$d leads as "converted" to enable full learning mode.', 'formrank-lead-scoring' ),
                                absint( $learning_stats['total_conversions'] ),
                                10
                            );
                            ?>
                        </p>
                    </div>
                <?php else : ?>
                    <div class="formrank-notice formrank-notice-success">
                        <p>
                            <strong><?php esc_html_e( 'Learning Mode Active!', 'formrank-lead-scoring' ); ?></strong><br>
                            <?php esc_html_e( 'The AI is learning from your conversion data and improving scoring accuracy.', 'formrank-lead-scoring' ); ?>
                        </p>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Available Plugins -->
            <div class="formrank-card">
                <h2><?php esc_html_e( 'Available Form Plugins', 'formrank-lead-scoring' ); ?></h2>

                <?php if ( empty( $summary['plugins'] ) ) : ?>
                    <p><?php esc_html_e( 'No supported form plugins detected.', 'formrank-lead-scoring' ); ?></p>
                    <p><?php esc_html_e( 'Supported plugins: WPForms, Gravity Forms, Contact Form 7 (with Flamingo), Fluent Forms, Formidable Forms, Ninja Forms.', 'formrank-lead-scoring' ); ?></p>
                <?php else : ?>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php esc_html_e( 'Plugin', 'formrank-lead-scoring' ); ?></th>
                                <th><?php esc_html_e( 'Forms', 'formrank-lead-scoring' ); ?></th>
                                <th><?php esc_html_e( 'Entries', 'formrank-lead-scoring' ); ?></th>
                                <th><?php esc_html_e( 'Permission', 'formrank-lead-scoring' ); ?></th>
                                <th><?php esc_html_e( 'Actions', 'formrank-lead-scoring' ); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ( $summary['plugins'] as $key => $plugin ) : ?>
                                <tr data-plugin="<?php echo esc_attr( $key ); ?>">
                                    <td>
                                        <strong><?php echo esc_html( $plugin['name'] ); ?></strong>
                                    </td>
                                    <td><?php echo esc_html( $plugin['form_count'] ); ?></td>
                                    <td><?php echo esc_html( number_format( $plugin['entry_count'] ) ); ?></td>
                                    <td class="permission-cell">
                                        <?php if ( $this->has_permission( $key ) ) : ?>
                                            <span class="formrank-badge formrank-badge-success">
                                                <?php esc_html_e( 'Granted', 'formrank-lead-scoring' ); ?>
                                            </span>
                                        <?php else : ?>
                                            <span class="formrank-badge formrank-badge-warning">
                                                <?php esc_html_e( 'Not Granted', 'formrank-lead-scoring' ); ?>
                                            </span>
                                        <?php endif; ?>
                                    </td>
                                    <td class="actions-cell">
                                        <?php if ( ! $this->has_permission( $key ) ) : ?>
                                            <button type="button"
                                                    class="button button-primary formrank-grant-permission"
                                                    data-plugin="<?php echo esc_attr( $key ); ?>">
                                                <?php esc_html_e( 'Grant Access', 'formrank-lead-scoring' ); ?>
                                            </button>
                                        <?php else : ?>
                                            <?php if ( $plugin['entry_count'] > 0 ) : ?>
                                                <button type="button"
                                                        class="button button-primary formrank-import-plugin"
                                                        data-plugin="<?php echo esc_attr( $key ); ?>"
                                                        data-count="<?php echo esc_attr( $plugin['entry_count'] ); ?>">
                                                    <?php esc_html_e( 'Import All', 'formrank-lead-scoring' ); ?>
                                                </button>
                                            <?php endif; ?>
                                            <button type="button"
                                                    class="button formrank-revoke-permission"
                                                    data-plugin="<?php echo esc_attr( $key ); ?>">
                                                <?php esc_html_e( 'Revoke', 'formrank-lead-scoring' ); ?>
                                            </button>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>

            <!-- Import Progress -->
            <div class="formrank-card formrank-import-progress" style="display: none;">
                <h2><?php esc_html_e( 'Import Progress', 'formrank-lead-scoring' ); ?></h2>
                <div class="formrank-progress-bar">
                    <div class="formrank-progress-fill" style="width: 0%;"></div>
                </div>
                <p class="formrank-progress-text">
                    <span class="imported-count">0</span> / <span class="total-count">0</span>
                    <?php esc_html_e( 'entries imported', 'formrank-lead-scoring' ); ?>
                </p>
                <p class="formrank-progress-status"></p>
            </div>

            <!-- Permission Request Modal -->
            <div class="formrank-permission-modal" id="permission-modal" style="display: none;">
                <div class="formrank-permission-modal-content">
                    <h2><?php esc_html_e( 'Grant Data Access Permission', 'formrank-lead-scoring' ); ?></h2>

                    <div class="formrank-permission-info">
                        <p>
                            <?php esc_html_e( 'FormRank needs your permission to access form entries from', 'formrank-lead-scoring' ); ?>
                            <strong class="plugin-name"></strong>.
                        </p>

                        <h3><?php esc_html_e( 'What we will access:', 'formrank-lead-scoring' ); ?></h3>
                        <ul>
                            <li><?php esc_html_e( 'Form submission data (names, emails, messages, etc.)', 'formrank-lead-scoring' ); ?></li>
                            <li><?php esc_html_e( 'Submission timestamps', 'formrank-lead-scoring' ); ?></li>
                            <li><?php esc_html_e( 'Form IDs and entry IDs', 'formrank-lead-scoring' ); ?></li>
                        </ul>

                        <h3><?php esc_html_e( 'What we will NOT do:', 'formrank-lead-scoring' ); ?></h3>
                        <ul>
                            <li><?php esc_html_e( 'We will NOT modify or delete any original data', 'formrank-lead-scoring' ); ?></li>
                            <li><?php esc_html_e( 'We will NOT send data to external servers', 'formrank-lead-scoring' ); ?></li>
                            <li><?php esc_html_e( 'All processing happens locally on your server', 'formrank-lead-scoring' ); ?></li>
                        </ul>

                        <h3><?php esc_html_e( 'Why we need this:', 'formrank-lead-scoring' ); ?></h3>
                        <p>
                            <?php esc_html_e( 'Importing historical data helps the AI learn your lead patterns and improve scoring accuracy. The more data available, the better the AI can identify high-quality leads.', 'formrank-lead-scoring' ); ?>
                        </p>
                    </div>

                    <div class="formrank-modal-actions">
                        <button type="button" class="button button-primary" id="confirm-permission">
                            <?php esc_html_e( 'Grant Permission', 'formrank-lead-scoring' ); ?>
                        </button>
                        <button type="button" class="button" id="cancel-permission">
                            <?php esc_html_e( 'Cancel', 'formrank-lead-scoring' ); ?>
                        </button>
                    </div>
                </div>
            </div>

            <!-- Recent Import Log -->
            <?php if ( ! empty( $import_log ) ) : ?>
                <div class="formrank-card">
                    <h2><?php esc_html_e( 'Recent Import History', 'formrank-lead-scoring' ); ?></h2>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php esc_html_e( 'Date', 'formrank-lead-scoring' ); ?></th>
                                <th><?php esc_html_e( 'Plugin', 'formrank-lead-scoring' ); ?></th>
                                <th><?php esc_html_e( 'Imported', 'formrank-lead-scoring' ); ?></th>
                                <th><?php esc_html_e( 'Skipped', 'formrank-lead-scoring' ); ?></th>
                                <th><?php esc_html_e( 'Errors', 'formrank-lead-scoring' ); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            $recent_log = array_slice( array_reverse( $import_log ), 0, 10 );
                            foreach ( $recent_log as $entry ) :
                                ?>
                                <tr>
                                    <td><?php echo esc_html( $entry['timestamp'] ); ?></td>
                                    <td><?php echo esc_html( ucfirst( $entry['plugin'] ) ); ?></td>
                                    <td><?php echo esc_html( $entry['imported'] ); ?></td>
                                    <td><?php echo esc_html( $entry['skipped'] ); ?></td>
                                    <td><?php echo esc_html( $entry['errors'] ); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Enqueue import page scripts
     *
     * @param string $hook Current admin page hook.
     */
    public function enqueue_import_scripts( $hook ) {
        if ( strpos( $hook, 'formrank-lead-scoring-import' ) === false ) {
            return;
        }

        wp_enqueue_script(
            'formrank-lead-scoring-import',
            FORMRANK_LS_URL . 'assets/js/admin-import.js',
            [ 'jquery' ],
            FORMRANK_LS_VERSION,
            true
        );

        wp_localize_script( 'formrank-lead-scoring-import', 'formrankImport', [
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'nonce'    => wp_create_nonce( 'formrank_import_nonce' ),
            'strings'  => [
                'confirm_revoke'   => __( 'Are you sure you want to revoke access? You can grant it again later.', 'formrank-lead-scoring' ),
                'confirm_import'   => __( 'This will import all entries from this plugin. Continue?', 'formrank-lead-scoring' ),
                'starting'         => __( 'Starting import...', 'formrank-lead-scoring' ),
                'imported'         => __( 'Imported:', 'formrank-lead-scoring' ),
                'skipped'          => __( 'Skipped:', 'formrank-lead-scoring' ),
                'complete'         => __( 'Import complete!', 'formrank-lead-scoring' ),
                'error'            => __( 'Error:', 'formrank-lead-scoring' ),
                'connection_error' => __( 'Connection error. Please try again.', 'formrank-lead-scoring' ),
                'grant_failed'     => __( 'Failed to grant permission', 'formrank-lead-scoring' ),
                'revoke_failed'    => __( 'Failed to revoke permission', 'formrank-lead-scoring' ),
            ],
        ] );
    }
}
