<?php
/**
 * Dashboard - analytics and overview
 *
 * @package FormRankLS\Admin
 */

namespace FormRankLS\Admin;

if (!defined('ABSPATH')) {
    exit;
}

use FormRankLS\Core\API_Client;
use FormRankLS\Integrations\Integration_Base;

class Dashboard {

    /**
     * Get dashboard statistics
     */
    public function get_stats(): array {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        // Total leads
        $total_leads = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table}");

        // Leads by label
        $leads_by_label = $wpdb->get_results(
            "SELECT score_label, COUNT(*) as count
             FROM {$table}
             WHERE score_label IS NOT NULL
             GROUP BY score_label",
            OBJECT_K
        );

        // Leads this month
        $month_start = gmdate('Y-m-01 00:00:00');
        $leads_this_month = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table} WHERE created_at >= %s",
            $month_start
        ));

        // Average score
        $avg_score = (float) $wpdb->get_var(
            "SELECT AVG(score) FROM {$table} WHERE score IS NOT NULL"
        );

        // Recent leads
        $recent_leads = $wpdb->get_results(
            "SELECT id, name, email, company, score, score_label, created_at
             FROM {$table}
             ORDER BY created_at DESC
             LIMIT 5",
            ARRAY_A
        );

        // Hot leads this week
        $week_start = gmdate('Y-m-d 00:00:00', strtotime('-7 days'));
        $hot_leads_week = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table}
             WHERE score_label = 'hot' AND created_at >= %s",
            $week_start
        ));

        // Leads by day (last 7 days)
        $leads_by_day = $wpdb->get_results(
            "SELECT DATE(created_at) as date, COUNT(*) as count
             FROM {$table}
             WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
             GROUP BY DATE(created_at)
             ORDER BY date ASC",
            ARRAY_A
        );

        // Leads by source form
        $leads_by_form = $wpdb->get_results(
            "SELECT form_plugin, form_id, COUNT(*) as count
             FROM {$table}
             GROUP BY form_plugin, form_id
             ORDER BY count DESC
             LIMIT 5",
            ARRAY_A
        );

        // Usage stats
        $integration = new Integration_Base();
        $usage_stats = $integration->get_usage_stats();

        // API status
        $api_client = new API_Client();
        $api_configured = $api_client->has_api_key();

        return [
            'total_leads' => $total_leads,
            'leads_by_label' => [
                'hot' => $leads_by_label['hot']->count ?? 0,
                'warm' => $leads_by_label['warm']->count ?? 0,
                'neutral' => $leads_by_label['neutral']->count ?? 0,
                'cool' => $leads_by_label['cool']->count ?? 0,
                'cold' => $leads_by_label['cold']->count ?? 0
            ],
            'leads_this_month' => $leads_this_month,
            'avg_score' => round($avg_score, 1),
            'recent_leads' => $recent_leads,
            'hot_leads_week' => $hot_leads_week,
            'leads_by_day' => $leads_by_day,
            'leads_by_form' => $leads_by_form,
            'usage' => $usage_stats,
            'api_configured' => $api_configured
        ];
    }

    /**
     * Get score distribution for chart
     */
    public function get_score_distribution(): array {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        $distribution = $wpdb->get_results(
            "SELECT
                CASE
                    WHEN score >= 80 THEN '80-100'
                    WHEN score >= 60 THEN '60-79'
                    WHEN score >= 40 THEN '40-59'
                    WHEN score >= 20 THEN '20-39'
                    ELSE '1-19'
                END as range_label,
                COUNT(*) as count
             FROM {$table}
             WHERE score IS NOT NULL
             GROUP BY range_label
             ORDER BY
                CASE range_label
                    WHEN '80-100' THEN 1
                    WHEN '60-79' THEN 2
                    WHEN '40-59' THEN 3
                    WHEN '20-39' THEN 4
                    ELSE 5
                END",
            ARRAY_A
        );

        return $distribution;
    }

    /**
     * Get conversion funnel data
     */
    public function get_funnel_data(): array {
        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        $statuses = $wpdb->get_results(
            "SELECT status, COUNT(*) as count
             FROM {$table}
             GROUP BY status",
            OBJECT_K
        );

        return [
            'new' => $statuses['new']->count ?? 0,
            'contacted' => $statuses['contacted']->count ?? 0,
            'qualified' => $statuses['qualified']->count ?? 0,
            'archived' => $statuses['archived']->count ?? 0
        ];
    }
}
