<?php
/**
 * Admin Controller - main admin functionality
 *
 * @package FormRankLS\Admin
 */

namespace FormRankLS\Admin;

if (!defined('ABSPATH')) {
    exit;
}

use FormRankLS\Core\API_Client;
use FormRankLS\Core\Scoring_Engine;
use FormRankLS\Integrations\Integration_Base;

class Admin {

    private $plugin_name;
    private $version;

    public function __construct($plugin_name, $version) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
    }

    /**
     * Get asset file suffix (empty for dev, .min for production)
     *
     * @return string Asset suffix
     */
    private function get_asset_suffix(): string {
        // Use minified assets unless SCRIPT_DEBUG is enabled
        if (defined('SCRIPT_DEBUG') && SCRIPT_DEBUG) {
            return '';
        }

        // Check if minified files exist
        $min_css = FORMRANK_LS_PATH . 'assets/css/admin.min.css';
        $min_js = FORMRANK_LS_PATH . 'assets/js/admin.min.js';

        if (file_exists($min_css) && file_exists($min_js)) {
            return '.min';
        }

        return '';
    }

    /**
     * Register the admin menu
     */
    public function add_admin_menu() {
        // Custom icon - use PNG file from assets/images/
        // If menu-icon.png doesn't exist, fall back to dashicons
        $icon_path = FORMRANK_LS_PATH . 'assets/images/menu-icon.png';
        $icon_url = file_exists($icon_path)
            ? FORMRANK_LS_URL . 'assets/images/menu-icon.png'
            : 'dashicons-chart-line';

        // Main menu
        add_menu_page(
            __('FormRank Lead Scoring', 'formrank-lead-scoring'),
            __('FormRank', 'formrank-lead-scoring'),
            'manage_options',
            'formrank-lead-scoring',
            [$this, 'render_leads_page'],
            $icon_url,
            30
        );

        // Leads submenu (same as main)
        add_submenu_page(
            'formrank-lead-scoring',
            __('All Leads', 'formrank-lead-scoring'),
            __('All Leads', 'formrank-lead-scoring'),
            'manage_options',
            'formrank-lead-scoring',
            [$this, 'render_leads_page']
        );

        // Dashboard submenu
        add_submenu_page(
            'formrank-lead-scoring',
            __('Dashboard', 'formrank-lead-scoring'),
            __('Dashboard', 'formrank-lead-scoring'),
            'manage_options',
            'formrank-lead-scoring-dashboard',
            [$this, 'render_dashboard_page']
        );

        // Conversions submenu
        add_submenu_page(
            'formrank-lead-scoring',
            __('Conversions', 'formrank-lead-scoring'),
            __('Conversions', 'formrank-lead-scoring'),
            'manage_options',
            'formrank-lead-scoring-conversions',
            [$this, 'render_conversions_page']
        );

        // Settings submenu
        add_submenu_page(
            'formrank-lead-scoring',
            __('Settings', 'formrank-lead-scoring'),
            __('Settings', 'formrank-lead-scoring'),
            'manage_options',
            'formrank-lead-scoring-settings',
            [$this, 'render_settings_page']
        );

        // Hidden lead detail page
        add_submenu_page(
            null,
            __('Lead Details', 'formrank-lead-scoring'),
            __('Lead Details', 'formrank-lead-scoring'),
            'manage_options',
            'formrank-lead-scoring-lead',
            [$this, 'render_lead_detail_page']
        );
    }

    /**
     * Register plugin settings
     */
    public function register_settings() {
        $settings = new Settings();
        $settings->register();
    }

    /**
     * Enqueue admin styles
     */
    public function enqueue_styles($hook) {
        if (strpos($hook, 'formrank-lead-scoring') === false) {
            return;
        }

        // Use minified version in production
        $suffix = $this->get_asset_suffix();

        wp_enqueue_style(
            $this->plugin_name,
            FORMRANK_LS_URL . 'assets/css/admin' . $suffix . '.css',
            [],
            $this->version
        );
    }

    /**
     * Enqueue admin scripts
     */
    public function enqueue_scripts($hook) {
        if (strpos($hook, 'formrank-lead-scoring') === false) {
            return;
        }

        // Use minified version in production
        $suffix = $this->get_asset_suffix();

        wp_enqueue_script(
            $this->plugin_name,
            FORMRANK_LS_URL . 'assets/js/admin' . $suffix . '.js',
            ['jquery'],
            $this->version,
            true
        );

        wp_localize_script($this->plugin_name, 'formrankLS', [
            'ajax_url'    => admin_url('admin-ajax.php'),
            'nonce'       => wp_create_nonce('formrank_ajax_nonce'),
            'rest_url'    => rest_url(),
            'rest_nonce'  => wp_create_nonce('wp_rest'),
            'upgrade_url' => class_exists('\FormRankLS\Admin\Upgrade_Notice') ? \FormRankLS\Admin\Upgrade_Notice::get_upgrade_url() : '',
            'lead_id'     => absint(isset($_GET['id']) ? $_GET['id'] : 0), // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            'auto_enrich' => isset($_GET['enrich']) && $_GET['enrich'] === '1' ? true : false, // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            'confirm_remove_demo' => __('Remove all demo leads? This cannot be undone.', 'formrank-lead-scoring'),
            'removing' => __('Removing...', 'formrank-lead-scoring'),
            'remove_demo' => __('Remove Demo Data', 'formrank-lead-scoring'),
            'strings' => [
                // Confirm modal buttons
                'confirm_btn' => __('Confirm', 'formrank-lead-scoring'),
                'cancel_btn' => __('Cancel', 'formrank-lead-scoring'),
                'close_btn' => __('Close', 'formrank-lead-scoring'),
                'upgrade_btn' => __('Upgrade to PRO', 'formrank-lead-scoring'),
                // Rescore
                'rescore_title' => __('Rescore Lead', 'formrank-lead-scoring'),
                'confirm_rescore' => __('Are you sure you want to rescore this lead using the rule-based engine?', 'formrank-lead-scoring'),
                'rescore_confirm_btn' => __('Rescore', 'formrank-lead-scoring'),
                'rescoring' => __('Rescoring...', 'formrank-lead-scoring'),
                // AI Rescore
                'ai_rescore_title' => __('Score with AI', 'formrank-lead-scoring'),
                'confirm_ai_rescore' => __('Score this lead with AI? This will use 1 of your monthly AI rescores.', 'formrank-lead-scoring'),
                'ai_rescore_confirm_btn' => __('Score with AI', 'formrank-lead-scoring'),
                'ai_scoring' => __('AI Scoring...', 'formrank-lead-scoring'),
                'ai_success' => __('AI Scored!', 'formrank-lead-scoring'),
                'ai_limit_title' => __('AI Limit Reached', 'formrank-lead-scoring'),
                'ai_limit_reached' => __('You\'ve reached your monthly AI rescore limit. Upgrade to Pro for unlimited AI scoring.', 'formrank-lead-scoring'),
                'ai_scored_label' => __('AI Scored', 'formrank-lead-scoring'),
                'ai_scored_tooltip' => __('Scored using Claude AI', 'formrank-lead-scoring'),
                // Enrichment
                'enrich_confirm_title' => __('Enrich Lead', 'formrank-lead-scoring'),
                'enrich_confirm_msg'   => __('This will use 1 of your enrichments this month. Continue?', 'formrank-lead-scoring'),
                'enrich_btn'           => __('Enrich', 'formrank-lead-scoring'),
                'enriching'            => __('Enriching...', 'formrank-lead-scoring'),
                'enrich_success'       => __('Enriched!', 'formrank-lead-scoring'),
                'enrich_error'         => __('Enrichment failed.', 'formrank-lead-scoring'),
                'enrich_cooldown'      => __('Please wait 30 seconds before re-enriching.', 'formrank-lead-scoring'),
                'enrich_no_quota'      => __('No enrichments remaining this month.', 'formrank-lead-scoring'),
                'enrich_upgrade_title' => __('PRO Feature', 'formrank-lead-scoring'),
                'enrich_upgrade_msg'   => __('Lead enrichment requires a PRO plan. Upgrade to access AI-powered lead intelligence.', 'formrank-lead-scoring'),
                'enrich_timeout'       => __('Enrichment is taking longer than expected. The data may appear shortly — try refreshing.', 'formrank-lead-scoring'),
                'show_more'            => __('Show more', 'formrank-lead-scoring'),
                'show_less'            => __('Show less', 'formrank-lead-scoring'),
                'copied'               => __('Copied!', 'formrank-lead-scoring'),
                're_enrich'            => __('Re-enrich', 'formrank-lead-scoring'),
                'disclaimer'           => __('Data sourced from public profiles. Verify before acting.', 'formrank-lead-scoring'),
                'no_person_data'       => __('No person data found', 'formrank-lead-scoring'),
                'no_company_data'      => __('No company data found', 'formrank-lead-scoring'),
                'enriched'             => __('Enriched', 'formrank-lead-scoring'),
                // Conversions
                'convert_title' => __('Mark as Converted', 'formrank-lead-scoring'),
                'confirm_converted' => __('Mark this lead as converted? This helps improve AI scoring accuracy.', 'formrank-lead-scoring'),
                'convert_confirm_btn' => __('Mark Converted', 'formrank-lead-scoring'),
                // Bulk actions
                'bulk_delete_title' => __('Delete Leads', 'formrank-lead-scoring'),
                /* translators: %d: number of selected leads */
                'bulk_delete_msg' => __('Permanently delete <strong>%d</strong> selected lead(s)? This cannot be undone.', 'formrank-lead-scoring'),
                'bulk_delete_btn' => __('Delete Permanently', 'formrank-lead-scoring'),
                'bulk_archive_title' => __('Archive Leads', 'formrank-lead-scoring'),
                /* translators: %d: number of selected leads */
                'bulk_archive_msg' => __('Archive <strong>%d</strong> selected lead(s)?', 'formrank-lead-scoring'),
                'bulk_archive_btn' => __('Archive', 'formrank-lead-scoring'),
                'bulk_convert_title' => __('Mark as Converted', 'formrank-lead-scoring'),
                /* translators: %d: number of selected leads */
                'bulk_convert_msg' => __('Mark <strong>%d</strong> selected lead(s) as converted? This helps improve AI scoring accuracy.', 'formrank-lead-scoring'),
                'bulk_convert_btn' => __('Mark Converted', 'formrank-lead-scoring'),
                'bulk_rescore_title' => __('Rescore Leads', 'formrank-lead-scoring'),
                /* translators: %d: number of selected leads */
                'bulk_rescore_msg' => __('Rescore <strong>%d</strong> selected lead(s) using rule-based engine?', 'formrank-lead-scoring'),
                'bulk_rescore_btn' => __('Rescore', 'formrank-lead-scoring'),
                'bulk_ai_title' => __('Score with AI', 'formrank-lead-scoring'),
                /* translators: %d: number of selected leads */
                'bulk_ai_msg' => __('Score <strong>%d</strong> selected lead(s) with AI? This will use your monthly AI rescore quota.', 'formrank-lead-scoring'),
                'bulk_ai_btn' => __('Score with AI', 'formrank-lead-scoring'),
                // General
                'success' => __('Success!', 'formrank-lead-scoring'),
                'error' => __('An error occurred.', 'formrank-lead-scoring')
            ]
        ]);
    }

    /**
     * Render leads list page
     */
    public function render_leads_page() {
        $lead_list = new Lead_List();
        $lead_list->prepare_items();

        include FORMRANK_LS_PATH . 'templates/admin/lead-list.php';
    }

    /**
     * Handle CSV export request
     *
     * Must be called on admin_init hook (before any output)
     */
    public function handle_csv_export(): void {
        // Only run on our page with export action
        if (!isset($_GET['page']) || sanitize_text_field(wp_unslash($_GET['page'])) !== 'formrank-lead-scoring') {
            return;
        }

        if (!isset($_GET['action']) || sanitize_text_field(wp_unslash($_GET['action'])) !== 'export_csv') {
            return;
        }

        // Verify nonce
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'formrank_export_csv')) {
            wp_die(esc_html__('Security check failed.', 'formrank-lead-scoring'));
        }

        // Check capabilities
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Permission denied.', 'formrank-lead-scoring'));
        }

        // Generate filename and set headers early, before any DB queries that could trigger warnings
        $filename = 'formrank-export-' . gmdate('Y-m-d-His') . '.csv';
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Pragma: no-cache');
        header('Expires: 0');

        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';
        $enrichments_table = $wpdb->prefix . 'formrank_enrichments';

        // Guard: check if the enrichments table exists (defensive for pre-1.3.0 databases)
        $has_enrichments_table = (bool) $wpdb->get_var(
            $wpdb->prepare('SHOW TABLES LIKE %s', $enrichments_table)
        );

        // Build query based on filters
        $where = '1=1';
        $params = [];

        // Filter by score label
        if (!empty($_GET['label'])) {
            $where .= ' AND l.score_label = %s';
            $params[] = sanitize_text_field(wp_unslash($_GET['label']));
        }

        // Filter by status
        if (!empty($_GET['status'])) {
            $where .= ' AND l.status = %s';
            $params[] = sanitize_text_field(wp_unslash($_GET['status']));
        }

        // Search filter
        if (!empty($_GET['s'])) {
            $search = '%' . $wpdb->esc_like(sanitize_text_field(wp_unslash($_GET['s']))) . '%';
            $where .= ' AND (l.name LIKE %s OR l.email LIKE %s OR l.company LIKE %s)';
            $params = array_merge($params, [$search, $search, $search]);
        }

        // Get leads — LEFT JOIN enrichments when table exists
        if ($has_enrichments_table) {
            $query = "SELECT l.*,
                e.job_title AS enrich_job_title,
                e.company_name AS enrich_company_name,
                e.seniority_level AS enrich_seniority_level,
                e.linkedin_url AS enrich_linkedin_url,
                e.company_industry AS enrich_company_industry,
                e.company_size AS enrich_company_size,
                e.confidence AS enrich_confidence,
                e.enriched_at AS enrich_enriched_at
            FROM {$table} l
            LEFT JOIN {$enrichments_table} e ON e.lead_id = l.id
            WHERE {$where} ORDER BY l.created_at DESC";
        } else {
            $query = "SELECT * FROM {$table} l WHERE {$where} ORDER BY l.created_at DESC";
        }

        // phpcs:disable WordPress.DB.PreparedSQL.NotPrepared -- Table names are safely constructed from $wpdb->prefix.
        $leads = $wpdb->get_results(
            $params ? $wpdb->prepare($query, $params) : $query,
            // phpcs:enable WordPress.DB.PreparedSQL.NotPrepared
            ARRAY_A
        );

        // Open output stream
        $output = fopen('php://output', 'w');

        // Add BOM for Excel compatibility with UTF-8
        fprintf($output, chr(0xEF) . chr(0xBB) . chr(0xBF));

        // CSV headers (15 standard + 8 enrichment)
        $headers = [
            __('ID', 'formrank-lead-scoring'),
            __('Name', 'formrank-lead-scoring'),
            __('Email', 'formrank-lead-scoring'),
            __('Company', 'formrank-lead-scoring'),
            __('Phone', 'formrank-lead-scoring'),
            __('Score', 'formrank-lead-scoring'),
            __('Score Label', 'formrank-lead-scoring'),
            __('Status', 'formrank-lead-scoring'),
            __('Scoring Method', 'formrank-lead-scoring'),
            __('Score Reasoning', 'formrank-lead-scoring'),
            __('Form Source', 'formrank-lead-scoring'),
            __('Source URL', 'formrank-lead-scoring'),
            __('Notes', 'formrank-lead-scoring'),
            __('Created At', 'formrank-lead-scoring'),
            __('Scored At', 'formrank-lead-scoring'),
            __('Enriched Job Title', 'formrank-lead-scoring'),
            __('Enriched Company', 'formrank-lead-scoring'),
            __('Enriched Seniority', 'formrank-lead-scoring'),
            __('Enriched LinkedIn URL', 'formrank-lead-scoring'),
            __('Enriched Industry', 'formrank-lead-scoring'),
            __('Enriched Company Size', 'formrank-lead-scoring'),
            __('Enrichment Confidence', 'formrank-lead-scoring'),
            __('Enriched At', 'formrank-lead-scoring'),
        ];

        fputcsv($output, $headers);

        // Write data rows
        foreach ($leads as $lead) {
            $row = [
                $lead['id'],
                $lead['name'] ?: '',
                $lead['email'] ?: '',
                $lead['company'] ?: '',
                $lead['phone'] ?: '',
                $lead['score'] ?: '',
                $lead['score_label'] ?: '',
                $lead['status'] ?: '',
                $lead['scoring_method'] ?: 'local',
                $lead['score_reasoning'] ?: '',
                ucfirst($lead['form_plugin']) . ' #' . $lead['form_id'],
                $lead['source_url'] ?: '',
                $lead['notes'] ?: '',
                $lead['created_at'] ?: '',
                $lead['scored_at'] ?: '',
                $lead['enrich_job_title'] ?? '',
                $lead['enrich_company_name'] ?? '',
                $lead['enrich_seniority_level'] ?? '',
                $lead['enrich_linkedin_url'] ?? '',
                $lead['enrich_company_industry'] ?? '',
                $lead['enrich_company_size'] ?? '',
                $lead['enrich_confidence'] ?? '',
                $lead['enrich_enriched_at'] ?? '',
            ];

            fputcsv($output, $row);
        }

        fclose($output); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Closing php://output stream for CSV export.
        exit;
    }

    /**
     * Handle JSON export request
     *
     * Must be called on admin_init hook (before any output)
     */
    public function handle_json_export(): void {
        // Only run on our page with export action
        if (!isset($_GET['page']) || sanitize_text_field(wp_unslash($_GET['page'])) !== 'formrank-lead-scoring') {
            return;
        }

        if (!isset($_GET['action']) || sanitize_text_field(wp_unslash($_GET['action'])) !== 'export_json') {
            return;
        }

        // Verify nonce
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'formrank_export_json')) {
            wp_die(esc_html__('Security check failed.', 'formrank-lead-scoring'));
        }

        // Check capabilities
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Permission denied.', 'formrank-lead-scoring'));
        }

        // Set JSON download headers
        $filename = 'formrank-export-' . gmdate('Y-m-d-His') . '.json';
        header('Content-Type: application/json; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Pragma: no-cache');
        header('Expires: 0');

        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';
        $enrichments_table = $wpdb->prefix . 'formrank_enrichments';

        // Guard: check if the enrichments table exists (defensive for pre-1.3.0 databases)
        $has_enrichments_table = (bool) $wpdb->get_var(
            $wpdb->prepare('SHOW TABLES LIKE %s', $enrichments_table)
        );

        // Build query based on filters
        $where = '1=1';
        $params = [];

        // Filter by score label
        if (!empty($_GET['label'])) {
            $where .= ' AND l.score_label = %s';
            $params[] = sanitize_text_field(wp_unslash($_GET['label']));
        }

        // Filter by status
        if (!empty($_GET['status'])) {
            $where .= ' AND l.status = %s';
            $params[] = sanitize_text_field(wp_unslash($_GET['status']));
        }

        // Search filter
        if (!empty($_GET['s'])) {
            $search = '%' . $wpdb->esc_like(sanitize_text_field(wp_unslash($_GET['s']))) . '%';
            $where .= ' AND (l.name LIKE %s OR l.email LIKE %s OR l.company LIKE %s)';
            $params = array_merge($params, [$search, $search, $search]);
        }

        // Get leads — LEFT JOIN enrichments when table exists
        if ($has_enrichments_table) {
            $query = "SELECT l.*,
                e.job_title AS enrich_job_title,
                e.company_name AS enrich_company_name,
                e.seniority_level AS enrich_seniority_level,
                e.linkedin_url AS enrich_linkedin_url,
                e.company_industry AS enrich_company_industry,
                e.company_size AS enrich_company_size,
                e.confidence AS enrich_confidence,
                e.enriched_at AS enrich_enriched_at
            FROM {$table} l
            LEFT JOIN {$enrichments_table} e ON e.lead_id = l.id
            WHERE {$where} ORDER BY l.created_at DESC";
        } else {
            $query = "SELECT * FROM {$table} l WHERE {$where} ORDER BY l.created_at DESC";
        }

        // phpcs:disable WordPress.DB.PreparedSQL.NotPrepared -- Table names are safely constructed from $wpdb->prefix.
        $leads = $wpdb->get_results(
            $params ? $wpdb->prepare($query, $params) : $query,
            // phpcs:enable WordPress.DB.PreparedSQL.NotPrepared
            ARRAY_A
        );

        // Build output array with enrichment sub-object per lead
        $leads_output = [];
        foreach ($leads as $lead) {
            // Determine enrichment object: null if un-enriched, object if enriched
            $has_enrichment = !empty($lead['enrich_enriched_at']);
            $enrichment = $has_enrichment ? [
                'job_title'        => $lead['enrich_job_title'] ?? null,
                'company_name'     => $lead['enrich_company_name'] ?? null,
                'seniority_level'  => $lead['enrich_seniority_level'] ?? null,
                'linkedin_url'     => $lead['enrich_linkedin_url'] ?? null,
                'company_industry' => $lead['enrich_company_industry'] ?? null,
                'company_size'     => $lead['enrich_company_size'] ?? null,
                'confidence'       => $lead['enrich_confidence'] ?? null,
                'enriched_at'      => $lead['enrich_enriched_at'] ?? null,
            ] : null;

            $leads_output[] = [
                'id'             => (int) $lead['id'],
                'name'           => $lead['name'] ?: null,
                'email'          => $lead['email'] ?: null,
                'company'        => $lead['company'] ?: null,
                'phone'          => $lead['phone'] ?: null,
                'score'          => $lead['score'] !== null ? (int) $lead['score'] : null,
                'score_label'    => $lead['score_label'] ?: null,
                'status'         => $lead['status'] ?: null,
                'scoring_method' => $lead['scoring_method'] ?: 'local',
                'score_reasoning' => $lead['score_reasoning'] ?: null,
                'form_plugin'    => $lead['form_plugin'] ?: null,
                'form_id'        => $lead['form_id'] ?: null,
                'source_url'     => $lead['source_url'] ?: null,
                'notes'          => $lead['notes'] ?: null,
                'created_at'     => $lead['created_at'] ?: null,
                'scored_at'      => $lead['scored_at'] ?: null,
                'enrichment'     => $enrichment,
            ];
        }

        echo wp_json_encode($leads_output, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON output for file download.
        exit;
    }

    /**
     * Handle single lead archive action
     */
    public function handle_archive_action(): void {
        if (!isset($_GET['page']) || sanitize_text_field(wp_unslash($_GET['page'])) !== 'formrank-lead-scoring') {
            return;
        }

        if (!isset($_GET['action']) || sanitize_text_field(wp_unslash($_GET['action'])) !== 'archive') {
            return;
        }

        $lead_id = isset($_GET['id']) ? absint(wp_unslash($_GET['id'])) : 0;

        if (!$lead_id) {
            return;
        }

        // Verify nonce
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'archive_lead_' . $lead_id)) {
            wp_die(esc_html__('Security check failed.', 'formrank-lead-scoring'));
        }

        // Check capabilities
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Permission denied.', 'formrank-lead-scoring'));
        }

        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        $wpdb->update(
            $table,
            [
                'status' => 'archived',
                'updated_at' => current_time('mysql')
            ],
            ['id' => $lead_id],
            ['%s', '%s'],
            ['%d']
        );

        // Redirect to remove action from URL
        wp_safe_redirect(admin_url('admin.php?page=formrank-lead-scoring&archived=1'));
        exit;
    }

    /**
     * Handle bulk actions on leads
     */
    public function handle_bulk_actions(): void {
        if (!isset($_GET['page']) || sanitize_text_field(wp_unslash($_GET['page'])) !== 'formrank-lead-scoring') {
            return;
        }

        // Check if bulk action is being performed
        $action = '';

        if (isset($_POST['action']) && sanitize_text_field(wp_unslash($_POST['action'])) !== '-1') {
            $action = sanitize_text_field(wp_unslash($_POST['action']));
        } elseif (isset($_POST['action2']) && sanitize_text_field(wp_unslash($_POST['action2'])) !== '-1') {
            $action = sanitize_text_field(wp_unslash($_POST['action2']));
        }

        if (empty($action) || !isset($_POST['lead_ids']) || !is_array($_POST['lead_ids'])) {
            return;
        }

        // Verify nonce (WP_List_Table uses _wpnonce with bulk-leads)
        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'bulk-leads')) {
            wp_die(esc_html__('Security check failed.', 'formrank-lead-scoring'));
        }

        // Check capabilities
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Permission denied.', 'formrank-lead-scoring'));
        }

        $lead_ids = array_map('absint', wp_unslash($_POST['lead_ids']));
        $lead_ids = array_filter($lead_ids);

        if (empty($lead_ids)) {
            return;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        $processed = 0;
        $redirect_args = ['page' => 'formrank-lead-scoring'];

        switch ($action) {
            case 'archive':
                foreach ($lead_ids as $lead_id) {
                    $result = $wpdb->update(
                        $table,
                        [
                            'status' => 'archived',
                            'updated_at' => current_time('mysql')
                        ],
                        ['id' => $lead_id],
                        ['%s', '%s'],
                        ['%d']
                    );
                    if ($result !== false) {
                        $processed++;
                    }
                }
                $redirect_args['bulk_archived'] = $processed;
                break;

            case 'delete':
                foreach ($lead_ids as $lead_id) {
                    $result = $wpdb->delete($table, ['id' => $lead_id], ['%d']);
                    if ($result !== false) {
                        $processed++;
                    }
                }
                $redirect_args['bulk_deleted'] = $processed;
                break;

            case 'rescore':
                $integration = new Integration_Base();
                foreach ($lead_ids as $lead_id) {
                    try {
                        $integration->score_lead($lead_id);
                        $processed++;
                    } catch (\Exception $e) {
                        // Log error but continue with other leads
                        error_log('FormRank: Failed to rescore lead ' . $lead_id . ': ' . $e->getMessage());
                    }
                }
                $redirect_args['bulk_rescored'] = $processed;
                break;

            case 'ai_rescore':
                $scoring_engine = new Scoring_Engine();
                $skipped = 0;
                foreach ($lead_ids as $lead_id) {
                    try {
                        // Check AI availability before each scoring
                        $availability = $scoring_engine->check_ai_availability();
                        if (!$availability['available']) {
                            $skipped += count($lead_ids) - $processed - $skipped;
                            break;
                        }
                        $scoring_engine->rescore_lead($lead_id);
                        $processed++;
                    } catch (\Exception $e) {
                        error_log('FormRank: Failed to AI rescore lead ' . $lead_id . ': ' . $e->getMessage());
                        $skipped++;
                    }
                }
                $redirect_args['bulk_ai_rescored'] = $processed;
                if ($skipped > 0) {
                    $redirect_args['bulk_ai_skipped'] = $skipped;
                }
                break;

            case 'mark_converted':
                foreach ($lead_ids as $lead_id) {
                    $result = $wpdb->update(
                        $table,
                        [
                            'status' => 'converted',
                            'updated_at' => current_time('mysql')
                        ],
                        ['id' => $lead_id],
                        ['%s', '%s'],
                        ['%d']
                    );
                    if ($result !== false) {
                        $processed++;
                    }
                }
                $redirect_args['bulk_converted'] = $processed;
                break;

            default:
                return;
        }

        // Redirect to show results
        wp_safe_redirect(add_query_arg($redirect_args, admin_url('admin.php')));
        exit;
    }

    /**
     * Render dashboard page
     */
    public function render_dashboard_page() {
        $dashboard = new Dashboard();
        $stats = $dashboard->get_stats();
        $demo_lead_count = Setup_Wizard::get_demo_lead_count();

        include FORMRANK_LS_PATH . 'templates/admin/dashboard.php';
    }

    /**
     * Render conversions analytics page
     */
    public function render_conversions_page() {
        $scoring_engine = new \FormRankLS\Core\Local_Scoring_Engine();
        $analytics = $scoring_engine->get_conversion_analytics();

        include FORMRANK_LS_PATH . 'templates/admin/conversions.php';
    }

    /**
     * Render settings page
     */
    public function render_settings_page() {
        include FORMRANK_LS_PATH . 'templates/admin/settings.php';
    }

    /**
     * Render lead detail page
     */
    public function render_lead_detail_page() {
        $lead_detail = new Lead_Detail();
        $lead = $lead_detail->get_lead();

        if (!$lead) {
            wp_die(esc_html__('Lead not found.', 'formrank-lead-scoring'));
        }

        include FORMRANK_LS_PATH . 'templates/admin/lead-detail.php';
    }

    /**
     * AJAX: Test API connection
     */
    public function ajax_test_api() {
        check_ajax_referer('formrank_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'formrank-lead-scoring')]);
        }

        $api_client = new API_Client();
        $result = $api_client->test_connection();

        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }

    /**
     * AJAX: AI Rescore a lead
     */
    public function ajax_rescore_lead() {
        check_ajax_referer('formrank_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'formrank-lead-scoring')]);
        }

        $lead_id = isset($_POST['lead_id']) ? absint(wp_unslash($_POST['lead_id'])) : 0;

        if (!$lead_id) {
            wp_send_json_error(['message' => __('Invalid lead ID.', 'formrank-lead-scoring')]);
        }

        try {
            $scoring_engine = new Scoring_Engine();

            // Check AI availability first
            $availability = $scoring_engine->check_ai_availability();
            if (!$availability['available']) {
                wp_send_json_error([
                    'message' => $availability['reason'],
                    'quota_exceeded' => true,
                    'usage' => $availability['usage'],
                    'limit' => $availability['limit']
                ]);
                return;
            }

            $result = $scoring_engine->rescore_lead($lead_id);
            $ai_stats = $scoring_engine->get_ai_stats();
            $actual_method = $result['scoring_method'] ?? 'ai';

            $response_data = [
                'score' => $result['score'],
                'label' => $result['label'],
                'reasoning' => $result['reasoning'],
                'scoring_method' => $actual_method,
                'ai_usage' => $ai_stats['usage'],
                'ai_limit' => $ai_stats['limit'],
                'ai_remaining' => $ai_stats['remaining'],
            ];

            if ($actual_method === 'ai') {
                $response_data['message'] = __('Lead scored with AI successfully!', 'formrank-lead-scoring');
            } else {
                $fallback_reason = $result['fallback_reason'] ?? '';
                $response_data['message'] = __('AI unavailable, scored with rules instead.', 'formrank-lead-scoring');
                if (!empty($fallback_reason)) {
                    $response_data['fallback_reason'] = $fallback_reason;
                }
            }

            wp_send_json_success($response_data);
        } catch (\Exception $e) {
            wp_send_json_error(['message' => $e->getMessage()]);
        }
    }

    /**
     * AJAX: Get AI scoring stats
     */
    public function ajax_get_ai_stats() {
        check_ajax_referer('formrank_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'formrank-lead-scoring')]);
        }

        $scoring_engine = new Scoring_Engine();
        $stats = $scoring_engine->get_ai_stats();
        $availability = $scoring_engine->check_ai_availability();

        wp_send_json_success([
            'stats' => $stats,
            'availability' => $availability
        ]);
    }

    /**
     * AJAX: Update lead status
     */
    public function ajax_update_status() {
        check_ajax_referer('formrank_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'formrank-lead-scoring')]);
        }

        $lead_id = isset($_POST['lead_id']) ? absint(wp_unslash($_POST['lead_id'])) : 0;
        $status = isset($_POST['status']) ? sanitize_text_field(wp_unslash($_POST['status'])) : '';

        if (!$lead_id || !$status) {
            wp_send_json_error(['message' => __('Invalid request.', 'formrank-lead-scoring')]);
        }

        $valid_statuses = ['new', 'contacted', 'qualified', 'converted', 'archived'];

        if (!in_array($status, $valid_statuses, true)) {
            wp_send_json_error(['message' => __('Invalid status.', 'formrank-lead-scoring')]);
        }

        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        $updated = $wpdb->update(
            $table,
            [
                'status' => $status,
                'updated_at' => current_time('mysql')
            ],
            ['id' => $lead_id]
        );

        if ($updated !== false) {
            wp_send_json_success(['message' => __('Status updated.', 'formrank-lead-scoring')]);
        } else {
            wp_send_json_error(['message' => __('Failed to update status.', 'formrank-lead-scoring')]);
        }
    }

    /**
     * AJAX: Get dashboard stats
     */
    public function ajax_get_stats() {
        check_ajax_referer('formrank_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'formrank-lead-scoring')]);
        }

        $dashboard = new Dashboard();
        $stats = $dashboard->get_stats();

        wp_send_json_success($stats);
    }

    /**
     * AJAX: Save lead notes
     */
    public function ajax_save_notes() {
        check_ajax_referer('formrank_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'formrank-lead-scoring')]);
        }

        $lead_id = isset($_POST['lead_id']) ? absint(wp_unslash($_POST['lead_id'])) : 0;
        $notes = isset($_POST['notes']) ? sanitize_textarea_field(wp_unslash($_POST['notes'])) : '';

        if (!$lead_id) {
            wp_send_json_error(['message' => __('Invalid lead ID.', 'formrank-lead-scoring')]);
        }

        global $wpdb;
        $table = $wpdb->prefix . 'formrank_leads';

        $updated = $wpdb->update(
            $table,
            [
                'notes' => $notes,
                'updated_at' => current_time('mysql')
            ],
            ['id' => $lead_id]
        );

        if ($updated !== false) {
            wp_send_json_success(['message' => __('Notes saved.', 'formrank-lead-scoring')]);
        } else {
            wp_send_json_error(['message' => __('Failed to save notes.', 'formrank-lead-scoring')]);
        }
    }

    /**
     * AJAX: Get conversion analytics
     */
    public function ajax_get_conversion_analytics() {
        check_ajax_referer('formrank_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'formrank-lead-scoring')]);
        }

        $scoring_engine = new \FormRankLS\Core\Local_Scoring_Engine();
        $analytics = $scoring_engine->get_conversion_analytics();

        wp_send_json_success($analytics);
    }

    /**
     * Show enrichment data-transmission notice after the first enrichment.
     *
     * Hooked to admin_notices. Appears only once on FormRank admin pages for
     * administrators, after a PRO enrichment has been performed but before the
     * user has dismissed it.
     */
    public function show_enrichment_notice(): void {
        // Guard: Only on FormRank admin pages.
        $screen = get_current_screen();
        if (!$screen || strpos($screen->id, 'formrank-lead-scoring') === false) {
            return;
        }

        // Guard: Only for site administrators.
        if (!current_user_can('manage_options')) {
            return;
        }

        // Guard: Only after the first enrichment has been performed.
        if (!get_option('formrank_enrichment_notice_pending')) {
            return;
        }

        // Guard: Only until the notice has been dismissed.
        if (get_option('formrank_enrichment_notice_dismissed')) {
            return;
        }

        ?>
        <div class="notice notice-info is-dismissible" id="formrank-enrichment-notice">
            <p>
                <strong><?php esc_html_e('Lead Enrichment is Active', 'formrank-lead-scoring'); ?></strong>
            </p>
            <p>
                <?php esc_html_e('When you enrich a lead, their name and email address are sent to Anthropic\'s AI API via a secure proxy for web-based person research. No data is stored by the AI provider beyond processing.', 'formrank-lead-scoring'); ?>
                <a href="<?php echo esc_url(admin_url('admin.php?page=formrank-lead-scoring-settings')); ?>">
                    <?php esc_html_e('Learn more', 'formrank-lead-scoring'); ?>
                </a>
            </p>
        </div>
        <?php
    }

    /**
     * AJAX: Persist enrichment notice dismissal.
     *
     * Hooked to wp_ajax_formrank_dismiss_enrichment_notice.
     */
    public function ajax_activate_license(): void {
        check_ajax_referer('formrank_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'formrank-lead-scoring')]);
            return;
        }

        $license_key = sanitize_text_field(wp_unslash($_POST['license_key'] ?? ''));
        if (empty($license_key)) {
            wp_send_json_error(['message' => __('Please enter a license key.', 'formrank-lead-scoring')]);
            return;
        }

        $fs = \FormRankLS\fr_fs();
        $result = $fs->activate_migrated_license($license_key);

        // Check for error responses from Freemius SDK
        if (is_object($result) && isset($result->error)) {
            $error_message = isset($result->error->message) ? $result->error->message : __('License activation failed.', 'formrank-lead-scoring');
            wp_send_json_error(['message' => $error_message]);
            return;
        }

        // Verify the license actually activated by checking Freemius state
        if (!$fs->is_paying()) {
            wp_send_json_error(['message' => __('License key was not recognized. Please check your key and try again.', 'formrank-lead-scoring')]);
            return;
        }

        // Clear cached usage stats so PRO limits apply immediately
        delete_transient('formrank_usage_stats');
        delete_transient('formrank_enrich_stats');

        $plan = \FormRankLS\formrank_get_plan();
        wp_send_json_success([
            'message' => sprintf(
                /* translators: %s: plan name (Pro/Agency) */
                __('License activated! You are now on the %s plan.', 'formrank-lead-scoring'),
                ucfirst($plan)
            ),
        ]);
    }

    public function ajax_deactivate_license(): void {
        check_ajax_referer('formrank_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'formrank-lead-scoring')]);
            return;
        }

        $fs = \FormRankLS\fr_fs();

        // Deactivate the license
        $fs->delete_account_event();

        // Clear cached data
        delete_transient('formrank_usage_stats');
        delete_transient('formrank_enrich_stats');
        delete_option('formrank_license_activated');

        wp_send_json_success(['message' => __('License deactivated.', 'formrank-lead-scoring')]);
    }

    public function ajax_dismiss_enrichment_notice(): void {
        check_ajax_referer('formrank_ajax_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'formrank-lead-scoring')]);
            return;
        }

        update_option('formrank_enrichment_notice_dismissed', 1, false); // autoload=false
        wp_send_json_success();
    }
}
