/**
 * FormRank Lead Scoring Admin JavaScript
 */

(function($) {
    'use strict';

    // Wait for document ready
    $(document).ready(function() {
        FormRankLS.init();
    });

    var FormRankLS = {

        init: function() {
            this.createConfirmModal();
            this.bindEvents();
            this.initShowMore();
            this.bindEnrichmentNotice();
            this.bindLicenseActions();
        },

        /**
         * Escape HTML entities for safe insertion
         */
        escHtml: function(str) {
            if (!str) return '';
            return $('<span>').text(String(str)).html();
        },

        /**
         * Escape attribute values
         */
        escAttr: function(str) {
            if (!str) return '';
            return String(str).replace(/&/g, '&amp;').replace(/"/g, '&quot;').replace(/'/g, '&#39;').replace(/</g, '&lt;').replace(/>/g, '&gt;');
        },

        /**
         * Create the confirm modal DOM element once
         */
        createConfirmModal: function() {
            if ($('#formrank-confirm-modal').length) return;

            var modal = '<div id="formrank-confirm-modal" class="formrank-modal-overlay">' +
                '<div class="formrank-modal">' +
                    '<div class="formrank-modal-icon"></div>' +
                    '<h3 class="formrank-modal-title"></h3>' +
                    '<p class="formrank-modal-message"></p>' +
                    '<div class="formrank-modal-actions">' +
                        '<button type="button" class="button formrank-modal-cancel"></button>' +
                        '<button type="button" class="button formrank-modal-confirm"></button>' +
                    '</div>' +
                '</div>' +
            '</div>';

            $('body').append(modal);
        },

        /**
         * Show a styled confirmation modal
         *
         * @param {Object} options - Modal configuration
         * @param {string} options.title - Modal title
         * @param {string} options.message - Modal message
         * @param {string} options.confirmText - Confirm button text (default: "Confirm")
         * @param {string} options.cancelText - Cancel button text (default: "Cancel")
         * @param {string} options.type - Modal type: "warning", "info", "danger" (default: "info")
         * @param {string} options.icon - Dashicon class (auto-set based on type if not provided)
         * @returns {Promise<boolean>} Resolves true on confirm, false on cancel
         */
        showConfirm: function(options) {
            var opts = $.extend({
                title: 'Confirm Action',
                message: 'Are you sure?',
                confirmText: formrankLS.strings.confirm_btn || 'Confirm',
                cancelText: formrankLS.strings.cancel_btn || 'Cancel',
                type: 'info',
                icon: ''
            }, options);

            // Auto-set icon based on type
            if (!opts.icon) {
                switch (opts.type) {
                    case 'warning': opts.icon = 'dashicons-warning'; break;
                    case 'danger': opts.icon = 'dashicons-trash'; break;
                    case 'ai': opts.icon = 'dashicons-cloud'; break;
                    default: opts.icon = 'dashicons-info-outline'; break;
                }
            }

            var $modal = $('#formrank-confirm-modal');
            var $content = $modal.find('.formrank-modal');

            // Set content
            $modal.find('.formrank-modal-icon')
                .html('<span class="dashicons ' + opts.icon + '"></span>')
                .attr('class', 'formrank-modal-icon formrank-modal-icon-' + opts.type);
            $modal.find('.formrank-modal-title').text(opts.title);
            $modal.find('.formrank-modal-message').html(opts.message);
            $modal.find('.formrank-modal-cancel').text(opts.cancelText);
            $modal.find('.formrank-modal-confirm')
                .text(opts.confirmText)
                .attr('class', 'button formrank-modal-confirm formrank-modal-confirm-' + opts.type);

            // Show with animation
            $modal.addClass('active');
            setTimeout(function() {
                $content.addClass('active');
            }, 10);

            return new Promise(function(resolve) {
                // Clean up previous handlers
                $modal.find('.formrank-modal-confirm').off('click').on('click', function() {
                    FormRankLS.hideConfirmModal();
                    resolve(true);
                });

                $modal.find('.formrank-modal-cancel').off('click').on('click', function() {
                    FormRankLS.hideConfirmModal();
                    resolve(false);
                });

                // Close on overlay click
                $modal.off('click.modal').on('click.modal', function(e) {
                    if ($(e.target).is($modal)) {
                        FormRankLS.hideConfirmModal();
                        resolve(false);
                    }
                });

                // Close on Escape
                $(document).off('keydown.formrank-modal').on('keydown.formrank-modal', function(e) {
                    if (e.key === 'Escape') {
                        FormRankLS.hideConfirmModal();
                        resolve(false);
                    }
                });
            });
        },

        /**
         * Hide the confirm modal
         */
        hideConfirmModal: function() {
            var $modal = $('#formrank-confirm-modal');
            $modal.find('.formrank-modal').removeClass('active');
            setTimeout(function() {
                $modal.removeClass('active');
            }, 200);
            $(document).off('keydown.formrank-modal');
        },

        bindEvents: function() {
            // Test API connection
            $('#formrank-test-api').on('click', this.testApiConnection);

            // Rescore lead (legacy)
            $(document).on('click', '.formrank-rescore', this.rescoreLead);

            // AI Rescore lead
            $(document).on('click', '.formrank-ai-rescore', this.aiRescoreLead);

            // Update status
            $('#formrank-status').on('change', this.updateStatus);

            // Save notes
            $('#formrank-save-notes').on('click', this.saveNotes);

            // Mark as converted
            $(document).on('click', '.formrank-mark-converted', this.markAsConverted);

            // Intercept bulk action form submit for confirmations
            $('#formrank-leads-form').on('submit', this.handleBulkActionSubmit);

            // Enrichment
            $(document).on('click', '.formrank-enrich-btn', this.enrichLead);
            $(document).on('click', '.formrank-copy-btn', this.copyToClipboard);
            $(document).on('click', '.formrank-show-more-btn', this.toggleShowMore);

            // Auto-enrich from list page link (?enrich=1)
            if (formrankLS.auto_enrich) {
                setTimeout(function() {
                    var $btn = $('.formrank-enrich-btn').first();
                    if ($btn.length && !$btn.prop('disabled')) {
                        $btn.trigger('click');
                    }
                }, 500);
            }

            // Remove demo data
            $('#formrank-remove-demo-data').on('click', function() {
                var $btn = $(this);
                if (!confirm(formrankLS.confirm_remove_demo || 'Remove all demo leads? This cannot be undone.')) {
                    return;
                }
                $btn.prop('disabled', true).text(formrankLS.removing || 'Removing...');
                $.ajax({
                    url: formrankLS.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'formrank_remove_demo_data',
                        nonce: formrankLS.nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            $('#formrank-demo-banner').slideUp(300, function() {
                                $(this).remove();
                            });
                            location.reload();
                        } else {
                            alert(response.data.message || 'Failed to remove demo data.');
                            $btn.prop('disabled', false).text(formrankLS.remove_demo || 'Remove Demo Data');
                        }
                    },
                    error: function() {
                        alert('Connection error.');
                        $btn.prop('disabled', false).text(formrankLS.remove_demo || 'Remove Demo Data');
                    }
                });
            });

            // List page: AJAX enrich without navigating away
            $(document).on('click', '.formrank-list-enrich-btn', this.listEnrichLead);
        },

        /**
         * Test API Connection
         */
        testApiConnection: function(e) {
            e.preventDefault();

            var $button = $(this);
            var $status = $('#formrank-api-status');
            var $apiKey = $('#formrank_api_key');

            $button.prop('disabled', true);
            $status.removeClass('success error').addClass('loading').text(formrankLS.strings.rescoring);

            $.ajax({
                url: formrankLS.ajax_url,
                type: 'POST',
                data: {
                    action: 'formrank_test_api',
                    nonce: formrankLS.nonce,
                    api_key: $apiKey.val()
                },
                success: function(response) {
                    $button.prop('disabled', false);

                    if (response.success) {
                        $status.removeClass('loading error').addClass('success').text(response.data.message);
                    } else {
                        $status.removeClass('loading success').addClass('error').text(response.data.message);
                    }
                },
                error: function() {
                    $button.prop('disabled', false);
                    $status.removeClass('loading success').addClass('error').text(formrankLS.strings.error);
                }
            });
        },

        /**
         * Rescore a Lead (Legacy - local scoring)
         */
        rescoreLead: function(e) {
            e.preventDefault();

            var $link = $(this);
            var leadId = $link.data('id');
            var originalText = $link.text();

            FormRankLS.showConfirm({
                title: formrankLS.strings.rescore_title || 'Rescore Lead',
                message: formrankLS.strings.confirm_rescore,
                confirmText: formrankLS.strings.rescore_confirm_btn || 'Rescore',
                type: 'info'
            }).then(function(confirmed) {
                if (!confirmed) return;

                $link.text(formrankLS.strings.rescoring).addClass('disabled');

                $.ajax({
                    url: formrankLS.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'formrank_rescore_lead',
                        nonce: formrankLS.nonce,
                        lead_id: leadId
                    },
                    success: function(response) {
                        if (response.success) {
                            location.reload();
                        } else {
                            FormRankLS.showNotice(response.data.message || formrankLS.strings.error, 'error');
                            $link.text(originalText).removeClass('disabled');
                        }
                    },
                    error: function() {
                        FormRankLS.showNotice(formrankLS.strings.error, 'error');
                        $link.text(originalText).removeClass('disabled');
                    }
                });
            });
        },

        /**
         * AI Rescore a Lead (uses Claude API)
         */
        aiRescoreLead: function(e) {
            e.preventDefault();

            var $button = $(this);
            var leadId = $button.data('id');
            var originalHtml = $button.html();

            FormRankLS.showConfirm({
                title: formrankLS.strings.ai_rescore_title || 'Score with AI',
                message: formrankLS.strings.confirm_ai_rescore || 'Score this lead with AI? This will use 1 of your monthly AI rescores.',
                confirmText: formrankLS.strings.ai_rescore_confirm_btn || 'Score with AI',
                type: 'ai'
            }).then(function(confirmed) {
                if (!confirmed) return;

                // Show loading state
                $button.addClass('loading').prop('disabled', true);
                $button.find('.dashicons').removeClass('dashicons-cloud').addClass('dashicons-update');

                $.ajax({
                    url: formrankLS.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'formrank_rescore_lead',
                        nonce: formrankLS.nonce,
                        lead_id: leadId
                    },
                    success: function(response) {
                        if (response.success) {
                            var method = response.data.scoring_method || 'ai';

                            if (method === 'ai') {
                                // AI scoring succeeded
                                $button.removeClass('loading').html('<span class="dashicons dashicons-yes-alt"></span> ' + (formrankLS.strings.ai_success || 'AI Scored!'));
                                FormRankLS.updateScoringMethodBadge('ai');
                            } else {
                                // Fell back to local scoring - show reason if available
                                $button.removeClass('loading').html('<span class="dashicons dashicons-warning"></span> ' + (response.data.message || 'Scored with rules'));
                                var noticeMsg = response.data.message || 'AI was unavailable. Lead was scored with rules instead.';
                                if (response.data.fallback_reason) {
                                    noticeMsg += ' Reason: ' + response.data.fallback_reason;
                                }
                                FormRankLS.showNotice(noticeMsg, 'warning');
                            }

                            // Update usage display if present
                            if (response.data.ai_remaining !== undefined) {
                                $('.formrank-ai-usage-text').text(
                                    response.data.ai_usage + ' / ' + response.data.ai_limit + ' AI rescores used this month'
                                );
                                var pct = (response.data.ai_usage / response.data.ai_limit) * 100;
                                $('.formrank-ai-usage-fill').css('width', pct + '%');
                            }

                            // Refresh after short delay to show updated score
                            setTimeout(function() {
                                location.reload();
                            }, 1200);
                        } else {
                            // Handle quota exceeded
                            if (response.data.quota_exceeded) {
                                FormRankLS.showConfirm({
                                    title: formrankLS.strings.ai_limit_title || 'AI Limit Reached',
                                    message: formrankLS.strings.ai_limit_reached || 'AI rescore limit reached. Upgrade to Pro for unlimited AI scoring.',
                                    confirmText: formrankLS.strings.upgrade_btn || 'Upgrade to Pro',
                                    cancelText: formrankLS.strings.close_btn || 'Close',
                                    type: 'warning'
                                }).then(function(upgrade) {
                                    if (upgrade && formrankLS.upgrade_url) {
                                        window.location.href = formrankLS.upgrade_url;
                                    }
                                });
                            } else {
                                FormRankLS.showNotice(response.data.message || formrankLS.strings.error, 'error');
                            }
                            $button.removeClass('loading').prop('disabled', false).html(originalHtml);
                        }
                    },
                    error: function(xhr) {
                        var errorMsg = formrankLS.strings.error || 'An error occurred';

                        try {
                            var response = JSON.parse(xhr.responseText);
                            if (response.data && response.data.message) {
                                errorMsg = response.data.message;
                            }
                        } catch(err) {}

                        FormRankLS.showNotice(errorMsg, 'error');
                        $button.removeClass('loading').prop('disabled', false).html(originalHtml);
                    }
                });
            });
        },

        /**
         * Update the scoring method badge on the page after AI scoring
         */
        updateScoringMethodBadge: function(method) {
            var $badgeContainer = $('.formrank-scoring-method');
            if (!$badgeContainer.length) return;

            if (method === 'ai') {
                $badgeContainer.html(
                    '<span class="formrank-method-badge formrank-method-ai" title="' + (formrankLS.strings.ai_scored_tooltip || 'Scored using Claude AI') + '">' +
                        '<span class="dashicons dashicons-cloud"></span> ' +
                        (formrankLS.strings.ai_scored_label || 'AI Scored') +
                    '</span>'
                );
            }

            // Also update list-page badges if visible
            $('.formrank-list-method-badge.local').each(function() {
                var $badge = $(this);
                $badge.removeClass('local').addClass('ai')
                    .attr('title', formrankLS.strings.ai_scored_tooltip || 'AI Scored')
                    .html('<span class="dashicons dashicons-cloud"></span>');
            });
        },

        /**
         * Show a temporary notice at the top of the page
         */
        showNotice: function(message, type) {
            type = type || 'info';
            var cssClass = type === 'error' ? 'notice-error' : (type === 'success' ? 'notice-success' : 'notice-info');
            var $notice = $('<div class="notice ' + cssClass + ' is-dismissible"><p>' + $('<span>').text(message).html() + '</p></div>');
            $('.wrap h1').first().after($notice);
            setTimeout(function() { $notice.fadeOut(function() { $notice.remove(); }); }, 5000);
        },

        /**
         * Handle bulk action form submission with confirmation
         */
        handleBulkActionSubmit: function(e) {
            var action = '';
            var $form = $(this);

            // Get selected action from either dropdown
            var action1 = $form.find('select[name="action"]').val();
            var action2 = $form.find('select[name="action2"]').val();
            action = (action1 && action1 !== '-1') ? action1 : action2;

            if (!action || action === '-1') return true;

            // Check if any items selected
            var $checked = $form.find('input[name="lead_ids[]"]:checked');
            if (!$checked.length) return true;

            var count = $checked.length;
            var config = FormRankLS.getBulkActionConfig(action, count);

            if (!config) return true;

            e.preventDefault();

            FormRankLS.showConfirm(config).then(function(confirmed) {
                if (confirmed) {
                    // Bypass the handler on resubmit
                    $form.off('submit', FormRankLS.handleBulkActionSubmit);
                    $form.submit();
                }
            });
        },

        /**
         * Get confirm modal config for bulk actions
         */
        getBulkActionConfig: function(action, count) {
            var configs = {
                'delete': {
                    title: formrankLS.strings.bulk_delete_title || 'Delete Leads',
                    message: (formrankLS.strings.bulk_delete_msg || 'Permanently delete <strong>%d</strong> selected lead(s)? This cannot be undone.').replace('%d', count),
                    confirmText: formrankLS.strings.bulk_delete_btn || 'Delete Permanently',
                    type: 'danger'
                },
                'archive': {
                    title: formrankLS.strings.bulk_archive_title || 'Archive Leads',
                    message: (formrankLS.strings.bulk_archive_msg || 'Archive <strong>%d</strong> selected lead(s)?').replace('%d', count),
                    confirmText: formrankLS.strings.bulk_archive_btn || 'Archive',
                    type: 'info'
                },
                'mark_converted': {
                    title: formrankLS.strings.bulk_convert_title || 'Mark as Converted',
                    message: (formrankLS.strings.bulk_convert_msg || 'Mark <strong>%d</strong> selected lead(s) as converted? This helps improve AI scoring accuracy.').replace('%d', count),
                    confirmText: formrankLS.strings.bulk_convert_btn || 'Mark Converted',
                    type: 'info'
                },
                'rescore': {
                    title: formrankLS.strings.bulk_rescore_title || 'Rescore Leads',
                    message: (formrankLS.strings.bulk_rescore_msg || 'Rescore <strong>%d</strong> selected lead(s) using rule-based engine?').replace('%d', count),
                    confirmText: formrankLS.strings.bulk_rescore_btn || 'Rescore',
                    type: 'info'
                },
                'ai_rescore': {
                    title: formrankLS.strings.bulk_ai_title || 'Score with AI',
                    message: (formrankLS.strings.bulk_ai_msg || 'Score <strong>%d</strong> selected lead(s) with AI? This will use your monthly AI rescore quota.').replace('%d', count),
                    confirmText: formrankLS.strings.bulk_ai_btn || 'Score with AI',
                    type: 'ai'
                }
            };

            return configs[action] || null;
        },

        /**
         * Update Lead Status
         */
        updateStatus: function() {
            var $select = $(this);
            var leadId = $select.data('id');
            var status = $select.val();

            $select.prop('disabled', true);

            $.ajax({
                url: formrankLS.ajax_url,
                type: 'POST',
                data: {
                    action: 'formrank_update_status',
                    nonce: formrankLS.nonce,
                    lead_id: leadId,
                    status: status
                },
                success: function(response) {
                    $select.prop('disabled', false);

                    if (!response.success) {
                        FormRankLS.showNotice(response.data.message || formrankLS.strings.error, 'error');
                    }
                },
                error: function() {
                    $select.prop('disabled', false);
                    FormRankLS.showNotice(formrankLS.strings.error, 'error');
                }
            });
        },

        /**
         * Save Notes
         */
        saveNotes: function(e) {
            e.preventDefault();

            var $button = $(this);
            var $textarea = $('#formrank-notes');
            var leadId = $textarea.data('id');
            var notes = $textarea.val();
            var originalText = $button.text();

            $button.prop('disabled', true).text('Saving...');

            $.ajax({
                url: formrankLS.ajax_url,
                type: 'POST',
                data: {
                    action: 'formrank_save_notes',
                    nonce: formrankLS.nonce,
                    lead_id: leadId,
                    notes: notes
                },
                success: function(response) {
                    $button.prop('disabled', false);

                    if (response.success) {
                        $button.text('Saved!');
                        setTimeout(function() {
                            $button.text(originalText);
                        }, 2000);
                    } else {
                        $button.text(originalText);
                        FormRankLS.showNotice(response.data.message || formrankLS.strings.error, 'error');
                    }
                },
                error: function() {
                    $button.prop('disabled', false).text(originalText);
                    FormRankLS.showNotice(formrankLS.strings.error, 'error');
                }
            });
        },

        /**
         * Enrich a lead from the detail page
         */
        enrichLead: function(e) {
            e.preventDefault();
            var $button = $(this);
            var leadId = $button.data('id') || formrankLS.lead_id;
            var originalHtml = $button.html();
            var isReenrich = $button.hasClass('button-small'); // Re-enrich is button-small

            if ($button.prop('disabled') || $button.hasClass('loading')) return;

            // Confirmation before enrich
            FormRankLS.showConfirm({
                title: formrankLS.strings.enrich_confirm_title,
                message: formrankLS.strings.enrich_confirm_msg,
                confirmText: formrankLS.strings.enrich_btn,
                type: 'ai'
            }).then(function(confirmed) {
                if (!confirmed) return;

                // Loading state: spinner replaces button text
                $button.addClass('loading').prop('disabled', true)
                    .html('<span class="dashicons dashicons-update formrank-spin"></span> ' + formrankLS.strings.enriching);

                $.ajax({
                    url: formrankLS.rest_url + 'formrank/v1/enrich',
                    method: 'POST',
                    headers: { 'X-WP-Nonce': formrankLS.rest_nonce },
                    contentType: 'application/json',
                    data: JSON.stringify({ lead_id: parseInt(leadId, 10) }),
                    timeout: 120000, // 2 min timeout (Worker + Claude web search)
                    success: function(response) {
                        if (response.success) {
                            FormRankLS.handleEnrichSuccess(response, $button);
                        } else {
                            FormRankLS.handleEnrichError(response, $button, originalHtml);
                        }
                    },
                    error: function(xhr) {
                        var errorData = {};
                        try { errorData = JSON.parse(xhr.responseText); } catch(err) {}
                        FormRankLS.handleEnrichError(errorData, $button, originalHtml);
                    }
                });
            });
        },

        /**
         * AJAX enrich from list page without navigation
         */
        listEnrichLead: function(e) {
            e.preventDefault();
            var $link = $(this);
            var leadId = $link.data('lead-id');
            var originalText = $link.text();
            var $row = $link.closest('tr');
            var $dot = $row.find('.formrank-enrichment-dot');

            if ($link.hasClass('loading')) return;

            // Confirmation before enrich
            FormRankLS.showConfirm({
                title: formrankLS.strings.enrich_confirm_title,
                message: formrankLS.strings.enrich_confirm_msg,
                confirmText: formrankLS.strings.enrich_btn,
                type: 'ai'
            }).then(function(confirmed) {
                if (!confirmed) return;

                // Loading state
                $link.addClass('loading').text(formrankLS.strings.enriching);

                $.ajax({
                    url: formrankLS.rest_url + 'formrank/v1/enrich',
                    method: 'POST',
                    headers: { 'X-WP-Nonce': formrankLS.rest_nonce },
                    contentType: 'application/json',
                    data: JSON.stringify({ lead_id: parseInt(leadId, 10) }),
                    timeout: 120000,
                    success: function(response) {
                        if (response.success) {
                            // Update dot to green
                            $dot.css('background', '#22c55e').attr('title', formrankLS.strings.enriched || 'Enriched');
                            // Update link text to "Re-enrich"
                            $link.removeClass('loading').text(formrankLS.strings.re_enrich || 'Re-enrich');
                            // Update list usage counter if present
                            FormRankLS.updateEnrichUsage(response.usage || 0, response.limit || 0, response.remaining || 0);
                            FormRankLS.showNotice(formrankLS.strings.enrich_success, 'success');
                        } else {
                            $link.removeClass('loading').text(originalText);
                            FormRankLS.showNotice(response.message || formrankLS.strings.enrich_error, 'error');
                        }
                    },
                    error: function(xhr) {
                        var errorData = {};
                        try { errorData = JSON.parse(xhr.responseText); } catch(err) {}
                        var code = errorData.code || '';
                        if (code === 'quota_exceeded') {
                            $link.removeClass('loading').text(formrankLS.strings.enrich_no_quota).css('color', '#9ca3af');
                            FormRankLS.showNotice(errorData.message || formrankLS.strings.enrich_no_quota, 'warning');
                        } else if (code === 'upgrade_required') {
                            $link.removeClass('loading').text(originalText);
                            FormRankLS.showConfirm({
                                title: formrankLS.strings.enrich_upgrade_title,
                                message: formrankLS.strings.enrich_upgrade_msg,
                                confirmText: formrankLS.strings.upgrade_btn,
                                cancelText: formrankLS.strings.close_btn,
                                type: 'warning'
                            }).then(function(upgrade) {
                                if (upgrade && formrankLS.upgrade_url) {
                                    window.location.href = formrankLS.upgrade_url;
                                }
                            });
                        } else {
                            $link.removeClass('loading').text(originalText);
                            FormRankLS.showNotice(errorData.message || formrankLS.strings.enrich_error, 'error');
                        }
                    }
                });
            });
        },

        /**
         * Handle successful enrichment response
         */
        handleEnrichSuccess: function(response, $button) {
            var data = response.data || response;
            var person = data.person || {};
            var company = data.company || {};
            var confidence = data.confidence || 'not_found';
            var enrichedAt = response.enriched_at || '';
            var usage = response.usage || 0;
            var limit = response.limit || 0;
            var remaining = response.remaining || 0;

            // Build enrichment card HTML from response (mirrors PHP partial structure)
            var cardHtml = FormRankLS.buildEnrichmentCard(person, company, confidence, enrichedAt);

            // Replace the enrichment section content with the new card (fade-in)
            var $section = $('#formrank-enrichment-section');
            if ($section.length) {
                $section.css('opacity', 0).html(cardHtml).animate({ opacity: 1 }, 400);
            }

            // Update usage counter
            FormRankLS.updateEnrichUsage(usage, limit, remaining);

            // Show success notice
            FormRankLS.showNotice(formrankLS.strings.enrich_success, 'success');

            // Initialize show-more buttons for truncated descriptions
            setTimeout(function() { FormRankLS.initShowMore(); }, 100);
        },

        /**
         * Build enrichment card HTML from JSON response (mirrors PHP partial structure)
         */
        buildEnrichmentCard: function(person, company, confidence, enrichedAt) {
            var colors = { high: '#22c55e', medium: '#eab308', low: '#f97316', not_found: '#9ca3af' };
            var labels = { high: 'High Confidence', medium: 'Medium Confidence', low: 'Low Confidence', not_found: 'Not Found' };
            var textColors = { medium: '#000' };
            var badgeTextColor = textColors[confidence] || '#fff';

            var html = '';

            // Header with confidence badge
            html += '<h2>' + FormRankLS.escHtml('Lead Intelligence');
            html += ' <span class="formrank-confidence-badge" style="background:' + (colors[confidence] || colors.not_found) + ';color:' + badgeTextColor + ';padding:2px 10px;border-radius:12px;font-size:12px;font-weight:500;margin-left:8px;vertical-align:middle;">' + FormRankLS.escHtml(labels[confidence] || 'Unknown') + '</span>';
            html += '</h2>';

            // Timestamp
            if (enrichedAt) {
                html += '<p class="formrank-enrichment-timestamp" style="color:#6b7280;font-size:13px;margin-top:4px;">Enriched just now</p>';
            }

            // Usage counter placeholder
            html += '<div id="formrank-enrich-usage" class="formrank-enrich-usage"></div>';

            // Person section
            var personFields = [
                { key: 'full_name', label: 'Name', link: false },
                { key: 'job_title', label: 'Title', link: false },
                { key: 'company', label: 'Company', link: false },
                { key: 'seniority_level', label: 'Seniority', link: false },
                { key: 'linkedin_url', label: 'LinkedIn', link: true },
                { key: 'twitter_url', label: 'Twitter', link: true },
                { key: 'github_url', label: 'GitHub', link: true },
                { key: 'personal_site', label: 'Website', link: true },
                { key: 'other_profiles', label: 'Other', link: false }
            ];

            var hasPersonData = false;
            var personRowsHtml = '';
            for (var i = 0; i < personFields.length; i++) {
                var f = personFields[i];
                var val = person[f.key];
                if (!val || (Array.isArray(val) && val.length === 0)) continue;
                hasPersonData = true;
                var displayVal = Array.isArray(val) ? val.join(', ') : val;
                var cellHtml = '';
                if (f.link && typeof displayVal === 'string' && displayVal.indexOf('http') === 0) {
                    cellHtml = '<a href="' + FormRankLS.escAttr(displayVal) + '" target="_blank" rel="noopener">' + FormRankLS.escHtml(displayVal) + '</a>';
                } else {
                    cellHtml = FormRankLS.escHtml(displayVal);
                }
                personRowsHtml += '<tr><th>' + FormRankLS.escHtml(f.label) + '</th><td><span class="formrank-enrichment-field-value" data-copy-value="' + FormRankLS.escAttr(displayVal) + '">' + cellHtml + '<span class="formrank-copy-btn dashicons dashicons-clipboard" title="Copy"></span></span></td></tr>';
            }

            html += '<h3>Person</h3>';
            if (hasPersonData) {
                html += '<table class="formrank-info-table formrank-enrichment-fields">' + personRowsHtml + '</table>';
            } else {
                html += '<p class="formrank-section-empty" style="color:#6b7280;font-style:italic;">' + FormRankLS.escHtml(formrankLS.strings.no_person_data || 'No person data found') + '</p>';
            }

            // Company section
            if (company && typeof company === 'object') {
                var companyFields = [
                    { key: 'name', label: 'Name', link: false },
                    { key: 'industry', label: 'Industry', link: false },
                    { key: 'size', label: 'Size', link: false },
                    { key: 'website', label: 'Website', link: true },
                    { key: 'location', label: 'Location', link: false },
                    { key: 'description', label: 'Description', link: false }
                ];

                var hasCompanyData = false;
                var companyRowsHtml = '';
                for (var j = 0; j < companyFields.length; j++) {
                    var cf = companyFields[j];
                    var cval = company[cf.key];
                    if (!cval) continue;
                    hasCompanyData = true;
                    var cCellHtml = '';
                    if (cf.key === 'description') {
                        cCellHtml = '<div class="formrank-company-description"><p>' + FormRankLS.escHtml(cval) + '</p></div><button type="button" class="formrank-show-more-btn button-link" style="display:none;">' + (formrankLS.strings.show_more || 'Show more') + '</button>';
                        companyRowsHtml += '<tr><th>' + FormRankLS.escHtml(cf.label) + '</th><td>' + cCellHtml + '</td></tr>';
                    } else if (cf.link && typeof cval === 'string' && cval.indexOf('http') === 0) {
                        cCellHtml = '<a href="' + FormRankLS.escAttr(cval) + '" target="_blank" rel="noopener">' + FormRankLS.escHtml(cval) + '</a>';
                        companyRowsHtml += '<tr><th>' + FormRankLS.escHtml(cf.label) + '</th><td><span class="formrank-enrichment-field-value" data-copy-value="' + FormRankLS.escAttr(cval) + '">' + cCellHtml + '<span class="formrank-copy-btn dashicons dashicons-clipboard" title="Copy"></span></span></td></tr>';
                    } else {
                        companyRowsHtml += '<tr><th>' + FormRankLS.escHtml(cf.label) + '</th><td><span class="formrank-enrichment-field-value" data-copy-value="' + FormRankLS.escAttr(cval) + '">' + FormRankLS.escHtml(cval) + '<span class="formrank-copy-btn dashicons dashicons-clipboard" title="Copy"></span></span></td></tr>';
                    }
                }

                html += '<h3>Company</h3>';
                if (hasCompanyData) {
                    html += '<table class="formrank-info-table formrank-enrichment-fields">' + companyRowsHtml + '</table>';
                } else {
                    html += '<p class="formrank-section-empty" style="color:#6b7280;font-style:italic;">' + FormRankLS.escHtml(formrankLS.strings.no_company_data || 'No company data found') + '</p>';
                }
            } else {
                html += '<h3>Company</h3><p class="formrank-section-empty" style="color:#6b7280;font-style:italic;">' + FormRankLS.escHtml(formrankLS.strings.no_company_data || 'No company data found') + '</p>';
            }

            // Re-enrich button
            html += '<div class="formrank-enrich-actions" style="margin-top:16px;"><button type="button" class="button button-small formrank-enrich-btn" data-id="' + FormRankLS.escAttr(formrankLS.lead_id || '') + '">' + FormRankLS.escHtml(formrankLS.strings.re_enrich || 'Re-enrich') + '</button></div>';

            // Disclaimer
            html += '<p class="formrank-enrichment-disclaimer" style="color:#9ca3af;font-size:12px;margin-top:8px;font-style:italic;">' + FormRankLS.escHtml(formrankLS.strings.disclaimer) + '</p>';

            return html;
        },

        /**
         * Handle enrichment error with contextual messages per error code
         */
        handleEnrichError: function(errorData, $button, originalHtml) {
            var code = errorData.code || '';
            var message = errorData.message || formrankLS.strings.enrich_error;

            switch (code) {
                case 'upgrade_required':
                    // Modal with upgrade link
                    FormRankLS.showConfirm({
                        title: formrankLS.strings.enrich_upgrade_title,
                        message: formrankLS.strings.enrich_upgrade_msg,
                        confirmText: formrankLS.strings.upgrade_btn,
                        cancelText: formrankLS.strings.close_btn,
                        type: 'warning'
                    }).then(function(upgrade) {
                        if (upgrade && formrankLS.upgrade_url) {
                            window.location.href = formrankLS.upgrade_url;
                        }
                    });
                    $button.removeClass('loading').prop('disabled', false).html(originalHtml);
                    break;

                case 'quota_exceeded':
                    // Disable button with "No enrichments remaining"
                    $button.removeClass('loading').prop('disabled', true)
                        .html(formrankLS.strings.enrich_no_quota);
                    FormRankLS.showNotice(message, 'warning');
                    break;

                case 'cooldown_active':
                    // Inline notice, re-enable after 30s
                    FormRankLS.showNotice(formrankLS.strings.enrich_cooldown, 'info');
                    $button.removeClass('loading').prop('disabled', true).html(originalHtml);
                    setTimeout(function() {
                        $button.prop('disabled', false);
                    }, 30000);
                    break;

                default:
                    // Generic error or timeout
                    if (errorData.status === 0 || !errorData.code) {
                        // Connection timeout — enrichment may still succeed server-side
                        FormRankLS.showNotice(formrankLS.strings.enrich_timeout, 'warning');
                    } else {
                        FormRankLS.showNotice(message, 'error');
                    }
                    $button.removeClass('loading').prop('disabled', false).html(originalHtml);
                    break;
            }
        },

        /**
         * Copy field value to clipboard
         */
        copyToClipboard: function(e) {
            e.preventDefault();
            e.stopPropagation();
            var $icon = $(this);
            var text = $icon.closest('.formrank-enrichment-field-value').data('copy-value') || '';

            if (!text) return;

            if (navigator.clipboard && window.isSecureContext) {
                navigator.clipboard.writeText(text).then(function() {
                    FormRankLS.showCopyFeedback($icon);
                });
            } else {
                // Legacy fallback
                var $tmp = $('<textarea>').val(text).appendTo('body');
                $tmp[0].select();
                document.execCommand('copy');
                $tmp.remove();
                FormRankLS.showCopyFeedback($icon);
            }
        },

        /**
         * Show brief visual feedback after copy
         */
        showCopyFeedback: function($icon) {
            var originalClass = $icon.attr('class');
            $icon.removeClass('dashicons-clipboard').addClass('dashicons-yes').css('color', '#22c55e');
            setTimeout(function() {
                $icon.attr('class', originalClass).css('color', '');
            }, 1500);
        },

        /**
         * Toggle company description show more / show less
         */
        toggleShowMore: function(e) {
            e.preventDefault();
            var $btn = $(this);
            var $desc = $btn.prev('.formrank-company-description');

            if ($desc.hasClass('expanded')) {
                $desc.removeClass('expanded');
                $btn.text(formrankLS.strings.show_more || 'Show more');
            } else {
                $desc.addClass('expanded');
                $btn.text(formrankLS.strings.show_less || 'Show less');
            }
        },

        /**
         * Initialize show-more buttons — show only when description is clamped
         */
        initShowMore: function() {
            $('.formrank-company-description').each(function() {
                var $desc = $(this);
                var $btn = $desc.next('.formrank-show-more-btn');
                // Show "Show more" only if content is actually clamped
                if (this.scrollHeight > this.clientHeight + 2) {
                    $btn.show();
                } else {
                    $btn.hide();
                }
            });
        },

        /**
         * Update the enrichment usage counter in place
         */
        updateEnrichUsage: function(usage, limit, remaining) {
            var text = usage + ' of ' + limit + ' enrichments used this month';
            var html = '<span class="dashicons dashicons-search" style="font-size:16px;width:16px;height:16px;vertical-align:text-bottom;margin-right:4px;"></span>' + FormRankLS.escHtml(text);
            $('#formrank-enrich-usage').html(html).css({
                margin: '8px 0', padding: '8px 12px', background: '#f0f0f1',
                borderRadius: '4px', fontSize: '13px', color: '#50575e', display: 'inline-block'
            });
        },

        /**
         * Bind dismiss handler for the enrichment data-disclosure notice.
         *
         * Posts to wp_ajax_formrank_dismiss_enrichment_notice when WordPress
         * core fires the dismiss click on #formrank-enrichment-notice.
         */
        bindEnrichmentNotice: function() {
            $(document).on('click', '#formrank-enrichment-notice .notice-dismiss', function() {
                $.post(formrankLS.ajax_url, {
                    action: 'formrank_dismiss_enrichment_notice',
                    nonce: formrankLS.nonce
                });
            });
        },

        bindLicenseActions: function() {
            var $msg = $('#formrank-license-message');

            $('#formrank-activate-license').on('click', function() {
                var $btn = $(this);
                var key = $('#formrank-license-key').val().trim();
                if (!key) {
                    $msg.text('Please enter a license key.').css('color', '#dc2626').show();
                    return;
                }
                $btn.prop('disabled', true).text('Activating...');
                $msg.hide();

                $.post(formrankLS.ajax_url, {
                    action: 'formrank_activate_license',
                    license_key: key,
                    nonce: formrankLS.nonce
                }, function(response) {
                    if (response.success) {
                        $msg.text(response.data.message).css('color', '#059669').show();
                        setTimeout(function() { location.reload(); }, 1500);
                    } else {
                        $msg.text(response.data.message || 'Activation failed.').css('color', '#dc2626').show();
                        $btn.prop('disabled', false).text('Activate License');
                    }
                }).fail(function() {
                    $msg.text('Network error. Please try again.').css('color', '#dc2626').show();
                    $btn.prop('disabled', false).text('Activate License');
                });
            });

            $('#formrank-deactivate-license').on('click', function() {
                if (!confirm('Are you sure you want to deactivate your license?')) return;
                var $btn = $(this);
                $btn.prop('disabled', true).text('Deactivating...');

                $.post(formrankLS.ajax_url, {
                    action: 'formrank_deactivate_license',
                    nonce: formrankLS.nonce
                }, function(response) {
                    if (response.success) {
                        setTimeout(function() { location.reload(); }, 1000);
                    } else {
                        $btn.prop('disabled', false).text('Deactivate License');
                    }
                });
            });
        },

        /**
         * Mark Lead as Converted
         */
        markAsConverted: function(e) {
            e.preventDefault();

            var $button = $(this);
            var leadId = $button.data('id');
            var originalHtml = $button.html();

            FormRankLS.showConfirm({
                title: formrankLS.strings.convert_title || 'Mark as Converted',
                message: formrankLS.strings.confirm_converted || 'Mark this lead as converted? This helps improve AI scoring accuracy.',
                confirmText: formrankLS.strings.convert_confirm_btn || 'Mark Converted',
                type: 'info',
                icon: 'dashicons-yes-alt'
            }).then(function(confirmed) {
                if (!confirmed) return;

                $button.prop('disabled', true).html('<span class="dashicons dashicons-update spin"></span> Updating...');

                $.ajax({
                    url: formrankLS.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'formrank_update_status',
                        nonce: formrankLS.nonce,
                        lead_id: leadId,
                        status: 'converted'
                    },
                    success: function(response) {
                        if (response.success) {
                            $button.html('<span class="dashicons dashicons-yes-alt"></span> Converted!');
                            setTimeout(function() {
                                location.reload();
                            }, 800);
                        } else {
                            FormRankLS.showNotice(response.data.message || formrankLS.strings.error, 'error');
                            $button.prop('disabled', false).html(originalHtml);
                        }
                    },
                    error: function() {
                        FormRankLS.showNotice(formrankLS.strings.error, 'error');
                        $button.prop('disabled', false).html(originalHtml);
                    }
                });
            });
        }
    };

    // Expose to window for potential external use
    window.FormRankLS = FormRankLS;

    /**
     * Conversion Tracking Module
     */
    var FormRankConversions = {

        /**
         * Initialize conversion tracking UI
         */
        init: function() {
            this.bindEvents();
            this.initProgressAnimations();
        },

        /**
         * Bind event handlers
         */
        bindEvents: function() {
            // Refresh analytics
            $(document).on('click', '.formrank-refresh-analytics', this.refreshAnalytics);

            // Segment chart hover
            $(document).on('mouseenter', '.formrank-dist-row', this.highlightSegment);
            $(document).on('mouseleave', '.formrank-dist-row', this.unhighlightSegment);
        },

        /**
         * Initialize progress bar animations
         */
        initProgressAnimations: function() {
            // Only run on pages with analytics
            if (!$('.formrank-analytics-grid').length && !$('.formrank-learning-status-card').length) {
                return;
            }

            // Animate bars on page load
            $('.formrank-usage-fill, .formrank-dist-bar, .formrank-funnel-bar').each(function() {
                var $bar = $(this);
                var targetWidth = $bar[0].style.width;

                if (targetWidth) {
                    // Start at 0 and animate to target
                    $bar.css('width', '0');
                    setTimeout(function() {
                        $bar.css('width', targetWidth);
                    }, 100);
                }
            });
        },

        /**
         * Refresh analytics via AJAX
         */
        refreshAnalytics: function(e) {
            if (e) e.preventDefault();

            var $grid = $('.formrank-analytics-grid');

            if (!$grid.length) return;

            $grid.addClass('formrank-loading');

            $.ajax({
                url: formrankLS.ajax_url,
                type: 'POST',
                data: {
                    action: 'formrank_get_conversion_analytics',
                    nonce: formrankLS.nonce
                },
                success: function(response) {
                    if (response.success) {
                        // Update segment chart
                        FormRankConversions.updateSegmentChart(response.data.segments);

                        // Update accuracy display
                        FormRankConversions.updateAccuracyDisplay(response.data.accuracy);

                        // Update funnel
                        FormRankConversions.updateFunnel(response.data.funnel);
                    }
                    $grid.removeClass('formrank-loading');
                },
                error: function() {
                    $grid.removeClass('formrank-loading');
                }
            });
        },

        /**
         * Update segment chart
         */
        updateSegmentChart: function(segments) {
            $.each(segments, function(key, data) {
                var $row = $('.formrank-segment-chart .formrank-dist-row').filter(function() {
                    return $(this).find('.formrank-segment-name').text().toLowerCase() === key;
                });

                if ($row.length) {
                    $row.find('.formrank-dist-bar').css('width', data.rate + '%');
                    $row.find('.formrank-dist-rate').text(data.rate + '%');
                    $row.find('.formrank-dist-count').text('(' + data.converted + '/' + data.total + ')');
                }
            });
        },

        /**
         * Update accuracy display
         */
        updateAccuracyDisplay: function(accuracy) {
            $('.formrank-accuracy-value').text(accuracy.accuracy + '%');
            $('.formrank-accuracy-metric').each(function() {
                var $metric = $(this);
                var label = $metric.find('.formrank-metric-label').text().toLowerCase();

                if (label.indexOf('precision') !== -1) {
                    $metric.find('.formrank-metric-value').text(accuracy.precision + '%');
                } else if (label.indexOf('recall') !== -1) {
                    $metric.find('.formrank-metric-value').text(accuracy.recall + '%');
                }
            });

            // Update confusion matrix
            $('.formrank-tp').find(':not(.formrank-confusion-icon)').last().text(accuracy.true_positives);
            $('.formrank-fp').find(':not(.formrank-confusion-icon)').last().text(accuracy.false_positives);
        },

        /**
         * Update funnel visualization
         */
        updateFunnel: function(funnel) {
            var maxCount = 0;
            $.each(funnel, function(key, data) {
                if (data.count > maxCount) maxCount = data.count;
            });

            $.each(funnel, function(key, data) {
                var $stage = $('.formrank-funnel-stage').filter(function() {
                    return $(this).find('.formrank-funnel-' + key).length;
                });

                if ($stage.length) {
                    var width = maxCount > 0 ? (data.count / maxCount) * 100 : 0;
                    $stage.find('.formrank-funnel-bar').css('width', width + '%');
                    $stage.find('.formrank-funnel-count').text(data.count.toLocaleString());
                    $stage.find('.formrank-funnel-percent').text('(' + data.percent + '%)');
                }
            });
        },

        /**
         * Highlight segment on hover
         */
        highlightSegment: function() {
            $(this).css('background', '#f9fafb');
        },

        /**
         * Unhighlight segment
         */
        unhighlightSegment: function() {
            $(this).css('background', '');
        }
    };

    // Initialize conversions module on document ready
    $(document).ready(function() {
        FormRankConversions.init();
    });

    // Expose to window
    window.FormRankConversions = FormRankConversions;

    // Settings page: weight slider live update
    $(document).ready(function() {
        $('.weight-slider').on('input', function() {
            $('#' + $(this).data('target')).text(this.value + '%');
        });

        // Settings page: retention toggle handler
        $('#formrank-retention-enabled').on('change', function() {
            var $options = $('.formrank-retention-options');
            var $inputs = $options.find('select, input');

            if (this.checked) {
                $options.css('opacity', '1');
                $inputs.prop('disabled', false);
            } else {
                $options.css('opacity', '0.5');
                $inputs.prop('disabled', true);
            }
        });
    });

})(jQuery);
