<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
} // Exit if accessed directly

define( 'FORMLIFT_FIELD_MAX_WIDTH', 100 );

class FormLift_Form {
	var $ID;
	var $form_settings;
	var $style_settings;
	var $default_form_settings;
	var $default_style_settings;
	var $form_name;
	var $uniqueId;

	static $times_form_called = 0;

	/**
	 * Main construct function. This gets all the data related to the form and sets the
	 * class variables.
	 *
	 * @param $id int ID of the form
	 */

	public function __construct( $id ) {
		$this->ID = $id;

		if ( $this->ID != "settings_page" ) {
			$this->form_settings  = get_post_meta( $id, FORMLIFT_SETTINGS, true );
			$this->style_settings = get_post_meta( $id, FORMLIFT_STYLE, true );
		}

		$this->default_form_settings  = get_option( FORMLIFT_SETTINGS );
		$this->default_style_settings = get_option( FORMLIFT_STYLE );

		static::$times_form_called += 1;

		$this->uniqueId  = $this->ID . '-' . static::$times_form_called;
		$this->form_name = "infusion-form-" . $this->uniqueId;
	}

	/**
	 * Allows us to create a unique ID for every form and every field on a page.
	 * Solves the problem of having the same ID for two fields if a user
	 * places the form twice on one page.
	 *
	 * @return int
	 */
	public static function get_times_from_called() {
		return static::$times_form_called;
	}

	/**
	 * Returns the forms setting value. If none exists, return default
	 *
	 * @param $setting_key string
	 * @param $default
	 *
	 * @return mixed
	 */
	public function get_form_setting( $setting_key, $default = null ) {
		if ( $this->ID == "settings_page" ) {
			return $this->get_default_form_setting( $setting_key );
		} else if ( $this->ID != "settings_page" && isset( $this->form_settings[ $setting_key ] ) && ! empty( $this->form_settings[ $setting_key ] ) ) {
			return esc_html( $this->form_settings[ $setting_key ] );
		} else {
			return $default;
		}
	}

	/**
	 * Returns the default setting.
	 *
	 * @param $setting_key string
	 *
	 * @return mixed
	 */
	public function get_default_form_setting( $setting_key ) {
		return esc_html( get_formlift_setting( $setting_key ) );
	}

	/**
	 * Returns the forms style meta value. If none exists, return the default option.
	 *
	 * @param $class     string
	 * @param $attribute string
	 * @param $default
	 *
	 * @return mixed
	 */
	public function get_style_setting( $class, $attribute, $default = null ) {
		if ( $this->ID == "settings_page" ) {
			return $this->get_default_style_setting( $class, $attribute );
		} else if ( $this->ID != "settings_page" && isset( $this->style_settings[ $class ][ $attribute ] ) && ! empty( $this->style_settings[ $class ][ $attribute ] ) ) {
			return esc_html( $this->style_settings[ $class ][ $attribute ] );
		} else {
			return $default;
		}
	}

	/**
	 * return the default option.
	 *
	 * @param $class     string
	 * @param $attribute string
	 *
	 * @return mixed
	 */
	public function get_default_style_setting( $class, $attribute ) {
		return ( isset( $this->default_style_settings[ $class ][ $attribute ] ) ) ? esc_html( $this->default_style_settings[ $class ][ $attribute ] ) : '';
	}

	public function get_fields() {
		return get_post_meta( $this->ID, FORMLIFT_FIELDS, true );
	}

	/**
	 * This function is used to determine if this form is being called upon within the specified date range
	 * or whether or not it's submission count is under that which is specified by the user.
	 *
	 * @return bool
	 */
	private function display() {
		/* gets the display condition of the form as setting in the display settings*/
		$display = true;

		$display = apply_filters( 'formlift_should_display_form', $display, $this );

		if ( get_post_status( $this->ID ) != 'publish' ) {
			$display = false;
		}

		return $display;

	}

	/**
	 * return a string of html characters representing the form
	 *
	 * @return string
	 */
	public function get_form() {
		$code = "";

		do_action( 'formlift_before_get_form_code', $this->ID );

		$code = apply_filters( 'formlift_prepend_to_form_code', $code, $this->ID );

		$show_post_url = $this->get_form_setting( 'show_post_url' );

		$target = ( $this->get_form_setting( 'target_blank' ) ) ? '_blank' : '_self';

		//danger this is bad.
		if ( $show_post_url ) {
			$post_url = $this->get_form_setting( 'post_url' );
			$code     .= "<form id=\"$this->form_name\" class=\"$this->form_name formlift-form-container\" action=\"$post_url\" data-id=\"$this->uniqueId\" data-trackingid='$this->ID' method=\"post\" target=\"$target\">";
			$xid      = get_post_meta( $this->ID, 'inf_form_xid', true );
			$code     .= "<input type=\"hidden\" name=\"inf_form_xid\" value=\"{$xid}\">";
		} else {
			//default as it should be
			$action = $this->get_form_setting( 'enable_compatibility_mode', false ) ? "" : "return formliftSubmitV2( this )";
			$code   .= "<form id=\"$this->form_name\" class=\"$this->form_name formlift-form-container\" onsubmit=\"{$action}\" data-id=\"$this->uniqueId\" data-trackingid='$this->ID' target=\"$target\" method=\"post\">";
		}

		/* nonce */
//        $code.= wp_nonce_field( 'formlift_submit', 'formlift_submit_nonce', true, false );

		$code .= "<div class='formlift-infusion-form'>";
		/* needs to be submitted along for validation */
		$code .= "<input type='hidden' name='form_id' class='remove-on-submit' value='$this->ID'>";
		/* this is so ajax submission works */
		$code .= "<input type='hidden' name='action' class='remove-on-submit' value='formlift_submit_form'>";
		/* standard Infusionsoft fields */

		// in case the form isn't actually an infusionsoft form.
		$formName = get_post_meta( $this->ID, 'inf_form_name', true );
		if ( ! empty( $formName ) ) {
			$code .= "<input type='hidden' name='inf_form_name' value=\"{$formName}\">";
		}

		$infusionsoftVersion = get_post_meta( $this->ID, 'infusionsoft_version', true );
		if ( ! empty( $infusionsoftVersion ) ) {
			$code .= "<input type='hidden' name='infusionsoft_version' value=\"{$infusionsoftVersion}\">";
		}

		/* infusionsoft timezone code */
		$code .= "<input type='hidden' name='timeZone' id='time_zone-{$this->uniqueId}'>";
		$code .= "<script>jQuery(document).ready( function(){var timezone = jstz.determine();jQuery( '#time_zone-{$this->uniqueId}' ).val( timezone.name() );});</script>";

		//fields
		$code .= $this->get_field_code();

		//start message container
		$code .= "<div class=\"formlift_field formlift-col formlift-span_4_of_4\">";

		//Waiting loader
		$waitText = $this->get_form_setting( 'please_wait_text', get_formlift_setting( 'please_wait_text' ) );
		$code     .= "<div id=\"wait-$this->uniqueId\" class=\"formlift-wait-message\" style=\"display:none;\"><div class=\"formlift-loader\"></div><div class=\"formlift-wait-text-container\">$waitText</div></div>";

		//Success Message
		$success_message = $this->get_form_setting( 'success_message', get_formlift_setting( 'success_message' ) );
		$code            .= "<div id=\"success-$this->uniqueId\" class=\"formlift-success-message\" style=\"display:none;\"><div class=\"formlift-wait-text-container\"> {$success_message}</div></div>";

		//Error Message
		$error_message = $this->get_form_setting( "invalid_data_error", get_formlift_setting( 'invalid_data_error' ) );
		$code          .= "<div id=\"error-$this->uniqueId\" class=\"formlift-error-message\" style=\"display:none;\"><div class=\"formlift-wait-text-container\"> {$error_message}</div></div>";

		//Close message container
		$code .= "</div>";

		//End innner container
		$code .= "</div>";

		//End form
		$code .= "</form>";

		$version = FORMLIFT_VERSION;
		$code    .= "<!-- This web form is powered by FormLift v.{$version}. https://formlift.net -->";

		$code = apply_filters( 'formlift_append_to_form_code', $code, $this->ID );

		do_action( 'formlift_after_get_form_code', $this->ID );

		return $code;
	}

	public function get_preview_form() {
		/* load required scrips */
		self::add_scripts();

		$code = "<div id=\"{$this->form_name}\" class='{$this->form_name} formlift-form-container'>";
		$code .= "<div class='formlift-infusion-form'>";
		$code .= $this->get_field_code();
		$code .= "</div>";
		$code .= "</div>";

		return $code;
	}

	private function get_field_code() {
		$fields = get_post_meta( $this->ID, FORMLIFT_FIELDS, true );

		/**
		 * add compatibility for silly people
		 */
		if ( empty( $fields ) ) {
			$fields = get_post_meta( $this->ID, 'flp_form_bits', true );
		}

		$field_class = apply_filters( 'formlift_field_class', 'FormLift_Field' );

		$count           = 0;
		$compiled_fields = array();
		foreach ( $fields as $field_options ) {
			$field = new $field_class( $field_options, $this->ID );
			array_push( $compiled_fields, $field );
		}
		$code = "";
		foreach ( $compiled_fields as $i => $field ) {
			$code  .= $field;
			$count += $field->get_decimal_size();

			$size_next = ( isset( $compiled_fields[ $i + 1 ] ) ) ? $compiled_fields[ $i + 1 ]->get_decimal_size() : 0;

			if ( $count >= FORMLIFT_FIELD_MAX_WIDTH ) {
				$code  .= "<div class=\"formlift-clearfix\"></div>";
				$count = 0;
			} else if ( $count + $size_next > FORMLIFT_FIELD_MAX_WIDTH ) {
				$code  .= "<div class=\"formlift-clearfix\"></div>";
				$count = 0;
			}
		}

		return $code;
	}

	/**
	 * Get the stylesheet for a form and populate it with the meta from the form's meta array.
	 *
	 * @return String
	 */
	public function get_style_sheet() {

		$stylesheet = "";

		/* iterate over the globals and see if the form as overides */
		$settings = apply_filters( 'formlift_form_stylesheet', $this->default_style_settings, $this->ID );

		foreach ( $settings as $class => $attributes ) {
			$stylesheet .= "#$this->form_name ." . $class . "{";

			foreach ( $attributes as $attribute => $value ) {
				$stylesheet .= $attribute . ":" . $value . ";";
			}

			$stylesheet .= "}";
		}

		return "<style>$stylesheet</style>";
	}

	/**
	 * Init the form via shortcode call.
	 *
	 * @return mixed|String
	 */
	public static function do_shortcode( $atts ) {
		//self::add_scripts();

		$atters = shortcode_atts( array(
			'id' => ''
		), $atts );

		$id = absint( $atters['id'] );

		if ( ! $id ) {
			return "Invalid form ID provided.";
		}

		$post_type = get_post_type( $id );

		if ( 'infusion_form' !== $post_type ) {
			return "The provided ID is not related to a form.";
		}

		return new FormLift_Form( $atters['id'] );

	}

	public function __toString() {
		if ( $this->display() ) {
			$the_content = $this->get_style_sheet();
			$the_content .= $this->get_form();

			return $the_content;
		} else {
			$content = "";
			$content = apply_filters( "formlift_no_display_message", $content, $this );

			return $content;
		}
	}

	public static function add_admin_scripts() {
		/* only load on relevant admin pages */
		if ( is_admin() ) {
			$screen = get_current_screen();
			if ( $screen->post_type != "infusion_form" ) {
				return;
			}
		}

		self::add_scripts();
	}

	public static function add_scripts() {
		/* add jQuery */
		wp_enqueue_script( 'jquery' );
		/* date picker */
		wp_enqueue_script( 'jquery-ui-datepicker' );
		wp_enqueue_style( 'jquery-ui-datepicker', "https://code.jquery.com/ui/1.12.1/themes/base/jquery-ui.css" );
		/* time zone library */
		wp_enqueue_script( 'formlift-timezone', plugins_url( 'assets/js/jstz.js', __FILE__ ), array(), FORMLIFT_JS_VERSION );
		/* form submit */
		wp_enqueue_script( 'formlift-submit', plugins_url( 'assets/js/submit.js', __FILE__ ), array(), FORMLIFT_JS_VERSION );
		wp_localize_script( 'formlift-submit', 'formlift_ajax_object', array( 'ajax_url' => parse_url( admin_url( 'admin-ajax.php' ), PHP_URL_PATH ) ) );
		/* styles */
		wp_enqueue_style( 'formlift-grid', plugins_url( 'assets/css/responsive-grid-framework.css', __FILE__ ), array(), FORMLIFT_CSS_VERSION );
		wp_enqueue_style( 'formlift-styling', plugins_url( 'assets/css/style.css', __FILE__ ), array(), FORMLIFT_CSS_VERSION );
		//do_action( 'formlift_form_load_dependent_scripts');
	}
}

add_action( 'admin_enqueue_scripts', array( 'FormLift_Form', 'add_admin_scripts' ) );
add_action( 'wp_enqueue_scripts', array( 'FormLift_Form', 'add_scripts' ) );

add_shortcode( 'infusion_form', array( 'FormLift_Form', 'do_shortcode' ) );
add_shortcode( 'formlift', array( 'FormLift_Form', 'do_shortcode' ) );

function formlift_credit( $echo = false ) {
	$credit_class = uniqid( 'formlift_credit_' );

	$content = "
<style>
.formlift-form-container .formlift-infusion-form .formlift_field div.{$credit_class} a{
    color:inherit !important;
}

.formlift-form-container .formlift-infusion-form .formlift_field div.{$credit_class} a:hover{
    text-decoration:underline !important;
}
</style>";

	$content .= "<div class=\"{$credit_class}\" style='display: block !important;margin: 4px 0 0 0 !important;font-size: 12px !important;text-decoration:none !important;'><img class='emoji' src='https://s.w.org/images/core/emoji/2.4/svg/26a1.svg' width='14'> by <a title=\"FormLift\" href=\"https://formlift.net\" target=\"_blank\" rel=\"noopener\">FormLift</a></div>";

	if ( $echo ) {
		echo $content;

		return 1;
	} else {
		return $content;
	}
}
