<?php
/**
 * Plugin Name: Form for Capsule CRM
 * Description: A WordPress plugin to generate a lead capture form for Capsule CRM using a shortcode with customizable options.
 * Version: 1.0.0
 * Author: Forms for CRM
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: form-capsule-crm
 * Domain Path: /languages
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

class FORMFOCA_Main {

    private $table_name;

    public function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'formfoca_keys';

        register_activation_hook(__FILE__, [self::class, 'install']);
        add_shortcode('capsule_form', [$this, 'render_shortcode']);
        add_action('admin_post_formfoca_save_key', [$this, 'save_key']);
        
        // Add actions for enqueueing scripts and styles
        add_action('wp_enqueue_scripts', [$this, 'enqueue_assets']);
        
        // Add admin menu
        add_action('admin_menu', [$this, 'add_admin_menu']);
    }

    public static function install() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'formfoca_keys';
        $sql = "CREATE TABLE {$table_name} (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            form_key varchar(255) NOT NULL,
            PRIMARY KEY  (id)
        ) {$wpdb->get_charset_collate()};";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    public function save_key() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'form-capsule-crm'));
        }

        check_admin_referer('formfoca_save_key_nonce', 'formfoca_nonce');

        $form_key = isset($_POST['form_key']) ? sanitize_text_field(wp_unslash($_POST['form_key'])) : '';

        global $wpdb;
        // Clear existing keys
        $wpdb->delete($this->table_name, []);
        // Insert new key
        $wpdb->insert(
            $this->table_name,
            [
                'form_key' => $form_key
            ],
            [
                '%s'
            ]
        );

        // Clear the cache
        wp_cache_delete('formfoca_form_key');

        // Redirect back to the settings page
        wp_redirect(add_query_arg(
            ['page' => 'formfoca-settings', 'updated' => 'true'],
            admin_url('options-general.php')
        ));
        exit;
    }

    public function get_form_key() {
        $cache_key = 'formfoca_form_key';
        $form_key = wp_cache_get($cache_key);

        if (false === $form_key) {
            global $wpdb;
            $table = $this->table_name;
            $form_key = $wpdb->get_var(
                "SELECT form_key FROM {$table} LIMIT 1"
            );
            
            // Cache the result for 12 hours (43200 seconds)
            wp_cache_set($cache_key, $form_key, '', 43200);
        }

        return $form_key;
    }

    public function enqueue_assets() {
        // Register and enqueue styles
        wp_register_style(
            'formfoca-styles',
            plugins_url('css/capsule-form.css', __FILE__),
            [],
            '1.0.0'
        );
        wp_enqueue_style('formfoca-styles');
    }

    public function render_shortcode($atts) {
        $atts = shortcode_atts([
            'complete_url' => '',
        ], $atts);

        $form_key = $this->get_form_key();
        if (!$form_key) {
            return '<p>' . esc_html__('Please configure the Capsule CRM form key in the admin settings.', 'form-capsule-crm') . '</p>';
        }

        $complete_url = esc_url($atts['complete_url']);

        // Add WordPress theme-compatible styling
        $output = '<div class="formfoca-container">';
        $output .= "<form class='capsule-crm-form' action=\"https://service.capsulecrm.com/service/newlead\" accept-charset=\"UTF-8\" method=\"post\">
            <input type=\"hidden\" name=\"FORM_ID\" value=\"$form_key\">
            <input type=\"hidden\" name=\"COMPLETE_URL\" value=\"$complete_url\">
            
            <div class='form-row name-row'>
                <p class='form-column'>
                    <label for='capsule-first-name'>" . esc_html__('First name', 'form-capsule-crm') . "</label>
                    <input type='text' id='capsule-first-name' name='FIRST_NAME' class='input-text'>
                </p>
                
                <p class='form-column'>
                    <label for='capsule-last-name'>" . esc_html__('Last name', 'form-capsule-crm') . "</label>
                    <input type='text' id='capsule-last-name' name='LAST_NAME' class='input-text'>
                </p>
            </div>
            
            <p class='form-row'>
                <label for='capsule-email'>" . esc_html__('Email', 'form-capsule-crm') . "</label>
                <input type='email' id='capsule-email' name='EMAIL' class='input-text'>
            </p>
            
            <p class='form-row'>
                <label for='capsule-note'>" . esc_html__('Message', 'form-capsule-crm') . "</label>
                <textarea id='capsule-note' name='NOTE' class='input-text' rows='4'></textarea>
            </p>
            
            <p class='form-row'>
                <input type='submit' value='" . esc_attr__('Submit', 'form-capsule-crm') . "' class='button button-primary'>
            </p>
        </form>";
        $output .= '</div>';

        return $output;
    }
    
    public function add_admin_menu() {
        add_options_page(
            esc_html__('Form for Capsule CRM Settings', 'form-capsule-crm'),
            esc_html__('Capsule CRM Form', 'form-capsule-crm'),
            'manage_options',
            'formfoca-settings',
            [$this, 'render_settings_page']
        );
    }
    
    public function render_settings_page() {
        $form_key = $this->get_form_key();
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('Form for Capsule CRM Settings', 'form-capsule-crm'); ?></h1>
            <div class="card" style="max-width: 800px; margin-bottom: 20px; padding: 10px 20px;">
                <h3><?php echo esc_html__('How to Use the Shortcode', 'form-capsule-crm'); ?></h3>
                <p><?php echo esc_html__('Use the following shortcode to display the Capsule CRM form on any page or post:', 'form-capsule-crm'); ?></p>
                <code>[capsule_form]</code>
                <p><?php echo esc_html__('To specify a custom completion URL, use the complete_url parameter:', 'form-capsule-crm'); ?></p>
                <code>[capsule_form complete_url="https://your-site.com/thank-you"]</code>
            </div>
            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                <input type="hidden" name="action" value="formfoca_save_key">
                <?php wp_nonce_field('formfoca_save_key_nonce', 'formfoca_nonce'); ?>
                <table class="form-table">
                    <tr valign="top">
                        <th scope="row"><?php echo esc_html__('Form Key', 'form-capsule-crm'); ?></th>
                        <td><input type="text" name="form_key" value="<?php echo esc_attr($form_key); ?>" class="regular-text"></td>
                    </tr>
                </table>
                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }
}

// Initialize the plugin
$formfoca_plugin = new FORMFOCA_Main();
