<?php
/*
Plugin Name: Force Sells for Variations
Description: Adds products to specific variations and synchronizes quantities. Removes quantity change and remove buttons for these added products.
Version: 1.0
Author: Benny Windolph
Author URI: https://hechtinsgefecht.de/benny-windolph/
Text Domain: force-sells-for-variations
Domain Path: /languages
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
*/

// Ensure the plugin is not accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

// Add products to the cart when a variation is added
add_action( 'woocommerce_before_calculate_totals', 'fsfv_add_forced_sell_products' );

function fsfv_add_forced_sell_products( $cart ) {
    if ( is_admin() && ! defined( 'DOING_AJAX' ) )
        return;

    foreach ( $cart->get_cart() as $cart_item_key => $cart_item ) {
        $variation_id = $cart_item['variation_id'];
        $forced_sell_products = get_post_meta( $variation_id, 'forced_sell_products', true );

        if ( $forced_sell_products ) {
            $forced_product_ids = explode( ',', $forced_sell_products );

            foreach ( $forced_product_ids as $product_id ) {
                $product_id = trim( $product_id );

                if ( ! is_unique_product_in_cart( $product_id, $variation_id, $cart ) ) {
                    $cart->add_to_cart( $product_id, $cart_item['quantity'], 0, array(), array( 'parent_variation_id' => $variation_id, 'unique_key' => $variation_id . '_' . $product_id ) );
                }
            }
        }
    }
}

function fsfv_is_unique_product_in_cart( $product_id, $variation_id, $cart ) {
    foreach ( $cart->get_cart() as $cart_item_key => $cart_item ) {
        if ( $cart_item['product_id'] == $product_id && isset($cart_item['unique_key']) && $cart_item['unique_key'] == $variation_id . '_' . $product_id ) {
            return true;
        }
    }
    return false;
}

// Sync quantities of the products
add_action( 'woocommerce_after_cart_item_quantity_update', 'fsfv_sync_forced_sell_product_quantities', 20, 4 );

function fsfv_sync_forced_sell_product_quantities( $cart_item_key, $quantity, $old_quantity, $cart ) {
    $cart_item = $cart->get_cart_item( $cart_item_key );
    $variation_id = $cart_item['variation_id'];
    $forced_sell_products = get_post_meta( $variation_id, 'forced_sell_products', true );

    if ( $forced_sell_products ) {
        $forced_product_ids = explode( ',', $forced_sell_products );

        foreach ( $cart->get_cart() as $key => $item ) {
            if ( in_array( $item['product_id'], $forced_product_ids ) && isset($item['parent_variation_id']) && $item['parent_variation_id'] == $variation_id ) {
                $cart->set_quantity( $key, $quantity );
            }
        }
    }
}

// Remove the added products when the main variation is removed
add_action( 'woocommerce_cart_item_removed', 'fsfv_remove_forced_sell_products', 10, 2 );

function fsfv_remove_forced_sell_products( $cart_item_key, $cart ) {
    $cart_item = $cart->get_removed_cart_contents()[$cart_item_key];
    $variation_id = $cart_item['variation_id'];
    $forced_sell_products = get_post_meta( $variation_id, 'forced_sell_products', true );

    if ( $forced_sell_products ) {
        $forced_product_ids = explode( ',', $forced_sell_products );

        foreach ( $cart->get_cart() as $key => $item ) {
            if ( in_array( $item['product_id'], $forced_product_ids ) && isset($item['parent_variation_id']) && $item['parent_variation_id'] == $variation_id ) {
                $cart->remove_cart_item( $key );
            }
        }
    }
}

// Add note to additional products in the cart
add_filter( 'woocommerce_cart_item_name', 'fsfv_add_note_to_forced_sell_products', 10, 3 );

function fsfv_add_note_to_forced_sell_products( $product_name, $cart_item, $cart_item_key ) {
    if ( isset( $cart_item['parent_variation_id'] ) ) {
        $variation = wc_get_product( $cart_item['parent_variation_id'] );
        if ( $variation ) {
            $variation_name = $variation->get_name();
// Translators: %s is the name of the variation
            $product_name .= '<br><small>' . sprintf( __( 'Part of "%s"', 'force-sells-for-variations' ), $variation_name ) . '</small>';
        }
    }
    return $product_name;
}

// Disable quantity change and remove buttons for additional products
add_filter( 'woocommerce_cart_item_quantity', 'fsfv_disable_quantity_change_for_forced_sell_products', 10, 3 );
add_filter( 'woocommerce_cart_item_remove_link', 'fsfv_disable_remove_link_for_forced_sell_products', 10, 2 );

function fsfv_disable_quantity_change_for_forced_sell_products( $product_quantity, $cart_item_key, $cart_item ) {
    if ( isset( $cart_item['parent_variation_id'] ) ) {
        return '';
    }
    return $product_quantity;
}

function fsfv_disable_remove_link_for_forced_sell_products( $remove_link, $cart_item_key ) {
    $cart = WC()->cart->get_cart();
    if ( isset( $cart[$cart_item_key]['parent_variation_id'] ) ) {
        return '';
    }
    return $remove_link;
}

// Adjust cart item count to exclude added products
add_filter( 'woocommerce_cart_contents_count', 'fsfv_adjust_cart_contents_count' );

function fsfv_adjust_cart_contents_count( $count ) {
    $cart = WC()->cart->get_cart();
    $adjusted_count = 0;

    foreach ( $cart as $cart_item ) {
        if ( ! isset( $cart_item['parent_variation_id'] ) ) {
            $adjusted_count += $cart_item['quantity'];
        }
    }
    return $adjusted_count;
}
?>
