<?php
/**
 * Notifications class.
 *
 * @package Fomo_Notifications
 * @author WP Zinc
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles creating, editing and deleting Notifications.
 *
 * @package Fomo_Notifications
 * @author WP Zinc
 */
class Fomo_Notifications_Notifications {

	/**
	 * Returns all Notifications
	 *
	 * @since   1.0.0
	 *
	 * @return  bool|array
	 */
	public function get_all() {

		// Get all notifications.
		$notifications = new WP_Query(
			array(
				'post_type'              => 'fomo-notification',
				'post_status'            => 'publish',
				'cache_results'          => false,
				'update_post_meta_cache' => false,
				'update_post_term_cache' => false,
			)
		);

		// Bail if none exist.
		if ( count( $notifications->posts ) === 0 ) {
			return false;
		}

		// Build array.
		$notifications_arr = array();
		foreach ( $notifications->posts as $notification ) {
			// Get settings.
			$notification_settings = new Fomo_Notifications_Notification_Settings( $notification->ID );

			// Skip if an error occured.
			if ( is_wp_error( $notification_settings ) ) {
				continue;
			}

			$notifications_arr[ $notification->ID ] = array_merge(
				array(
					'title' => $notification->post_title,
				),
				$notification_settings->get()
			);
		}

		/**
		 * Filters the Notifications to return.
		 *
		 * @since   1.0.0
		 *
		 * @param   array       $notifications_arr Notifications.
		 * @param   WP_Query    $notifications     Notifications Query.
		 */
		$notifications_arr = apply_filters( 'fomo_notifications_notifications_get_all', $notifications_arr, $notifications );

		// Return filtered results.
		return $notifications_arr;

	}

	/**
	 * Returns the IDs of all Notifications.
	 *
	 * @since   1.0.0
	 *
	 * @return  array
	 */
	public function get_ids() {

		// Get all notifications.
		$notifications = new WP_Query(
			array(
				'post_type'              => 'fomo-notification',
				'post_status'            => 'publish',

				'fields'                 => 'ids',
				'cache_results'          => false,
				'update_post_meta_cache' => false,
				'update_post_term_cache' => false,
			)
		);

		return $notifications->posts;

	}

	/**
	 * Returns the number of Notifications.
	 *
	 * @since   1.0.0
	 *
	 * @return  int
	 */
	public function get_count() {

		return count( $this->get_ids() );

	}

	/**
	 * Creates a new Notification.
	 *
	 * @since   1.0.0
	 *
	 * @param   string $title      Notification Title.
	 * @param   array  $settings   Notification Settings.
	 * @return  int                 Notification ID.
	 */
	public function create( $title, $settings ) {

		// Create Notification Post.
		$post_id = wp_insert_post(
			array(
				'post_title'  => $title,
				'post_type'   => 'fomo-notification',
				'post_status' => 'publish',
			),
			true
		);

		// Bail if the post was not created.
		if ( is_wp_error( $post_id ) ) {
			return $post_id;
		}

		// Load settings.
		$notification = new Fomo_Notifications_Notification_Settings( $post_id );

		// Save settings.
		$notification->save( $settings );

		// Return.
		return $post_id;

	}

}
