<?php
/**
 * Fired when the plugin is uninstalled.
 *
 * When populating this file, consider the following flow:
 * - This method should be static
 * - Check if the $_REQUEST['plugin'] content actually is this plugin file
 * - Run an admin referrer check to make sure it goes through authentication
 * - Verify the output of $_GET makes sense
 * - Repeat with other user roles. Best directly by using the links/query string parameters.
 * - Repeat things for multisite. Once for a single site in the network, once sitewide.
 *
 * @link       https://flxwoo.com
 * @since      2.1.0
 *
 * @package    FlxWoo
 */

// If uninstall not called from WordPress, then exit.
if (!defined('WP_UNINSTALL_PLUGIN')) {
    exit;
}

/**
 * Clean up FlxWoo data on uninstall
 *
 * This removes:
 * - Plugin settings from wp_options table
 * - Feature flags and activity log data
 * - Custom database tables (flx_activity_log)
 * - All transients created by the plugin
 */

/**
 * Delete all plugin data for a single site
 */
function flx_woo_delete_site_data() {
    global $wpdb;

    // Delete all plugin options
    $options_to_delete = [
        'flx_woo_settings',
        'flx_woo_db_version',
        'flx_woo_feature_activity_log',
        'flx_woo_feature_flags',
        'flx_woo_kill_switch',
        'flx_woo_feature_retention_period',
        'flx_woo_last_render_time',
        'flx_woo_render_stats_24h',
        'flx_woo_allow_bypass',
        'flxwoo_analytics_enabled',
        'flxwoo_last_aggregation',
        'flx_woo_version',
    ];

    foreach ($options_to_delete as $option) {
        delete_option($option);
    }

    // Delete all transients with our prefixes
    $transient_prefixes = [
        'flx_woo_',
        'flxwoo_',
        '_transient_flx_woo_',
        '_transient_timeout_flx_woo_',
        '_transient_flxwoo_',
        '_transient_timeout_flxwoo_',
    ];

    foreach ($transient_prefixes as $prefix) {
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
                $wpdb->esc_like($prefix) . '%'
            )
        );
    }

    // Drop custom database table
    $table_name = $wpdb->prefix . 'flx_activity_log';
    $wpdb->query("DROP TABLE IF EXISTS {$table_name}");
}

// Delete data for single site or main site in multisite
flx_woo_delete_site_data();

// For multisite installations, clean up all sites
if (is_multisite()) {
    // Get all site IDs using WordPress function
    $flx_woo_site_ids = get_sites(['fields' => 'ids']);

    foreach ($flx_woo_site_ids as $flx_woo_site_id) {
        switch_to_blog($flx_woo_site_id);
        flx_woo_delete_site_data();
        restore_current_blog();
    }
}
