<?php
/**
 * Rate Limiter Utility
 *
 * Implements sliding window rate limiting for REST API endpoints.
 * Protects against abuse and DDoS attacks.
 *
 * Algorithm: Sliding Window Counter
 * - More accurate than fixed window (prevents burst at boundaries)
 * - Memory efficient (only stores timestamps)
 * - Simple to implement and reason about
 *
 * Storage: WordPress Transients
 * - Uses WordPress transient API for storage
 * - Automatic cleanup via WordPress cron
 * - Compatible with object caching plugins (Redis, Memcached)
 *
 * @package FlxWoo\Utils
 * @since 2.1.0
 */

namespace FlxWoo\Utils;

if (!defined('ABSPATH')) {
  exit;
}

class RateLimiter {
  /**
   * Rate limit configurations (requests per minute)
   */
  const RATE_LIMITS = [
    'site-info' => [
      'limit' => 120,
      'window' => 60, // seconds
    ],
    'render' => [
      'limit' => 60,
      'window' => 60, // seconds
    ],
  ];

  /**
   * Transient prefix for rate limit data
   */
  const TRANSIENT_PREFIX = 'flx_woo_rate_limit_';

  /**
   * Transient prefix for violation log tracking
   */
  const VIOLATION_LOG_PREFIX = 'flx_woo_rate_violation_';

  /**
   * Violation log interval (1 hour in seconds)
   */
  const VIOLATION_LOG_INTERVAL = 3600;

  /**
   * Check rate limit for a request
   *
   * @param string $identifier Rate limit identifier (e.g., 'site-info', 'render')
   * @return array{allowed: bool, current: int, limit: int, retry_after: int, reset_at: int, remaining: int}
   */
  public static function check_rate_limit(string $identifier): array {
    $config = self::RATE_LIMITS[$identifier] ?? null;

    if (!$config) {
      // Unknown identifier - allow by default
      return [
        'allowed' => true,
        'current' => 0,
        'limit' => 0,
        'retry_after' => 0,
        'reset_at' => 0,
        'remaining' => 999,
      ];
    }

    $ip = self::get_client_ip();
    $key = self::get_transient_key($ip, $identifier);
    $now = time();
    $window_start = $now - $config['window'];

    // Get existing timestamps for this key
    $timestamps = get_transient($key);
    if ($timestamps === false) {
      $timestamps = [];
    }

    // Filter out timestamps outside the current window (sliding window)
    $timestamps = array_filter($timestamps, function ($timestamp) use ($window_start) {
      return $timestamp > $window_start;
    });

    // Re-index array after filtering
    $timestamps = array_values($timestamps);

    // Calculate current count and remaining
    $current = count($timestamps);
    $remaining = max(0, $config['limit'] - $current - 1);
    $allowed = $current < $config['limit'];

    // Calculate reset time (end of current window)
    $oldest_timestamp = $timestamps[0] ?? $now;
    $reset_at = $oldest_timestamp + $config['window'];
    $retry_after = max(1, $reset_at - $now);

    // If allowed, add current timestamp
    if ($allowed) {
      $timestamps[] = $now;
      // Store with expiration = window duration
      set_transient($key, $timestamps, $config['window']);
    } else {
      // Rate limit exceeded - log first violation per hour
      self::log_rate_limit_violation($ip, $identifier, $current, $config['limit'], $retry_after);
    }

    return [
      'allowed' => $allowed,
      'current' => $current,
      'limit' => $config['limit'],
      'retry_after' => $retry_after,
      'reset_at' => $reset_at,
      'remaining' => $remaining,
    ];
  }

  /**
   * Extract client IP address from request
   *
   * Priority order:
   * 1. HTTP_X_FORWARDED_FOR header (first IP, if behind proxy)
   * 2. HTTP_X_REAL_IP header (if behind proxy)
   * 3. REMOTE_ADDR (direct connection)
   *
   * @return string Client IP address
   */
  private static function get_client_ip(): string {
    // Check X-Forwarded-For header (could be comma-separated list)
    if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
      $raw_value = $_SERVER['HTTP_X_FORWARDED_FOR']; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Validated after unslashing and sanitization
      $value = function_exists('wp_unslash') ? wp_unslash($raw_value) : stripslashes($raw_value);
      $forwarded_ips = explode(',', sanitize_text_field($value));
      $first_ip = trim($forwarded_ips[0]);
      if (!empty($first_ip)) {
        return $first_ip;
      }
    }

    // Check X-Real-IP header
    if (!empty($_SERVER['HTTP_X_REAL_IP'])) {
      $raw_value = $_SERVER['HTTP_X_REAL_IP']; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Validated after unslashing and sanitization
      $value = function_exists('wp_unslash') ? wp_unslash($raw_value) : stripslashes($raw_value);
      return sanitize_text_field($value);
    }

    // Fallback to REMOTE_ADDR (direct connection)
    if (isset($_SERVER['REMOTE_ADDR'])) {
      $raw_value = $_SERVER['REMOTE_ADDR']; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Validated after unslashing and sanitization
      $value = function_exists('wp_unslash') ? wp_unslash($raw_value) : stripslashes($raw_value);
      return sanitize_text_field($value);
    }

    return 'unknown';
  }

  /**
   * Generate transient key for rate limit storage
   *
   * Format: flx_woo_rate_limit_{sanitized_ip}_{identifier}
   *
   * @param string $ip IP address
   * @param string $identifier Rate limit identifier
   * @return string Transient key
   */
  private static function get_transient_key(string $ip, string $identifier): string {
    // Sanitize IP for use in transient key
    $sanitized_ip = str_replace(['.', ':'], '_', $ip);
    return self::TRANSIENT_PREFIX . $sanitized_ip . '_' . $identifier;
  }

  /**
   * Generate violation log key for rate limit logging
   *
   * @param string $ip IP address
   * @return string Transient key
   */
  private static function get_violation_log_key(string $ip): string {
    $sanitized_ip = str_replace(['.', ':'], '_', $ip);
    return self::VIOLATION_LOG_PREFIX . $sanitized_ip;
  }

  /**
   * Log rate limit violation (only first per IP per hour to reduce noise)
   *
   * @param string $ip Client IP address
   * @param string $identifier Rate limit identifier
   * @param int $current Current request count
   * @param int $limit Rate limit
   * @param int $retry_after Retry after seconds
   */
  private static function log_rate_limit_violation(
    string $ip,
    string $identifier,
    int $current,
    int $limit,
    int $retry_after
  ): void {
    $log_key = self::get_violation_log_key($ip);
    $last_log_time = get_transient($log_key);

    // Only log if we haven't logged for this IP in the last hour
    if ($last_log_time === false) {
      // Sanitize IP for GDPR (mask last octet)
      $sanitized_ip = self::sanitize_ip_for_logging($ip);

      Logger::warning('Rate limit exceeded for ' . $identifier, [
        'ip' => $sanitized_ip,
        'endpoint' => $identifier,
        'current' => $current,
        'limit' => $limit,
        'retry_after' => $retry_after,
      ]);

      // Store log timestamp (expires in 1 hour)
      set_transient($log_key, time(), self::VIOLATION_LOG_INTERVAL);
    }
  }

  /**
   * Sanitize IP address for logging (GDPR compliance)
   * Masks last octet: 192.168.1.100 -> 192.168.1.xxx
   *
   * @param string $ip IP address
   * @return string Sanitized IP address
   */
  private static function sanitize_ip_for_logging(string $ip): string {
    if ($ip === 'unknown') {
      return $ip;
    }

    // IPv4: mask last octet
    if (strpos($ip, '.') !== false) {
      $parts = explode('.', $ip);
      if (count($parts) === 4) {
        $parts[3] = 'xxx';
        return implode('.', $parts);
      }
    }

    // IPv6: mask last segment
    if (strpos($ip, ':') !== false) {
      $parts = explode(':', $ip);
      if (count($parts) > 0) {
        $parts[count($parts) - 1] = 'xxxx';
        return implode(':', $parts);
      }
    }

    return $ip;
  }

  /**
   * Get rate limit headers for response
   *
   * @param array $result Rate limit result from check_rate_limit()
   * @return array Headers array
   */
  public static function get_rate_limit_headers(array $result): array {
    return [
      'X-RateLimit-Limit' => (string) $result['limit'],
      'X-RateLimit-Remaining' => (string) $result['remaining'],
      'X-RateLimit-Reset' => (string) $result['reset_at'],
    ];
  }

  /**
   * Create 429 rate limit WP_Error
   *
   * @param array $result Rate limit result from check_rate_limit()
   * @return \WP_Error
   */
  public static function create_rate_limit_error(array $result): \WP_Error {
    return new \WP_Error(
      'rate_limit_exceeded',
      'Too many requests. Please try again later.',
      [
        'status' => 429,
        'retry_after' => $result['retry_after'],
        'headers' => self::get_rate_limit_headers($result),
      ]
    );
  }
}
