<?php
/**
 * FlxWoo Logging Utility
 *
 * Provides structured logging with error categories, context data,
 * and PII sanitization for WordPress debug.log.
 *
 * @package FlxWoo\Utils
 * @since 2.1.0
 */

namespace FlxWoo\Utils;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Logger
 *
 * Structured logging for FlxWoo with consistent formatting and sanitization.
 */
class Logger {
	/**
	 * Log levels
	 */
	const ERROR   = 'ERROR';
	const WARNING = 'WARNING';
	const INFO    = 'INFO';
	const DEBUG   = 'DEBUG';

	/**
	 * Log an error message
	 *
	 * Use for errors that prevent normal operation.
	 *
	 * @param string $message Error message
	 * @param array  $context Additional context data
	 */
	public static function error( string $message, array $context = array() ): void {
		self::log( self::ERROR, $message, $context );
	}

	/**
	 * Log a warning message
	 *
	 * Use for issues that don't prevent operation but should be addressed.
	 *
	 * @param string $message Warning message
	 * @param array  $context Additional context data
	 */
	public static function warning( string $message, array $context = array() ): void {
		self::log( self::WARNING, $message, $context );
	}

	/**
	 * Log an info message
	 *
	 * Use for important events (e.g., successful operations).
	 *
	 * @param string $message Info message
	 * @param array  $context Additional context data
	 */
	public static function info( string $message, array $context = array() ): void {
		self::log( self::INFO, $message, $context );
	}

	/**
	 * Log a debug message
	 *
	 * Use for detailed debugging information.
	 *
	 * @param string $message Debug message
	 * @param array  $context Additional context data
	 */
	public static function debug( string $message, array $context = array() ): void {
		// Only log debug messages if WP_DEBUG is enabled
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			self::log( self::DEBUG, $message, $context );
		}
	}

	/**
	 * Core logging method
	 *
	 * @param string $level   Log level (ERROR, WARNING, INFO, DEBUG)
	 * @param string $message Log message
	 * @param array  $context Additional context data
	 */
	private static function log( string $level, string $message, array $context = array() ): void {
		// Only log if WP_DEBUG_LOG is enabled
		if ( ! defined( 'WP_DEBUG_LOG' ) || ! WP_DEBUG_LOG ) {
			return;
		}

		// Format: [FlxWoo] [LEVEL] Message
		$formatted = sprintf( '[FlxWoo] [%s] %s', $level, $message );

		// Add context data if provided
		if ( ! empty( $context ) ) {
			$sanitized_context = self::sanitize_context( $context );
			$formatted        .= ' | Context: ' . wp_json_encode( $sanitized_context );
		}

		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
		error_log( $formatted );
	}

	/**
	 * Sanitize context data to remove PII and sensitive information
	 *
	 * @param array $context Context data
	 * @return array Sanitized context
	 */
	private static function sanitize_context( array $context ): array {
		$sanitized = array();

		foreach ( $context as $key => $value ) {
			$key_lower = strtolower( $key );

			// Completely mask sensitive fields
			if ( self::is_sensitive_field( $key_lower ) ) {
				$sanitized[ $key ] = '[REDACTED]';
				continue;
			}

			// Sanitize email (keep domain)
			if ( str_contains( $key_lower, 'email' ) && is_string( $value ) && str_contains( $value, '@' ) ) {
				$sanitized[ $key ] = self::mask_email( $value );
				continue;
			}

			// Recursively sanitize arrays
			if ( is_array( $value ) ) {
				$sanitized[ $key ] = self::sanitize_context( $value );
				continue;
			}

			// Keep other data
			$sanitized[ $key ] = $value;
		}

		return $sanitized;
	}

	/**
	 * Check if a field name indicates sensitive data
	 *
	 * @param string $field_name Field name to check
	 * @return bool True if sensitive
	 */
	private static function is_sensitive_field( string $field_name ): bool {
		$sensitive_patterns = array(
			'password',
			'passwd',
			'pwd',
			'secret',
			'token',
			'api_key',
			'apikey',
			'auth',
			'authorization',
			'credit_card',
			'card_number',
			'cvv',
			'cvc',
			'ssn',
			'tax_id',
			'payment_method_token',
		);

		foreach ( $sensitive_patterns as $pattern ) {
			if ( str_contains( $field_name, $pattern ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Mask email address (keep domain for debugging)
	 *
	 * @param string $email Email to mask
	 * @return string Masked email (e.g., j***@example.com)
	 */
	private static function mask_email( string $email ): string {
		$parts = explode( '@', $email, 2 );
		if ( count( $parts ) !== 2 ) {
			return '[INVALID_EMAIL]';
		}

		$local  = $parts[0];
		$domain = $parts[1];

		// Keep first character and mask the rest
		$masked_local = substr( $local, 0, 1 ) . str_repeat( '*', max( 1, strlen( $local ) - 1 ) );

		return $masked_local . '@' . $domain;
	}
}
