<?php
namespace FlxWoo\Rest\Endpoints;

if (!defined('ABSPATH')) exit;

/**
 * Site information REST API endpoints
 * Provides site metadata and WooCommerce configuration
 */
class SiteEndpoints {

  /**
   * Cache duration for site info (5 minutes)
   */
  const CACHE_DURATION = 300;

  /**
   * Get basic site information
   * GET /wp-json/flx-woo/v1/site-info
   *
   * @param \WP_REST_Request $request REST request object
   * @return \WP_REST_Response REST response
   */
  public function get_site_info(\WP_REST_Request $request) {
    try {
      // Get basic site information
      $site_info = [
        'home_url' => home_url(),
        'site-name' => get_bloginfo('name'),
        'language' => get_bloginfo('language'),
        'charset' => get_bloginfo('charset'),
        'timezone' => get_option('timezone_string') ?: 'UTC',
        'version' => get_bloginfo('version'),
        'description' => get_bloginfo('description'),
        // WooCommerce currency settings
        'currency' => get_woocommerce_currency(),
        'currency_symbol' => get_woocommerce_currency_symbol(),
        'currency_position' => get_option('woocommerce_currency_pos'),
        'price_thousand_separator' => wc_get_price_thousand_separator(),
        'price_decimal_separator' => wc_get_price_decimal_separator(),
        'price_decimals' => wc_get_price_decimals(),
        // Date/Time formats
        'date_format' => get_option('date_format'),
        'time_format' => get_option('time_format'),
      ];

      $response = new \WP_REST_Response($site_info, 200);

      // Set cache headers
      $response->header('Cache-Control', 'public, max-age=' . self::CACHE_DURATION);
      $response->header('Expires', gmdate('D, d M Y H:i:s', time() + self::CACHE_DURATION) . ' GMT');

      return $response;

    } catch (\Exception $e) {
      // Log full error details for debugging
      Logger::debug('Site info error - ' . $e->getMessage());

      // Only expose detailed error messages in debug mode
      if (defined('WP_DEBUG') && WP_DEBUG) {
        $message = $e->getMessage();
      } else {
        $message = __('An unexpected error occurred', 'flx-woo');
      }

      return new \WP_REST_Response([
        'error' => __('Failed to retrieve site info', 'flx-woo'),
        'message' => $message
      ], 500);
    }
  }
}
