<?php
namespace FlxWoo\Renderer;

if (!defined('ABSPATH')) exit;

use FlxWoo\Data\UserContext;
use FlxWoo\Utils\Logger;

/**
 * Headless Rendering Engine
 *
 * Implements dual-rendering architecture:
 * 1. WordPress generates page (ensures all hooks run)
 * 2. Output buffering intercepts the generated HTML
 * 3. Next.js renderer replaces HTML with headless version
 * 4. Falls back to WordPress HTML on any failure
 *
 * This approach ensures compatibility with WooCommerce hooks/plugins
 * while providing modern React-based frontend rendering.
 */
class HeadlessRender {

  /**
   * Configuration constants
   */
  const ROUTE_PATTERN = '#^/[a-z0-9/_-]+$#i';
  const FALLBACK_REASONS = ['cart-missing', 'checkout-missing', 'order-missing'];
  const NEXTJS_ERROR_INDICATORS = [
    'Application error: a client-side exception has occurred',
    '__next-error-h1',
    'This page could not be found'
  ];
  const REQUIRED_HTML_TAGS = ['DOCTYPE', '<html', '<head', '<body'];

  /**
   * Page type to route mapping
   */
  const ROUTE_MAP = [
    'cart' => '/cart',
    'checkout' => '/checkout',
    'thank-you' => '/thank-you',
  ];

  /**
   * Payment gateway return detection parameters
   * These params indicate a payment gateway redirected back to WooCommerce
   */
  const PAYMENT_GATEWAY_PARAMS = [
    'key',              // WooCommerce order key (standard)
    'token',            // Transbank, PayPal, Stripe
    'TBK_TOKEN',        // Transbank specific
    'payment',          // Generic payment parameter
    'PayerID',          // PayPal
    'payment_intent',   // Stripe
    'redirect_status',  // Stripe
    'session_id',       // Stripe Checkout
    'order_id',         // Many gateways
    'transaction_id',   // Many gateways
    'reference',        // Bank transfers
    'authorization',    // Credit card gateways
  ];

  /**
   * Renderer configuration validated at initialization
   * @var string
   */
  private $renderer_url;
  private $renderer_version;
  private $renderer_timeout;
  private $config_valid = false;

  /**
   * Constructor - validates configuration constants once
   */
  public function __construct() {
    // Validate and cache constants once at initialization
    if (defined('FLX_WOO_RENDERER_URL') &&
        defined('FLX_WOO_RENDERER_VERSION') &&
        defined('FLX_WOO_RENDERER_TIMEOUT')) {

      $url = FLX_WOO_RENDERER_URL;
      $parsed = wp_parse_url($url);

      // Security: Validate HTTPS and proper URL format
      if (!$parsed ||
          !isset($parsed['scheme']) ||
          !isset($parsed['host'])) {
        Logger::error('Invalid FLX_WOO_RENDERER_URL - must be a valid URL with scheme and host', ['context' => 'configuration', 'renderer_url' => $url]);
        return;
      }

      // Security: Enforce HTTPS for production renderer (allow HTTP only for localhost in debug mode)
      $is_localhost = in_array($parsed['host'], ['localhost', '127.0.0.1', '::1'], true) ||
                      (substr($parsed['host'], -6) === '.local'); // Allow .local TLD for development
      $is_debug = defined('WP_DEBUG') && WP_DEBUG;

      if ($parsed['scheme'] !== 'https' && !($is_localhost && $is_debug)) {
        Logger::error('FLX_WOO_RENDERER_URL must use HTTPS (HTTP only allowed for localhost and .local domains in WP_DEBUG mode)', ['context' => 'security_configuration', 'renderer_url' => $url, 'scheme' => $parsed['scheme']]);
        return;
      }

      $this->renderer_url = rtrim($url, '/');  // Remove trailing slash
      $this->renderer_version = FLX_WOO_RENDERER_VERSION;
      $this->renderer_timeout = FLX_WOO_RENDERER_TIMEOUT;
      $this->config_valid = true;
    } else {
      Logger::error('Renderer constants not defined - headless rendering disabled', ['context' => 'configuration_missing']);
    }
  }

  /**
   * Get the page type for the current page
   * Returns page type string (e.g., 'cart', 'checkout', 'thank-you')
   * Returns empty string if page should not be rendered by Next.js
   */
  private function get_page_type() {
    if (function_exists('is_cart') && is_cart()) {
      return 'cart';
    }

    // Check for checkout and thank-you pages
    // Order matters: check thank-you first (more specific)
    if (function_exists('is_checkout') && is_checkout()) {
      // Thank-you / order-received page (specific endpoint)
      if (is_wc_endpoint_url('order-received')) {
        return 'thank-you';
      }
      // Regular checkout page
      return 'checkout';
    }

    // Add more page types here as needed

    // Return empty for pages we don't handle yet
    return '';
  }

  /**
   * Get the route for the current page
   * Returns validated route path (e.g., '/cart', '/checkout')
   * Returns empty string if page should not be rendered by Next.js
   * Routes are pre-validated to contain only safe characters
   */
  private function get_route() {
    $page_type = $this->get_page_type();

    if (empty($page_type)) {
      return '';
    }

    // Get route from mapping
    $route = self::ROUTE_MAP[$page_type] ?? '';

    // Validate route format (must start with / and contain only safe characters)
    // This is defensive validation since routes are hardcoded in ROUTE_MAP
    if (!empty($route) && !preg_match(self::ROUTE_PATTERN, $route)) {
      Logger::debug('Invalid route format: ' . $route, ['route' => $route]);
      return '';
    }

    // Special handling for checkout page: skip rendering if cart is empty
    // This handles payment gateway returns that redirect to /checkout/ without query params
    // after the order is created and cart is emptied (e.g., Transbank Webpay Plus)
    if ($route === '/checkout' && $this->ensure_wc_session_initialized()) {
      if (WC()->cart->is_empty()) {
        // Cart is empty on checkout page - likely a payment gateway return
        // Let WooCommerce handle the redirect to order-received page
        Logger::debug('Checkout page with empty cart - falling back to WordPress (likely payment gateway return)', ['context' => 'payment_gateway_return']);
        return '';
      }
    }

    return $route;
  }

  /**
   * Ensure WooCommerce session and cart are initialized
   * Helper method to avoid code duplication
   *
   * @return bool True if WooCommerce is available and initialized, false otherwise
   */
  private function ensure_wc_session_initialized() {
    if (!function_exists('WC')) {
      return false;
    }

    // Initialize WooCommerce session if needed
    if (is_null(WC()->session)) {
      WC()->initialize_session();
    }

    // Set customer session cookie to load from existing cookie
    WC()->session->set_customer_session_cookie(true);

    // Initialize cart if needed
    if (is_null(WC()->cart)) {
      wc_load_cart();
    }

    // Force cart to load from session
    WC()->cart->get_cart_from_session();

    return true;
  }

  /**
   * Validate HTML response structure
   *
   * @param string $body Response body
   * @return bool True if valid HTML, false otherwise
   */
  private function validate_html_response($body) {
    // Validate response is not empty
    if (empty($body) || trim($body) === '') {
      Logger::error('Empty response from Next.js', ['context' => 'html_validation']);
      return false;
    }

    // Check DOCTYPE is at the beginning (allow some whitespace)
    if (!preg_match('/^\s*<!DOCTYPE/i', $body)) {
      Logger::error('Response missing DOCTYPE declaration at start', ['context' => 'html_validation', 'body_preview' => substr($body, 0, 100)]);
      return false;
    }

    // Check for essential HTML structure
    foreach (self::REQUIRED_HTML_TAGS as $tag) {
      if (stripos($body, $tag) === false) {
        Logger::error("Response missing {$tag} tag", ['context' => 'html_validation', 'missing_tag' => $tag]);
        return false;
      }
    }

    // Check for Next.js error indicators
    foreach (self::NEXTJS_ERROR_INDICATORS as $indicator) {
      if (stripos($body, $indicator) !== false) {
        Logger::error('Next.js returned error page: ' . $indicator, ['context' => 'html_validation', 'error_indicator' => $indicator]);
        return false;
      }
    }

    return true;
  }

  /**
   * Generate deterministic site ID from home URL
   *
   * Site ID is used for site-based rate limiting on Next.js side.
   * Uses SHA-256 hash of home URL for deterministic, unique identifier.
   *
   * @return string 16-character site identifier (hex)
   */
  private function get_site_id() {
    $home_url = home_url();
    // Use first 16 characters of SHA-256 hash for readability
    // Provides 2^64 unique values (sufficient for millions of sites)
    return substr(hash('sha256', $home_url), 0, 16);
  }

  /**
   * Get data to send with route
   *
   * @param string $route The route being rendered
   */
  private function get_data($route) {
    $home_url = esc_url_raw(home_url());

    if (!preg_match('#^[a-z][a-z0-9+.\-]*://#i', $home_url)) {
      $scheme = is_ssl() ? 'https://' : 'http://';
      $home_url = $scheme . ltrim($home_url, '/');
    }

    $data = [
      'home_url' => $home_url
    ];

    // Get user context for the current route
    $user_context = new UserContext();
    $context = $user_context->get_context_for_route($route);

    if (!empty($context)) {
      $data['user_context'] = $context;
    }

    // WooCommerce manages session persistence automatically
    // Previous save_data() call was re-persisting old cart data

    return $data;
  }

  /**
   * Send route to Next.js and get rendered HTML back
   *
   * @param string $route The validated route (must be pre-validated by caller)
   */
  private function fetch_from_nextjs($route) {
    // Check configuration was validated at initialization
    if (!$this->config_valid) {
      return false;
    }

    $url = $this->renderer_url . '/api/' . $this->renderer_version . $route;

    $data = $this->get_data($route);
    $payload = wp_json_encode($data);

    if ($payload === false) {
      Logger::error('Failed to encode renderer payload', ['context' => 'json_encoding', 'route' => $route]);
      return false;
    }

    // Security: Check payload size before sending (1MB max request)
    $payload_size = strlen($payload);
    if ($payload_size > 1048576) {
      Logger::error('Payload too large: ' . $payload_size . ' bytes (max 1MB)', ['context' => 'security_violation', 'payload_size' => $payload_size, 'route' => $route]);
      return false;
    }

    $response = wp_remote_post($url, [
      'headers' => [
        'Content-Type' => 'application/json',
        'X-FlxWoo-Site-ID' => $this->get_site_id(),
      ],
      'body' => $payload,
      'timeout' => $this->renderer_timeout,
      'sslverify' => true,  // Security: Explicit SSL certificate verification
      'data_format' => 'body',
      'limit_response_size' => 5242880,  // Security: 5MB max response size
    ]);

    // Handle errors
    if (is_wp_error($response)) {
      Logger::error('Failed to connect to Next.js: ' . $response->get_error_message(), ['context' => 'nextjs_connection', 'renderer_url' => $this->renderer_url, 'route' => $route]);
      return false;
    }

    // Get status code
    $status_code = wp_remote_retrieve_response_code($response);

    // Get response body
    $body = wp_remote_retrieve_body($response);

    // Handle 503 fallback responses
    if ($status_code === 503) {
      $decoded = json_decode($body, true);

      if (is_array($decoded) && isset($decoded['reason'])) {
        if (in_array($decoded['reason'], self::FALLBACK_REASONS, true)) {
          Logger::debug(sprintf(
            'Next.js fallback (%s); falling back to WooCommerce rendering.',
            $decoded['reason']
          ), ['reason' => $decoded['reason'], 'context' => 'nextjs_fallback']);
          return false;
        }
      }

      Logger::error('Next.js returned fallback status 503 without expected reason.', ['context' => 'protocol_error', 'response_body' => substr($body, 0, 200)]);
      return false;
    }

    // Handle non-200 status codes
    if ($status_code !== 200) {
      Logger::error('Next.js returned status ' . $status_code, ['context' => 'http_error', 'status_code' => $status_code, 'response_preview' => substr($body, 0, 200)]);
      return false;
    }

    // Validate HTML response structure
    if (!$this->validate_html_response($body)) {
      return false;
    }

    return $body;
  }

  /**
   * Main callback for output buffering
   * This receives the WordPress-generated page and can replace it
   *
   * @param string $page WordPress-generated HTML
   * @return string HTML to send to browser (Next.js or WordPress fallback)
   */
  public function retrieve_page($page) {
    // Get the validated route for this page
    $route = $this->get_route();

    // If no route, return original WordPress page
    if (empty($route)) {
      return $page;
    }

    // Fetch rendered HTML from Next.js
    $nextjs_html = $this->fetch_from_nextjs($route);

    // If fetch succeeded, return Next.js HTML
    if ($nextjs_html !== false) {
      return $nextjs_html;
    }

    // Next.js fetch failed - validate WordPress fallback
    // This helps detect if both rendering methods failed
    if (empty($page) || strlen(trim($page)) < 100) {
      Logger::error('CRITICAL - Both Next.js and WordPress rendering failed (page suspiciously small)', ['context' => 'catastrophic_failure', 'page_length' => strlen($page)]);
      // Still return it - let WordPress error handling take over
    }

    // Return original WordPress page as fallback
    return $page;
  }

  /**
   * Start headless rendering
   * Sets up output buffering to intercept WordPress-generated pages
   */
  public function render_headless(): void {
    // Don't render for admin, AJAX, REST, feeds, etc.
    if (is_admin() || wp_doing_ajax() || defined('REST_REQUEST') ||
        (function_exists('wp_is_json_request') && wp_is_json_request()) ||
        is_feed() || is_embed() || wp_doing_cron() || is_robots() || is_trackback()) {
      return;
    }

    // Don't render for POST requests - these are form submissions being processed
    // WooCommerce will handle checkout processing and redirect to thank-you page
    // Example: "Place Order" button POSTs to /checkout, which should be processed
    // by WooCommerce natively, not rendered by Next.js
    if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'POST') {
      return;
    }

    // Don't render for payment gateway returns/callbacks to CHECKOUT page
    // Payment gateways (PayPal, Stripe, Transbank, etc.) redirect back to checkout
    // with query parameters after processing payment on their site
    // WooCommerce needs to process these returns natively before redirecting to thank-you page
    // IMPORTANT: We DO want to render the thank-you/order-received page with Next.js,
    // so only skip rendering if this is a checkout page (not order-received)
    if ($this->is_payment_gateway_return_to_checkout()) {
      return;
    }

    // Prevent caching of cart/checkout pages at ALL levels
    // WordPress caching plugins
    if (!defined('DONOTCACHEPAGE')) {
      // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedConstantFound -- DONOTCACHEPAGE is a WordPress ecosystem standard constant used by caching plugins (W3 Total Cache, WP Super Cache, etc.) to detect pages that should not be cached. Not a plugin-specific constant.
      define('DONOTCACHEPAGE', true);
    }
    nocache_headers();

    // Cloudflare and CDN cache bypass headers
    header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0, private', true);
    header('Pragma: no-cache', true);
    header('Expires: 0', true);
    header('CF-Cache-Status: BYPASS');
    header('CDN-Cache-Control: no-store');
    header('X-Accel-Expires: 0');
    header('X-Cache: BYPASS');
    header('Surrogate-Control: no-store');

    // Start output buffering - retrieve_page will be called when output is flushed
    ob_start([$this, 'retrieve_page']);
  }

  /**
   * Check if current request is a payment gateway return/callback to CHECKOUT page
   *
   * Payment gateways redirect back with various query parameters after processing
   * payment on their external site. WooCommerce must process these returns natively.
   *
   * IMPORTANT: This excludes the order-received/thank-you page because we DO want
   * to render that page with Next.js. Only returns true for checkout page returns.
   *
   * @return bool True if this is a payment gateway return to checkout (not order-received)
   */
  private function is_payment_gateway_return_to_checkout() {
    // Don't skip rendering if this is the order-received/thank-you page
    // We want Next.js to render the thank-you page
    if (function_exists('is_wc_endpoint_url') && is_wc_endpoint_url('order-received')) {
      return false;
    }

    // Check for payment gateway query parameters
    foreach (self::PAYMENT_GATEWAY_PARAMS as $param) {
      // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only detection of payment gateway returns. No nonce needed as we only check parameter existence without processing values or performing privileged operations.
      if (isset($_GET[$param]) && !empty($_GET[$param])) {
        // Payment gateway return detected
        Logger::debug(sprintf(
          'Payment gateway return detected (param: %s) - allowing WordPress to process natively',
          $param
        ), ['param' => $param, 'context' => 'payment_gateway_return']);
        return true;
      }
    }

    return false;
  }
}
