<?php
namespace FlxWoo\Hooks;

if (!defined('ABSPATH')) exit;

/**
 * Stripe Store API compatibility hooks
 * Fixes field name incompatibility between Store API sanitize_key() and Stripe plugin
 *
 * @since 2.0.0
 */
class StripeCompatibilityHooks {

  /**
   * Initialize hooks
   */
  public function init() {
    // Run BEFORE legacy payment processing (priority 999)
    // This allows us to add hyphenated field names that Stripe expects
    \add_action('woocommerce_rest_checkout_process_payment_with_context', [$this, 'add_hyphenated_stripe_fields'], 500, 2);
  }

  /**
   * Add hyphenated versions of Stripe fields to $context->payment_data
   *
   * The Store API's sanitize_key() converts hyphens to underscores,
   * but the Stripe plugin expects field names with hyphens.
   * This creates both versions so Stripe can find them.
   *
   * @param \Automattic\WooCommerce\StoreApi\Payments\PaymentContext $context Payment context
   * @param \Automattic\WooCommerce\StoreApi\Payments\PaymentResult $result Payment result
   */
  public function add_hyphenated_stripe_fields($context, $result) {
    // Only run for Stripe payments
    if ($context->payment_method !== 'stripe') {
      return;
    }

    // Get current payment data
    $payment_data = $context->payment_data;

    // Map underscore field names to hyphenated versions
    $field_mappings = [
      'wc_stripe_payment_method' => 'wc-stripe-payment-method',
      'wc_stripe_selected_upe_payment_type' => 'wc_stripe_selected_upe_payment_type',
      'wc_stripe_is_deferred_intent' => 'wc-stripe-is-deferred-intent',
      'wc_stripe_new_payment_method' => 'wc-stripe-new-payment-method',
    ];

    // Add hyphenated copies to payment_data
    // When Legacy handler sets $_POST = $context->payment_data, both versions will be available
    foreach ($field_mappings as $underscore_name => $hyphen_name) {
      if (isset($payment_data[$underscore_name]) && !isset($payment_data[$hyphen_name])) {
        $payment_data[$hyphen_name] = $payment_data[$underscore_name];
      }
    }

    // Update context with modified payment data
    $context->set_payment_data($payment_data);
  }
}
