<?php
namespace FlxWoo\Hooks;

if (!defined('ABSPATH')) exit;

use FlxWoo\Rest\RestEndpoints;
use FlxWoo\Utils\Logger;

/**
 * REST API Hooks
 *
 * Handles REST API initialization and WooCommerce session management for cart endpoints.
 *
 * NOTE: User authentication for cart operations happens directly in CartEndpoints
 * via authenticate_user_from_cookie() to ensure proper timing in the request lifecycle.
 */
class RestHooks {
  /**
   * Initialize REST API hooks
   */
  public function init() {
    // Initialize WooCommerce session early for our cart endpoints
    add_action('parse_request', [$this, 'maybe_init_wc_session'], 5);

    // Register our custom REST API routes
    add_action('rest_api_init', [$this, 'register_rest_routes']);
  }

  /**
   * Initialize WooCommerce session for our REST API cart and checkout endpoints
   *
   * NOTE: This is primarily for backward compatibility with deprecated custom endpoints
   * (/flx-woo/v1/cart/*, /flx-woo/v1/checkout). WooCommerce Store API, which is now
   * the primary integration method, has its own session management.
   *
   * This session initialization can be removed when custom endpoints are fully deprecated.
   *
   * WooCommerce doesn't initialize sessions for REST API requests by default.
   * This ensures cart and checkout operations work properly when called via our
   * custom REST API endpoints.
   *
   * Runs on 'parse_request' action (priority 5) to initialize early in request lifecycle.
   */
  public function maybe_init_wc_session() {
    // Early exit if not a cart or checkout endpoint request
    if (!$this->is_cart_or_checkout_endpoint_request()) {
      return;
    }

    // Ensure WooCommerce is loaded
    if (!function_exists('WC')) {
      return;
    }

    // Initialize WooCommerce session
    if (is_null(WC()->session)) {
      WC()->initialize_session();
    }

    // Set customer session cookie to load from existing cookie
    WC()->session->set_customer_session_cookie(true);

    // Initialize cart
    if (is_null(WC()->cart)) {
      wc_load_cart();
    }

    // Force cart to load from session
    WC()->cart->get_cart_from_session();
  }

  /**
   * Register custom REST API routes
   *
   * Runs on 'rest_api_init' action.
   */
  public function register_rest_routes() {
    try {
      (new RestEndpoints())->register_routes();
    } catch (\Exception $e) {
      Logger::debug('REST route registration error - ' . $e->getMessage());
      // Let WordPress continue without REST routes
    }
  }

  /**
   * Check if current request is for a cart or checkout endpoint
   *
   * Helper method to detect cart and checkout endpoint requests.
   * Used during parse_request when REST_REQUEST constant is not yet defined.
   *
   * @return bool True if cart or checkout endpoint request, false otherwise
   */
  private function is_cart_or_checkout_endpoint_request() {
    if (!isset($_SERVER['REQUEST_URI'])) {
      return false;
    }

    $request_uri = esc_url_raw(wp_unslash($_SERVER['REQUEST_URI']));
    $namespace = FLX_WOO_REST_NAMESPACE . '/' . FLX_WOO_REST_VERSION;

    // Check if URI matches our cart or checkout endpoint pattern
    return strpos($request_uri, "/wp-json/{$namespace}/cart/") !== false ||
           strpos($request_uri, "/wp-json/{$namespace}/checkout") !== false;
  }
}
