<?php
namespace FlxWoo\Hooks;

if (!defined('ABSPATH')) exit;

use FlxWoo\Renderer\HeadlessRender;
use FlxWoo\Data\UserContext;
use FlxWoo\Utils\Logger;

/**
 * Render Hooks - Optimized for Performance (Phase 1)
 *
 * PERFORMANCE OPTIMIZATION (v2.0.0 - October 2025):
 * - Uses early template_redirect to bypass WordPress theme loading
 * - Renders headless immediately without output buffering
 * - Eliminates double-rendering overhead (30-40% performance gain)
 * - Falls back to WordPress rendering on any error
 *
 * Key Changes from Original:
 * 1. Intercepts at template_redirect priority 1 (before theme loads)
 * 2. Skips WordPress template engine entirely
 * 3. Outputs Next.js HTML immediately and exits
 * 4. No output buffering needed
 */
class RenderHooks {

  /**
   * Initialize hooks
   *
   * Priority 5 ensures WooCommerce conditional tags are available but we still
   * intercept before WordPress loads theme templates (which happens at priority 10+)
   */
  public function init() {
    add_action('template_redirect', [$this, 'maybe_bypass_wordpress_template'], 5);

    // Automatically exclude WooCommerce pages from caching plugins
    // This prevents issues where cached pages show stale cart/checkout data
    add_filter('wpsc_exclude_pages', [$this, 'exclude_pages_from_cache'], 10, 1);
    add_filter('superpagecache_exclude_uris', [$this, 'exclude_pages_from_cache'], 10, 1);
    add_filter('litespeed_cache_excludes', [$this, 'exclude_pages_from_cache'], 10, 1);
    add_filter('rocket_cache_reject_uri', [$this, 'exclude_pages_from_cache'], 10, 1);
    add_filter('w3tc_cache_reject_uri', [$this, 'exclude_pages_from_cache'], 10, 1);
  }

  /**
   * Bypass WordPress template engine for headless pages
   *
   * This method runs at priority 5 on template_redirect, which is AFTER
   * WooCommerce sets up query variables but BEFORE WordPress loads theme
   * templates (priority 10+). This saves significant overhead by skipping
   * theme loading entirely.
   *
   * @return void Exits immediately if rendering succeeds, otherwise returns to allow WordPress to continue
   */
  public function maybe_bypass_wordpress_template() {
    // Don't render for admin, AJAX, REST, feeds, etc.
    if (is_admin() || wp_doing_ajax() || defined('REST_REQUEST') ||
        (function_exists('wp_is_json_request') && wp_is_json_request()) ||
        is_feed() || is_embed() || wp_doing_cron() || is_robots() || is_trackback()) {
      return;
    }

    // Don't render for POST requests - these are form submissions
    if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'POST') {
      return;
    }

    // Don't render for payment gateway returns to checkout page
    if ($this->is_payment_gateway_return_to_checkout()) {
      return;
    }

    // PERFORMANCE TESTING BYPASS: Allow testing original WooCommerce vs FlxWoo
    // Used for Lighthouse comparison and A/B testing
    if ($this->should_bypass_flxwoo()) {
      return; // Use original WooCommerce rendering
    }

    // Try to render headless immediately
    $rendered = $this->render_headless_immediately();

    if ($rendered) {
      // Success! HTML output and WordPress execution stopped
      // This is the optimal path - no theme loading overhead
      exit;
    }

    // Rendering failed or page not eligible - let WordPress continue normally
    // WordPress will load theme and render the page
  }

  /**
   * Render headless page immediately without WordPress template engine
   *
   * This is the core optimization: we gather data and fetch Next.js HTML
   * WITHOUT allowing WordPress to load theme templates.
   *
   * @return bool True if rendering succeeded, false otherwise
   */
  private function render_headless_immediately() {
    try {
      $headless_render = new HeadlessRender();

      // Get the route for this page
      $route = $this->get_route_for_current_page($headless_render);

      if (empty($route)) {
        // Not a headless page - let WordPress handle it
        return false;
      }

      // Initialize WooCommerce session if needed
      $this->ensure_wc_session_initialized();

      // Get context data for the route
      $data = $this->get_data_for_route($route);

      // Trigger pre-render actions (analytics tracking, etc.)
      // This allows plugins to act on the data before HTML is rendered and exit is called
      do_action('flx_woo_before_render', $route, $data);

      // Fetch HTML from Next.js
      $html = $this->fetch_nextjs_html($route, $data);

      if ($html === false) {
        // Next.js fetch failed - let WordPress handle it
        $this->track_render_stats(false); // Track failure
        return false;
      }

      // Output HTML with proper headers
      $this->output_html($html);

      return true;

    } catch (\Exception $e) {
      Logger::error('Early render error: ' . $e->getMessage(), ['file' => $e->getFile(), 'line' => $e->getLine()]);
      return false;
    }
  }

  /**
   * Get route for current page using HeadlessRender logic
   *
   * @param HeadlessRender $headless_render Instance to use for route detection
   * @return string Route string (e.g., '/cart') or empty string
   */
  private function get_route_for_current_page($headless_render) {
    // Use reflection to call private method get_route()
    // This reuses the existing logic from HeadlessRender
    $reflection = new \ReflectionClass($headless_render);
    $method = $reflection->getMethod('get_route');
    $method->setAccessible(true);
    return $method->invoke($headless_render);
  }

  /**
   * Get data to send to Next.js for the route
   *
   * @param string $route The route being rendered
   * @return array Data payload
   */
  private function get_data_for_route($route) {
    $home_url = esc_url_raw(home_url());

    if (!preg_match('#^[a-z][a-z0-9+.\-]*://#i', $home_url)) {
      $scheme = is_ssl() ? 'https://' : 'http://';
      $home_url = $scheme . ltrim($home_url, '/');
    }

    $data = [
      'home_url' => $home_url
    ];

    // Get user context for the current route
    $user_context = new UserContext();
    $context = $user_context->get_context_for_route($route);

    if (!empty($context)) {
      $data['user_context'] = $context;
    }

    // WooCommerce manages session persistence automatically
    // Previous save_data() call was re-persisting old cart data

    return $data;
  }

  /**
   * Fetch HTML from Next.js
   *
   * @param string $route The route to render
   * @param array $data Data payload
   * @return string|false HTML string or false on failure
   */
  private function fetch_nextjs_html($route, $data) {
    // Check configuration
    if (!defined('FLX_WOO_RENDERER_URL') ||
        !defined('FLX_WOO_RENDERER_VERSION') ||
        !defined('FLX_WOO_RENDERER_TIMEOUT')) {
      Logger::error('Renderer constants not defined');
      return false;
    }

    $url = FLX_WOO_RENDERER_URL . '/api/' . FLX_WOO_RENDERER_VERSION . $route;
    $payload = wp_json_encode($data);

    if ($payload === false) {
      Logger::error('Failed to encode renderer payload');
      return false;
    }

    $response = wp_remote_post($url, [
      'headers' => [
        'Content-Type' => 'application/json',
      ],
      'body' => $payload,
      'timeout' => FLX_WOO_RENDERER_TIMEOUT,
    ]);

    // Handle errors
    if (is_wp_error($response)) {
      Logger::error('Failed to connect to Next.js: ' . $response->get_error_message(), ['url' => $url]);
      return false;
    }

    $status_code = wp_remote_retrieve_response_code($response);
    $body = wp_remote_retrieve_body($response);

    // Handle 503 fallback
    if ($status_code === 503) {
      $decoded = json_decode($body, true);
      if (is_array($decoded) && isset($decoded['reason'])) {
        Logger::debug('Next.js fallback (' . $decoded['reason'] . ')', ['reason' => $decoded['reason']]);
      }
      return false;
    }

    // Handle non-200 status
    if ($status_code !== 200) {
      Logger::error('Next.js returned status ' . $status_code);
      return false;
    }

    // Basic validation
    if (empty($body) || !preg_match('/^\s*<!DOCTYPE/i', $body)) {
      Logger::error('Invalid HTML response from Next.js');
      return false;
    }

    return $body;
  }

  /**
   * Output HTML with proper headers
   *
   * CRITICAL: Cart/checkout pages must NEVER be cached at any level:
   * - Browser cache
   * - WordPress cache plugins
   * - CDN cache (Cloudflare, etc.)
   * - Reverse proxy cache (Varnish, Nginx, etc.)
   *
   * SECURITY NOTE:
   * The $html variable contains pre-rendered, complete HTML from the Next.js
   * rendering service. All XSS escaping is handled by Next.js templates using
   * escapeHtml() before transmission. This is architecturally equivalent to
   * outputting content from WordPress's the_content() filter - already processed
   * and safe HTML that should NOT be double-escaped.
   *
   * @param string $html Pre-sanitized HTML from Next.js renderer
   */
  private function output_html($html) {
    // Tell WordPress caching plugins NOT to cache this page
    if (!defined('DONOTCACHEPAGE')) {
      // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedConstantFound -- DONOTCACHEPAGE is a WordPress ecosystem standard constant used by caching plugins (W3 Total Cache, WP Super Cache, etc.) to detect pages that should not be cached. Not a plugin-specific constant.
      define('DONOTCACHEPAGE', true);
    }

    // Set WordPress nocache headers (Cache-Control, Pragma, Expires)
    nocache_headers();

    // Set content type header
    header('Content-Type: text/html; charset=UTF-8');

    // Cloudflare-specific cache bypass headers
    // cf-cache-status should show "BYPASS" instead of "HIT"
    header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0, private', true);
    header('Pragma: no-cache', true);
    header('Expires: 0', true);

    // Tell Cloudflare to bypass cache for this page
    header('CF-Cache-Status: BYPASS');
    header('CDN-Cache-Control: no-store');

    // Additional cache-prevention headers for other CDNs/proxies
    header('X-Accel-Expires: 0'); // Nginx
    header('X-Cache: BYPASS'); // Varnish/generic CDN marker
    header('Surrogate-Control: no-store'); // Akamai/Fastly

    // Track successful render timestamp for dashboard display
    // This records when actual user page renders happen (not just performance tests)
    update_option('flx_woo_last_render_time', time(), false);

    // Track render statistics for dashboard counter
    $this->track_render_stats(true);

    // Fire action hook for performance monitoring (v2.2.0+)
    // Allows PerformanceTestScheduler to trigger automatic tests via request-based fallback
    do_action('flx_woo_after_render');

    // Output pre-sanitized HTML from Next.js renderer
    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- HTML is pre-rendered and sanitized by Next.js templates using escapeHtml(). This is a complete HTML document from a trusted rendering service, similar to WordPress outputting the_content() filter results.
    echo $html;
  }

  /**
   * Ensure WooCommerce session and cart are initialized
   *
   * @return bool True if WooCommerce is available and initialized
   */
  private function ensure_wc_session_initialized() {
    if (!function_exists('WC')) {
      return false;
    }

    // Initialize WooCommerce session if needed
    if (is_null(WC()->session)) {
      WC()->initialize_session();
    }

    // Set customer session cookie
    WC()->session->set_customer_session_cookie(true);

    // Initialize cart if needed
    if (is_null(WC()->cart)) {
      wc_load_cart();
    }

    // Force cart to load from session
    WC()->cart->get_cart_from_session();

    return true;
  }

  /**
   * Check if current request is a payment gateway return to checkout page
   *
   * @return bool True if payment gateway return to checkout
   */
  private function is_payment_gateway_return_to_checkout() {
    // Don't skip rendering if this is the order-received/thank-you page
    if (function_exists('is_wc_endpoint_url') && is_wc_endpoint_url('order-received')) {
      return false;
    }

    // Payment gateway parameters from HeadlessRender
    $payment_gateway_params = [
      'key', 'token', 'TBK_TOKEN', 'payment', 'PayerID',
      'payment_intent', 'redirect_status', 'session_id',
      'order_id', 'transaction_id', 'reference', 'authorization',
    ];

    // Check for payment gateway query parameters
    foreach ($payment_gateway_params as $param) {
      // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only detection of payment gateway returns. No nonce needed as we only check parameter existence without processing values or performing privileged operations.
      if (isset($_GET[$param]) && !empty($_GET[$param])) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
          Logger::debug(sprintf('Payment gateway return detected (param: %s) - allowing WordPress to process', $param
          ));
        }
        return true;
      }
    }

    return false;
  }

  /**
   * Exclude WooCommerce dynamic pages from caching plugins
   *
   * Automatically tells caching plugins (Super Page Cache, WP Rocket, LiteSpeed, etc.)
   * to exclude cart/checkout pages from caching. This prevents stale data issues.
   *
   * Works with multiple caching plugins via their filter hooks:
   * - Super Page Cache: wpsc_exclude_pages, superpagecache_exclude_uris
   * - WP Rocket: rocket_cache_reject_uri
   * - W3 Total Cache: w3tc_cache_reject_uri
   * - LiteSpeed Cache: litespeed_cache_excludes
   *
   * @param array $excluded_pages Existing excluded pages from caching plugin
   * @return array Updated list with WooCommerce pages added
   */
  public function exclude_pages_from_cache($excluded_pages = []) {
    // Ensure we have an array to work with
    if (!is_array($excluded_pages)) {
      $excluded_pages = [];
    }

    // WooCommerce pages that should NEVER be cached (user-specific, dynamic content)
    $woo_pages = [
      '/cart',
      '/cart/',
      '/checkout',
      '/checkout/',
      '/checkout/order-received',
      '/checkout/order-received/*',
      '/my-account',
      '/my-account/*',
    ];

    // Merge with existing exclusions (avoid duplicates)
    return array_unique(array_merge($excluded_pages, $woo_pages));
  }

  /**
   * Check if FlxWoo rendering should be bypassed for performance testing
   *
   * Allows Lighthouse tests and A/B testing to compare original WooCommerce
   * rendering against FlxWoo rendering.
   *
   * Security: Only allows bypass for:
   * - Users with manage_woocommerce capability (admins)
   * - Specific IP addresses (localhost, CI/CD)
   * - Lighthouse/PageSpeed Insights user agents (for performance testing)
   * - When explicitly enabled in settings
   *
   * @return bool True if FlxWoo should be bypassed
   */
  private function should_bypass_flxwoo(): bool {
    // Check for bypass query parameter
    if (isset($_GET['flxwoo_bypass']) && $_GET['flxwoo_bypass'] === '1') {
      // Security: Only allow bypass for authorized users/IPs/user agents
      $allow_bypass = false;

      // Allow for admin users
      if (current_user_can('manage_woocommerce')) {
        $allow_bypass = true;
      }

      // Allow for localhost/development
      if (isset($_SERVER['REMOTE_ADDR']) &&
          in_array($_SERVER['REMOTE_ADDR'], ['127.0.0.1', '::1'])) {
        $allow_bypass = true;
      }

      // Allow for Lighthouse/PageSpeed Insights user agents
      if (isset($_SERVER['HTTP_USER_AGENT'])) {
        $user_agent = strtolower($_SERVER['HTTP_USER_AGENT']);
        $lighthouse_agents = [
          'lighthouse',           // Google Lighthouse
          'speed insights',       // PageSpeed Insights
          'chrome-lighthouse',    // Chrome Lighthouse
          'pagespeed',           // PageSpeed
          'gtmetrix',            // GTmetrix
          'webpagetest',         // WebPageTest
        ];

        foreach ($lighthouse_agents as $agent) {
          if (strpos($user_agent, $agent) !== false) {
            $allow_bypass = true;
            break;
          }
        }
      }

      // Allow if explicitly enabled in settings (for CI/CD, monitoring)
      if (get_option('flx_woo_allow_bypass', false)) {
        $allow_bypass = true;
      }

      if ($allow_bypass) {
        Logger::debug('FlxWoo rendering bypassed for performance testing', [
          'url' => $_SERVER['REQUEST_URI'] ?? '',
          'user' => is_user_logged_in() ? wp_get_current_user()->user_login : 'guest',
          'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
        ]);
        return true;
      }

      // Log blocked bypass attempts for security monitoring
      Logger::warning('FlxWoo bypass attempt blocked - unauthorized', [
        'url' => $_SERVER['REQUEST_URI'] ?? '',
        'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
      ]);
    }

    return false;
  }

  /**
   * Track render statistics for dashboard display
   *
   * Tracks successful and failed renders in a rolling 24-hour window.
   * This is separate from performance tests and tracks actual user page loads.
   *
   * @param bool $success Whether the render was successful
   * @return void
   */
  private function track_render_stats($success) {
    // Get current statistics (initialize if doesn't exist)
    $stats = get_option('flx_woo_render_stats_24h', [
      'total' => 0,
      'successful' => 0,
      'failed' => 0,
      'last_reset' => time(),
    ]);

    // Reset stats if more than 24 hours old
    if (time() - ($stats['last_reset'] ?? 0) > 86400) {
      $stats = [
        'total' => 0,
        'successful' => 0,
        'failed' => 0,
        'last_reset' => time(),
      ];
    }

    // Increment counters
    $stats['total']++;
    if ($success) {
      $stats['successful']++;
    } else {
      $stats['failed']++;
    }

    // Save updated statistics (no autoload for performance)
    update_option('flx_woo_render_stats_24h', $stats, false);
  }

  /**
   * LEGACY METHOD: Kept for backward compatibility
   *
   * This method is no longer called with Phase 1 optimization enabled,
   * but kept in case we need to rollback to old architecture.
   */
  public function render_headless_page() {
    try {
      (new HeadlessRender())->render_headless();
    } catch (\Exception $e) {
      Logger::error('Render error: ' . $e->getMessage(), ['file' => $e->getFile(), 'line' => $e->getLine()]);
      // Let WordPress continue with normal rendering
    }
  }
}
