<?php
/**
 * Rate Limit Hooks
 *
 * Integrates rate limiting with WordPress REST API.
 * Applies rate limiting to FlxWoo REST endpoints.
 *
 * @package FlxWoo\Hooks
 * @since 2.1.0
 */

namespace FlxWoo\Hooks;

use FlxWoo\Utils\RateLimiter;

if (!defined('ABSPATH')) {
  exit;
}

class RateLimitHooks {
  /**
   * Register hooks
   */
  public function register(): void {
    // Apply rate limiting to REST API requests
    add_filter('rest_pre_dispatch', [$this, 'check_rate_limit'], 10, 3);
  }

  /**
   * Check rate limit before dispatching REST request
   *
   * Applies rate limiting to FlxWoo endpoints only.
   *
   * @param mixed $result Response to return, or null to continue
   * @param \WP_REST_Server $server REST server instance
   * @param \WP_REST_Request $request Request object
   * @return mixed Response or WP_Error if rate limited
   */
  public function check_rate_limit($result, $server, $request) {
    // Only apply rate limiting to FlxWoo endpoints
    $route = $request->get_route();

    if (!$this->is_flxwoo_endpoint($route)) {
      return $result;
    }

    // Determine rate limit identifier based on route
    $identifier = $this->get_rate_limit_identifier($route);

    if (!$identifier) {
      // No rate limit configured for this route
      return $result;
    }

    // Check rate limit
    $rate_limit_result = RateLimiter::check_rate_limit($identifier);

    if (!$rate_limit_result['allowed']) {
      // Rate limit exceeded - return 429 error
      $error = RateLimiter::create_rate_limit_error($rate_limit_result);

      // Add Retry-After header
      $error_data = $error->get_error_data();
      if (isset($error_data['retry_after']) && !headers_sent()) {
        header('Retry-After: ' . $error_data['retry_after']);
      }

      // Add rate limit headers
      if (isset($error_data['headers']) && !headers_sent()) {
        foreach ($error_data['headers'] as $header_name => $header_value) {
          header($header_name . ': ' . $header_value);
        }
      }

      return $error;
    }

    // Rate limit check passed - add headers to response
    add_filter('rest_post_dispatch', function ($response) use ($rate_limit_result) {
      return $this->add_rate_limit_headers($response, $rate_limit_result);
    }, 10, 1);

    return $result;
  }

  /**
   * Check if route is a FlxWoo endpoint
   *
   * @param string $route REST API route
   * @return bool True if FlxWoo endpoint
   */
  private function is_flxwoo_endpoint(string $route): bool {
    return strpos($route, '/flx-woo/') === 0;
  }

  /**
   * Get rate limit identifier for route
   *
   * @param string $route REST API route
   * @return string|null Rate limit identifier or null if not configured
   */
  private function get_rate_limit_identifier(string $route): ?string {
    // Map routes to rate limit identifiers
    if (strpos($route, '/flx-woo/v1/site-info') === 0) {
      return 'site-info';
    }

    if (strpos($route, '/flx-woo/v1/render/') === 0) {
      return 'render';
    }

    return null;
  }

  /**
   * Add rate limit headers to REST response
   *
   * @param \WP_REST_Response $response REST response
   * @param array $rate_limit_result Rate limit result
   * @return \WP_REST_Response Modified response
   */
  private function add_rate_limit_headers($response, array $rate_limit_result) {
    if (!($response instanceof \WP_REST_Response)) {
      return $response;
    }

    $headers = RateLimiter::get_rate_limit_headers($rate_limit_result);

    foreach ($headers as $header_name => $header_value) {
      $response->header($header_name, $header_value);
    }

    return $response;
  }
}
