<?php
namespace FlxWoo\Hooks;

use FlxWoo\Cors\CorsHandler;

if (!defined('ABSPATH')) exit;

/**
 * Register CORS hooks for REST API
 *
 * This class only handles WordPress hook registration.
 * CORS logic is in FlxWoo\Cors\CorsHandler.
 */
class CorsHooks {

  /**
   * @var CorsHandler CORS handler instance
   */
  private $handler;

  /**
   * Constructor
   */
  public function __construct() {
    $this->handler = new CorsHandler();
  }

  /**
   * Initialize CORS hooks
   */
  public function init() {
    // Add CORS headers to REST API responses
    add_filter('rest_pre_serve_request', [$this, 'add_cors_headers'], 10, 4);

    // Handle preflight OPTIONS requests
    add_action('rest_api_init', [$this, 'handle_preflight_request']);
  }

  /**
   * Add CORS headers to REST API responses
   *
   * @param bool $served Whether the request has already been served
   * @param mixed $result Result to send to the client
   * @param WP_REST_Request $request Request object
   * @param WP_REST_Server $server Server instance
   * @return bool
   */
  public function add_cors_headers($served, $result, $request, $server) {
    $allowed_origin = $this->handler->get_allowed_origin();

    if ($allowed_origin) {
      $this->handler->send_cors_headers($allowed_origin);
    }

    return $served;
  }

  /**
   * Handle preflight OPTIONS requests
   */
  public function handle_preflight_request() {
    if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
      $allowed_origin = $this->handler->get_allowed_origin();

      if ($allowed_origin) {
        $this->handler->send_cors_headers($allowed_origin);
        header('Content-Length: 0');
        header('Content-Type: text/plain');
        status_header(204);
        exit;
      }
    }
  }
}
