<?php
namespace FlxWoo\Hooks;

if (!defined('ABSPATH')) exit;

/**
 * Plugin compatibility hooks
 * Handles conflicts between payment gateway plugins
 *
 * @since 2.1.0
 */
class CompatibilityHooks {

  /**
   * Initialize hooks
   */
  public function init() {
    // Fix Stripe + Transbank compatibility issue
    // Stripe expects all payment tokens to implement is_equal_payment_method()
    // but Transbank's tokens don't have this method, causing fatal errors

    // Check if both Stripe and Transbank are active
    if ($this->has_plugin_conflict()) {
      // NUCLEAR OPTION: Disable saved payment methods for Stripe when Transbank is active
      // This prevents Stripe from trying to access incompatible Transbank tokens
      add_filter('wc_stripe_upe_display_save_payment_method_checkbox', '__return_false', 1);
      add_filter('wc_stripe_display_save_payment_method_checkbox', '__return_false', 1);

      // Filter out incompatible tokens at HIGHEST priority to ensure it runs FIRST
      // This prevents fatal errors during Store API checkout
      add_filter('woocommerce_get_customer_payment_tokens', [$this, 'filter_incompatible_tokens'], PHP_INT_MIN, 3);
      add_filter('woocommerce_payment_tokens_get_customer_tokens', [$this, 'filter_incompatible_tokens'], PHP_INT_MIN, 3);

      // Also filter at the WC_Payment_Tokens level (used during Store API checkout)
      add_filter('woocommerce_payment_tokens_get_tokens', [$this, 'filter_incompatible_tokens'], PHP_INT_MIN, 2);

      Logger::debug('Applied Stripe + Transbank compatibility workarounds at highest priority');
    }
  }

  /**
   * Check if there's a plugin conflict (both Stripe and Transbank active)
   *
   * @return bool True if conflict exists
   */
  private function has_plugin_conflict() {
    // Check if both Stripe and Transbank plugins are active
    $stripe_active = class_exists('WC_Stripe_UPE_Payment_Gateway') || class_exists('WC_Gateway_Stripe');
    $transbank_active = class_exists('Transbank\\WooCommerce\\WebpayRest\\Tokenization\\WC_Payment_Token_Oneclick');

    return $stripe_active && $transbank_active;
  }

  /**
   * Filter out payment tokens that don't implement required methods
   *
   * This prevents Stripe from crashing when encountering Transbank tokens
   * that don't implement is_equal_payment_method()
   *
   * @param array $tokens Payment tokens
   * @param int $customer_id Customer ID (optional)
   * @param string $gateway_id Gateway ID (optional)
   * @return array Filtered tokens
   */
  public function filter_incompatible_tokens($tokens, $customer_id = null, $gateway_id = null) {
    if (empty($tokens) || !is_array($tokens)) {
      return $tokens;
    }

    // Filter out tokens that don't have required methods
    $filtered_tokens = array_filter($tokens, function($token) {
      // Keep only tokens that implement is_equal_payment_method()
      // This prevents Stripe from crashing when encountering Transbank tokens
      return method_exists($token, 'is_equal_payment_method');
    });

    // If we filtered out tokens, log it for debugging
    if (count($filtered_tokens) !== count($tokens) && defined('WP_DEBUG') && WP_DEBUG) {
      $removed_count = count($tokens) - count($filtered_tokens);
      $customer_info = $customer_id ? " for customer {$customer_id}" : "";
      Logger::debug("Filtered out {$removed_count} incompatible payment token(s){$customer_info}");
    }

    return $filtered_tokens;
  }
}
