<?php
/**
 * Analytics Hooks
 *
 * Integrates analytics tracking into WooCommerce checkout flow.
 * Hooks into WooCommerce events to track checkout started/completed/abandoned.
 *
 * @package FlxWoo
 * @version 2.3.0
 */

namespace FlxWoo\Hooks;

use FlxWoo\Analytics\EventTracker;

/**
 * AnalyticsHooks class
 *
 * Registers WordPress/WooCommerce hooks for analytics tracking.
 */
class AnalyticsHooks {

	/**
	 * Event tracker instance
	 *
	 * @var EventTracker
	 */
	private $tracker;

	/**
	 * Feature flag: analytics enabled
	 *
	 * @var bool
	 */
	private $enabled;

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->tracker = new EventTracker();
		$this->enabled = $this->is_analytics_enabled();
	}

	/**
	 * Register hooks
	 */
	public function register(): void {
		if ( ! $this->enabled ) {
			return;
		}

		// Checkout started - track when user lands on checkout page
		add_action( 'woocommerce_before_checkout_form', array( $this, 'track_checkout_started' ), 10 );

		// Checkout completed - track when order is successfully placed
		add_action( 'woocommerce_thankyou', array( $this, 'track_checkout_completed' ), 10, 1 );

		// FlxWoo pre-render hook - track checkout completion before early exit
		// This fires before FlxWoo exits, allowing analytics to work with early exit optimization
		add_action( 'flx_woo_before_render', array( $this, 'handle_pre_render_analytics' ), 10, 2 );

		// Checkout abandoned - track via AJAX (client-side detection)
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_abandonment_tracker' ) );
		add_action( 'wp_ajax_flxwoo_track_abandonment', array( $this, 'ajax_track_abandonment' ) );
		add_action( 'wp_ajax_nopriv_flxwoo_track_abandonment', array( $this, 'ajax_track_abandonment' ) );

		// Track cart abandonment via session expiry (optional)
		add_action( 'woocommerce_cart_emptied', array( $this, 'maybe_track_cart_abandonment' ) );
	}

	/**
	 * Track checkout started
	 *
	 * Triggered when user lands on checkout page.
	 */
	public function track_checkout_started(): void {
		// Check if we've already tracked this session
		if ( WC()->session && WC()->session->get( 'flxwoo_checkout_started' ) ) {
			return;
		}

		// Track event
		$this->tracker->track_checkout_started(
			array(
				'source'      => 'checkout_page',
				'device_type' => wp_is_mobile() ? 'mobile' : 'desktop',
			)
		);

		// Mark as tracked in session
		if ( WC()->session ) {
			WC()->session->set( 'flxwoo_checkout_started', time() );
		}
	}

	/**
	 * Track checkout completed
	 *
	 * Triggered when order is successfully placed.
	 *
	 * @param int $order_id WooCommerce order ID.
	 */
	public function track_checkout_completed( int $order_id ): void {
		if ( ! $order_id ) {
			return;
		}

		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return;
		}

		// Only track once per order
		if ( $order->get_meta( '_flxwoo_tracked' ) ) {
			return;
		}

		// Track event
		$this->tracker->track_checkout_completed(
			$order_id,
			array(
				'source'      => 'thank_you_page',
				'device_type' => wp_is_mobile() ? 'mobile' : 'desktop',
			)
		);

		// Mark order as tracked
		$order->update_meta_data( '_flxwoo_tracked', time() );
		$order->save();

		// Clear checkout started flag
		if ( WC()->session ) {
			WC()->session->set( 'flxwoo_checkout_started', null );
		}
	}

	/**
	 * Handle pre-render analytics tracking
	 *
	 * Triggered by FlxWoo before rendering and exiting. This allows analytics
	 * to track orders even when FlxWoo exits before WordPress loads templates.
	 *
	 * @param string $route FlxWoo route (e.g., '/cart', '/checkout', '/thank-you').
	 * @param array  $data  Context data for the route.
	 */
	public function handle_pre_render_analytics( string $route, array $data ): void {
		// Only track for thank-you page with valid order data
		if ( $route !== '/thank-you' ) {
			return;
		}

		// Extract order ID from context data
		// Structure: $data['user_context']['order']['order_id']
		// Note: get_data_for_route() wraps UserContext data in 'user_context' key
		if ( ! isset( $data['user_context']['order']['order_id'] ) ) {
			return;
		}

		$order_id = intval( $data['user_context']['order']['order_id'] );

		if ( $order_id <= 0 ) {
			return;
		}

		// Track checkout completion
		// Note: track_checkout_completed() has built-in duplicate prevention
		$this->track_checkout_completed( $order_id );
	}

	/**
	 * Enqueue abandonment tracker script
	 *
	 * Tracks when user leaves checkout page without completing.
	 */
	public function enqueue_abandonment_tracker(): void {
		if ( ! is_checkout() || is_order_received_page() ) {
			return;
		}

		// Only track if checkout was started
		if ( ! WC()->session || ! WC()->session->get( 'flxwoo_checkout_started' ) ) {
			return;
		}

		wp_enqueue_script(
			'flxwoo-abandonment-tracker',
			plugins_url( 'assets/js/abandonment-tracker.js', dirname( __DIR__ ) ),
			array( 'jquery' ),
			'2.3.0',
			true
		);

		wp_localize_script(
			'flxwoo-abandonment-tracker',
			'flxwooAbandonmentTracker',
			array(
				'ajaxUrl' => admin_url( 'admin-ajax.php' ),
				'nonce'   => wp_create_nonce( 'flxwoo_track_abandonment' ),
			)
		);
	}

	/**
	 * AJAX handler for abandonment tracking
	 *
	 * Called by client-side script when user leaves checkout.
	 */
	public function ajax_track_abandonment(): void {
		// Verify nonce
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'flxwoo_track_abandonment' ) ) {
			wp_send_json_error( 'Invalid nonce' );
			return;
		}

		// Get abandonment step (billing, shipping, payment, review)
		$step = isset( $_POST['step'] ) ? sanitize_text_field( $_POST['step'] ) : 'unknown';

		// Track event
		$this->tracker->track_checkout_abandoned( $step );

		// Clear checkout started flag
		if ( WC()->session ) {
			WC()->session->set( 'flxwoo_checkout_started', null );
		}

		wp_send_json_success();
	}

	/**
	 * Maybe track cart abandonment
	 *
	 * Triggered when cart is emptied. If checkout was started, consider it abandoned.
	 */
	public function maybe_track_cart_abandonment(): void {
		if ( ! WC()->session ) {
			return;
		}

		$checkout_started = WC()->session->get( 'flxwoo_checkout_started' );
		if ( ! $checkout_started ) {
			return;
		}

		// Only track if checkout was started recently (within 24 hours)
		$time_elapsed = time() - $checkout_started;
		if ( $time_elapsed > 24 * 60 * 60 ) {
			return;
		}

		// Track as abandoned (unknown step - cart cleared)
		$this->tracker->track_checkout_abandoned( 'unknown' );

		// Clear flag
		WC()->session->set( 'flxwoo_checkout_started', null );
	}

	/**
	 * Check if analytics tracking is enabled
	 *
	 * Checks feature flag and configuration.
	 *
	 * @return bool True if enabled
	 */
	private function is_analytics_enabled(): bool {
		// Check if WooCommerce is active
		if ( ! class_exists( 'WooCommerce' ) ) {
			return false;
		}

		// Get store ID (same as EventTracker)
		$store_id = $this->tracker->get_store_id();

		// Check feature flag via FeatureManager (v2.3.0)
		$enabled = \FlxWoo\FeatureFlags\FeatureManager::is_enabled(
			\FlxWoo\FeatureFlags\FeatureManager::ANALYTICS_TRACKING,
			$store_id
		);

		// Allow filtering
		return (bool) apply_filters( 'flxwoo_analytics_enabled', $enabled );
	}

	/**
	 * Get current checkout step
	 *
	 * Determines which step of checkout user is on.
	 *
	 * @return string Checkout step: billing, shipping, payment, review, unknown
	 */
	public static function get_checkout_step(): string {
		// This is a simplified version - can be enhanced with more detection logic
		// For now, return unknown - client-side script will determine step
		return 'unknown';
	}
}
