<?php
/**
 * Retention Manager for Activity Log
 *
 * Manages data retention policies and automated cleanup for the activity log.
 *
 * @package FlxWoo
 * @subpackage FeatureFlags
 * @since 2.4.0
 */

namespace FlxWoo\FeatureFlags;

use FlxWoo\Database\ActivityRepository;

/**
 * Class RetentionManager
 *
 * Handles retention policies and automated cleanup of activity log entries.
 */
class RetentionManager {
	/**
	 * WordPress cron hook name for scheduled cleanup
	 */
	const CRON_HOOK = 'flxwoo_activity_log_cleanup';

	/**
	 * WordPress option key for storing retention period
	 */
	const OPTION_KEY = 'flx_woo_retention_period';

	/**
	 * Default retention period in days
	 */
	const DEFAULT_RETENTION_PERIOD = 90;

	/**
	 * Get the current retention period in days
	 *
	 * @return int Number of days to retain activity log entries (0 = never delete)
	 */
	public static function get_retention_period(): int {
		$period = get_option( self::OPTION_KEY, self::DEFAULT_RETENTION_PERIOD );
		return absint( $period );
	}

	/**
	 * Set the retention period
	 *
	 * @param int $days Number of days to retain entries (0 = never delete)
	 * @return bool True on success, false on failure
	 */
	public static function set_retention_period( int $days ): bool {
		$days = absint( $days );

		// Validate: 0 or between 1-3650 days (10 years max)
		if ( $days !== 0 && ( $days < 1 || $days > 3650 ) ) {
			return false;
		}

		// Update option
		// Note: update_option returns false if value hasn't changed, so verify the value is correct
		update_option( self::OPTION_KEY, $days );

		// Verify the option was saved correctly
		return self::get_retention_period() === $days;
	}

	/**
	 * Clean up old activity log records based on retention period
	 *
	 * @return array {
	 *     Results of the cleanup operation
	 *
	 *     @type int    $deleted_count  Number of records deleted
	 *     @type string $cutoff_date    Date cutoff used (YYYY-MM-DD HH:MM:SS)
	 *     @type int    $retention_days Retention period used
	 *     @type bool   $success        Whether cleanup succeeded
	 *     @type string $message        Human-readable result message
	 * }
	 */
	public static function cleanup_old_records(): array {
		$retention_days = self::get_retention_period();

		// If retention is 0, never delete
		if ( $retention_days === 0 ) {
			return [
				'deleted_count'  => 0,
				'cutoff_date'    => null,
				'retention_days' => 0,
				'success'        => true,
				'message'        => __( 'Retention policy is set to "Never delete". No records removed.', 'flx-woo' ),
			];
		}

		// Calculate cutoff date
		$cutoff_date = gmdate( 'Y-m-d H:i:s', strtotime( "-{$retention_days} days" ) );

		// Delete old records
		$repo          = new ActivityRepository();
		$deleted_count = $repo->delete_before_date( $cutoff_date );

		return [
			'deleted_count'  => $deleted_count,
			'cutoff_date'    => $cutoff_date,
			'retention_days' => $retention_days,
			'success'        => true,
			'message'        => sprintf(
				/* translators: 1: number of deleted records, 2: retention period in days */
				_n(
					'Deleted %1$d record older than %2$d days.',
					'Deleted %1$d records older than %2$d days.',
					$deleted_count,
					'flx-woo'
				),
				$deleted_count,
				$retention_days
			),
		];
	}

	/**
	 * Schedule the cleanup cron job
	 *
	 * Schedules a daily cron job to run at 3:00 AM server time.
	 * If already scheduled, does nothing.
	 *
	 * @return bool True if scheduled, false if already scheduled
	 */
	public static function schedule_cleanup_cron(): bool {
		// Check if already scheduled
		if ( wp_next_scheduled( self::CRON_HOOK ) ) {
			return false; // Already scheduled
		}

		// Schedule daily at 3:00 AM tomorrow
		$timestamp = strtotime( 'tomorrow 03:00:00' );

		wp_schedule_event(
			$timestamp,
			'daily',
			self::CRON_HOOK
		);

		return true;
	}

	/**
	 * Unschedule the cleanup cron job
	 *
	 * @return bool True if unscheduled, false if not scheduled
	 */
	public static function unschedule_cleanup_cron(): bool {
		$timestamp = wp_next_scheduled( self::CRON_HOOK );

		if ( ! $timestamp ) {
			return false; // Not scheduled
		}

		wp_unschedule_event( $timestamp, self::CRON_HOOK );
		return true;
	}

	/**
	 * Get statistics about the activity log
	 *
	 * @return array {
	 *     Activity log statistics
	 *
	 *     @type int         $total_records Total number of records
	 *     @type string|null $oldest_date   Oldest record date (YYYY-MM-DD HH:MM:SS) or null
	 *     @type string|null $newest_date   Newest record date (YYYY-MM-DD HH:MM:SS) or null
	 *     @type float       $size_mb       Estimated database size in MB
	 *     @type int         $days_span     Days between oldest and newest record
	 * }
	 */
	public static function get_statistics(): array {
		global $wpdb;
		$table_name = $wpdb->prefix . 'flx_activity_log';

		// Check if table exists
		if ( $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table_name ) ) !== $table_name ) {
			return [
				'total_records' => 0,
				'oldest_date'   => null,
				'newest_date'   => null,
				'size_mb'       => 0.0,
				'days_span'     => 0,
			];
		}

		// Get total count
		$total_records = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table_name}" );

		// Get oldest and newest dates
		$oldest_date = $wpdb->get_var( "SELECT MIN(timestamp) FROM {$table_name}" );
		$newest_date = $wpdb->get_var( "SELECT MAX(timestamp) FROM {$table_name}" );

		// Calculate days span
		$days_span = 0;
		if ( $oldest_date && $newest_date ) {
			$oldest_time = strtotime( $oldest_date );
			$newest_time = strtotime( $newest_date );
			$days_span   = max( 1, (int) round( ( $newest_time - $oldest_time ) / DAY_IN_SECONDS ) );
		}

		// Estimate database size
		$size_query = $wpdb->prepare(
			"SELECT (data_length + index_length) / 1024 / 1024 AS size_mb
			FROM information_schema.TABLES
			WHERE table_schema = %s
			AND table_name = %s",
			DB_NAME,
			$table_name
		);
		$size_mb = (float) $wpdb->get_var( $size_query );

		return [
			'total_records' => $total_records,
			'oldest_date'   => $oldest_date,
			'newest_date'   => $newest_date,
			'size_mb'       => round( $size_mb, 2 ),
			'days_span'     => $days_span,
		];
	}

	/**
	 * Get the next scheduled cleanup time
	 *
	 * @return int|false Timestamp of next scheduled cleanup, or false if not scheduled
	 */
	public static function get_next_cleanup_time() {
		return wp_next_scheduled( self::CRON_HOOK );
	}

	/**
	 * Check if cleanup cron is scheduled
	 *
	 * @return bool True if scheduled, false otherwise
	 */
	public static function is_cron_scheduled(): bool {
		return (bool) wp_next_scheduled( self::CRON_HOOK );
	}

	/**
	 * Calculate how many records would be deleted with current retention period
	 *
	 * @return int Number of records that would be deleted
	 */
	public static function get_deletion_preview(): int {
		$retention_days = self::get_retention_period();

		// If retention is 0, nothing would be deleted
		if ( $retention_days === 0 ) {
			return 0;
		}

		global $wpdb;
		$table_name  = $wpdb->prefix . 'flx_activity_log';
		$cutoff_date = gmdate( 'Y-m-d H:i:s', strtotime( "-{$retention_days} days" ) );

		// Check if table exists
		if ( $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table_name ) ) !== $table_name ) {
			return 0;
		}

		$count = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$table_name} WHERE timestamp < %s",
				$cutoff_date
			)
		);

		return $count;
	}
}
