<?php
/**
 * Feature Flags Manager
 *
 * Manages feature flag configuration and validation for FlxWoo v2.3.0 features.
 * Provides gradual rollout capability and emergency kill switches.
 *
 * @package FlxWoo
 * @version 2.3.0
 */

namespace FlxWoo\FeatureFlags;

use FlxWoo\Constants\Constants;

/**
 * Feature flag manager class
 */
class FeatureManager {
	/**
	 * Available feature flags
	 */
	const ANALYTICS_TRACKING = 'analytics_tracking';
	const BENCHMARKING_DASHBOARD = 'benchmarking_dashboard';
	const AB_TESTING = 'ab_testing';
	const PLUGIN_COMPATIBILITY = 'plugin_compatibility';

	/**
	 * WordPress option key for feature flags
	 */
	const OPTION_KEY = 'flx_woo_feature_flags';

	/**
	 * Get all available feature flags
	 *
	 * @return array<string, array> Feature flag configurations
	 */
	public static function get_default_flags(): array {
		return [
			self::ANALYTICS_TRACKING => [
				'name' => self::ANALYTICS_TRACKING,
				'display_name' => 'Analytics',
				'description' => 'Track anonymous checkout events for benchmarking (GDPR/CCPA compliant)',
				'enabled_globally' => false,
				'rollout_percentage' => 0,
				'version' => '2.3.0',
				'risky_operation' => false,
			],
			self::BENCHMARKING_DASHBOARD => [
				'name' => self::BENCHMARKING_DASHBOARD,
				'display_name' => 'Benchmarking',
				'description' => 'Show store performance vs. industry averages',
				'enabled_globally' => false,
				'rollout_percentage' => 0,
				'version' => '2.3.0',
				'dependencies' => [ self::ANALYTICS_TRACKING ],
				'risky_operation' => false,
			],
			self::AB_TESTING => [
				'name' => self::AB_TESTING,
				'display_name' => 'A/B Testing',
				'description' => 'Test checkout variations to optimize conversion rates',
				'enabled_globally' => false,
				'rollout_percentage' => 0,
				'version' => '2.3.0',
				'dependencies' => [ self::ANALYTICS_TRACKING ],
				'risky_operation' => false,
			],
			self::PLUGIN_COMPATIBILITY => [
				'name' => self::PLUGIN_COMPATIBILITY,
				'display_name' => 'Compatibility',
				'description' => 'Automatically detect and report plugin compatibility issues',
				'enabled_globally' => false,
				'rollout_percentage' => 0,
				'version' => '2.3.0',
				'risky_operation' => false,
			],
		];
	}

	/**
	 * Get feature flag configuration from database
	 *
	 * @return array<string, array> Feature flag configurations
	 */
	public static function get_flags(): array {
		$flags = \get_option( self::OPTION_KEY, null );

		// Auto-initialize if option doesn't exist (for upgrades from older versions)
		if ( $flags === null ) {
			self::initialize();
			$flags = \get_option( self::OPTION_KEY, [] );
		}

		// Merge with defaults to ensure all flags exist (for new flags added in updates)
		$defaults = self::get_default_flags();
		foreach ( $defaults as $flag_name => $default_config ) {
			if ( ! isset( $flags[ $flag_name ] ) ) {
				$flags[ $flag_name ] = $default_config;
			}
		}

		return $flags;
	}

	/**
	 * Get a specific feature flag configuration
	 *
	 * @param string $flag_name Feature flag name.
	 * @return array|null Feature flag configuration or null if not found
	 */
	public static function get_flag( string $flag_name ): ?array {
		$flags = self::get_flags();
		return $flags[ $flag_name ] ?? null;
	}

	/**
	 * Update feature flag configuration
	 *
	 * @param string $flag_name Feature flag name.
	 * @param array  $config Feature flag configuration.
	 * @return bool True on success
	 */
	public static function update_flag( string $flag_name, array $config ): bool {
		$flags = self::get_flags();
		$flags[ $flag_name ] = array_merge( $flags[ $flag_name ] ?? [], $config );
		return \update_option( self::OPTION_KEY, $flags );
	}

	/**
	 * Check if a feature is enabled for the current store
	 *
	 * Decision flow:
	 * 1. Check global kill switch
	 * 2. If store is in disabled_stores → disabled (override)
	 * 3. If store is in enabled_stores → enabled
	 * 4. If enabled_globally → enabled
	 * 5. If rollout_percentage > 0 → check hash-based rollout
	 * 6. Otherwise → disabled
	 *
	 * @param string $flag_name Feature flag name.
	 * @param string $store_id SHA-256 hash of store identifier.
	 * @return bool True if feature is enabled
	 */
	public static function is_enabled( string $flag_name, string $store_id ): bool {
		// Check global kill switch
		if ( self::is_kill_switch_active() ) {
			return false;
		}

		$flag = self::get_flag( $flag_name );
		if ( ! $flag ) {
			return false; // Flag not found - default to disabled
		}

		// Override: explicitly disabled
		if ( isset( $flag['disabled_stores'] ) && in_array( $store_id, $flag['disabled_stores'], true ) ) {
			return false;
		}

		// Explicitly enabled for this store
		if ( isset( $flag['enabled_stores'] ) && in_array( $store_id, $flag['enabled_stores'], true ) ) {
			return true;
		}

		// Globally enabled
		if ( ! empty( $flag['enabled_globally'] ) ) {
			return true;
		}

		// Gradual rollout via percentage
		$rollout_percentage = $flag['rollout_percentage'] ?? 0;
		if ( $rollout_percentage > 0 ) {
			return self::is_in_rollout_percentage( $store_id, $rollout_percentage );
		}

		// Default: disabled
		return false;
	}

	/**
	 * Deterministic check if store is in rollout percentage
	 * Uses hash of store_id for consistent assignment
	 *
	 * @param string $store_id SHA-256 hash of store identifier.
	 * @param int    $percentage Rollout percentage (0-100).
	 * @return bool True if store is in rollout
	 */
	private static function is_in_rollout_percentage( string $store_id, int $percentage ): bool {
		if ( $percentage <= 0 ) {
			return false;
		}
		if ( $percentage >= 100 ) {
			return true;
		}

		// Simple hash-based assignment: take first 8 characters of store_id
		// Convert to number and modulo 100 for 0-99 range
		$hash_value = hexdec( substr( $store_id, 0, 8 ) );
		$bucket = $hash_value % 100;

		return $bucket < $percentage;
	}

	/**
	 * Check if global kill switch is active
	 *
	 * @return bool True if all features should be disabled
	 */
	public static function is_kill_switch_active(): bool {
		// Check constant (wp-config.php override)
		if ( \defined( 'FLX_WOO_KILL_SWITCH' ) && FLX_WOO_KILL_SWITCH ) {
			return true;
		}

		// Check option (admin toggle)
		return (bool) \get_option( 'flx_woo_kill_switch', false );
	}

	/**
	 * Enable/disable global kill switch
	 *
	 * @param bool $enabled True to enable kill switch (disable all features).
	 * @return bool True on success
	 */
	public static function set_kill_switch( bool $enabled ): bool {
		return \update_option( 'flx_woo_kill_switch', $enabled );
	}

	/**
	 * Get current store ID (SHA-256 hash)
	 *
	 * @return string SHA-256 hash of store URL
	 */
	public static function get_store_id(): string {
		$home_url = \get_home_url();
		return hash( 'sha256', $home_url );
	}

	/**
	 * Validate feature dependencies
	 *
	 * @param string $flag_name Feature flag name.
	 * @param array  $enabled_flags Currently enabled flag names.
	 * @return string|null Error message if dependencies not met, null if valid
	 */
	public static function validate_dependencies( string $flag_name, array $enabled_flags ): ?string {
		$flag = self::get_flag( $flag_name );
		if ( ! $flag ) {
			return 'Feature flag not found';
		}

		if ( empty( $flag['dependencies'] ) ) {
			return null;
		}

		$missing_dependencies = array_diff( $flag['dependencies'], $enabled_flags );
		if ( ! empty( $missing_dependencies ) ) {
			$dep_names = array_map(
				function( $dep ) {
					$dep_flag = self::get_flag( $dep );
					return $dep_flag['display_name'] ?? $dep;
				},
				$missing_dependencies
			);

			return sprintf(
				'Cannot enable %s - missing dependencies: %s',
				$flag['display_name'],
				implode( ', ', $dep_names )
			);
		}

		return null;
	}

	/**
	 * Get all features that depend on a given feature
	 *
	 * @param string $flag_name Feature flag name.
	 * @return array Array of dependent feature flag names
	 */
	public static function get_dependent_features( string $flag_name ): array {
		$flags = self::get_flags();
		$dependents = [];

		foreach ( $flags as $name => $config ) {
			if ( isset( $config['dependencies'] ) && in_array( $flag_name, $config['dependencies'], true ) ) {
				$dependents[] = $name;
			}
		}

		return $dependents;
	}

	/**
	 * Get all enabled features for the current store
	 *
	 * @return array Array of enabled feature flag names
	 */
	public static function get_enabled_features(): array {
		$store_id = self::get_store_id();
		$flags = self::get_flags();
		$enabled = [];

		foreach ( $flags as $flag_name => $config ) {
			if ( self::is_enabled( $flag_name, $store_id ) ) {
				$enabled[] = $flag_name;
			}
		}

		return $enabled;
	}

	/**
	 * Initialize feature flags (called on plugin activation)
	 *
	 * @return void
	 */
	public static function initialize(): void {
		$existing = \get_option( self::OPTION_KEY, [] );
		if ( empty( $existing ) ) {
			// First time initialization - use defaults
			\update_option( self::OPTION_KEY, self::get_default_flags() );
		}
	}
}
