<?php
/**
 * Activity Logger for Feature Flags
 *
 * Tracks changes to feature flags for audit trail and debugging.
 * Uses custom database table if available, falls back to options table.
 *
 * @package FlxWoo
 * @version 2.4.0
 */

namespace FlxWoo\FeatureFlags;

use FlxWoo\Database\ActivityRepository;

/**
 * Activity logger class
 */
class ActivityLogger {
	/**
	 * WordPress option key for activity log
	 */
	const OPTION_KEY = 'flx_woo_feature_activity_log';

	/**
	 * Maximum number of activity entries to store
	 */
	const MAX_ENTRIES = 50;

	/**
	 * Log a feature flag change
	 *
	 * @param string $flag_name Feature flag name.
	 * @param string $action Action performed (enabled, disabled, updated, kill_switch).
	 * @param mixed  $old_value Previous value.
	 * @param mixed  $new_value New value.
	 * @param int    $user_id User ID who made the change (0 for system).
	 * @return bool True on success
	 */
	public static function log_change( string $flag_name, string $action, $old_value, $new_value, int $user_id = 0 ): bool {
		$entry = [
			'timestamp' => time(),
			'flag_name' => $flag_name,
			'action' => $action,
			'old_value' => $old_value,
			'new_value' => $new_value,
			'user_id' => $user_id > 0 ? $user_id : \get_current_user_id(),
			'user_login' => self::get_user_login( $user_id > 0 ? $user_id : \get_current_user_id() ),
			'ip_address' => self::get_client_ip(),
		];

		// Use database if available, fallback to options
		if ( self::use_database() ) {
			$repo = new ActivityRepository();
			$result = $repo->insert( $entry );

			if ( $result === 0 ) {
				// Database insert failed - log error and fallback to options
				\error_log( sprintf(
					'FlxWoo ActivityLogger: Failed to insert activity log entry to database table. ' .
					'Flag: %s, Action: %s. Falling back to options table.',
					$flag_name,
					$action
				) );

				// Attempt fallback to options table
				return self::log_change_to_options( $entry );
			}

			return $result > 0;
		}

		// Table doesn't exist yet - use options table (this is normal during upgrades)
		return self::log_change_to_options( $entry );
	}

	/**
	 * Log change to options table (legacy/fallback method)
	 *
	 * @param array $entry Activity entry.
	 * @return bool True on success
	 */
	private static function log_change_to_options( array $entry ): bool {
		$log = self::get_activity_log();
		array_unshift( $log, $entry ); // Add to beginning

		// Limit to MAX_ENTRIES
		$log = array_slice( $log, 0, self::MAX_ENTRIES );

		return \update_option( self::OPTION_KEY, $log );
	}

	/**
	 * Get recent activity log
	 *
	 * @param int $limit Maximum number of entries to return.
	 * @return array Array of activity entries
	 */
	public static function get_recent_activity( int $limit = 10 ): array {
		// Use database if available, fallback to options
		if ( self::use_database() ) {
			$repo = new ActivityRepository();
			return $repo->get_recent( $limit );
		}

		// Fallback to options table
		$log = self::get_activity_log();
		return array_slice( $log, 0, $limit );
	}

	/**
	 * Get full activity log from database
	 *
	 * @return array Array of activity entries
	 */
	private static function get_activity_log(): array {
		$log = \get_option( self::OPTION_KEY, [] );
		return is_array( $log ) ? $log : [];
	}

	/**
	 * Clear all activity log entries
	 *
	 * @return bool True on success
	 */
	public static function clear_log(): bool {
		return \delete_option( self::OPTION_KEY );
	}

	/**
	 * Get user login name from user ID
	 *
	 * @param int $user_id User ID.
	 * @return string User login or 'System'
	 */
	private static function get_user_login( int $user_id ): string {
		if ( $user_id === 0 ) {
			return 'System';
		}

		$user = \get_user_by( 'id', $user_id );
		return $user ? $user->user_login : 'Unknown';
	}

	/**
	 * Get client IP address (sanitized for privacy)
	 *
	 * @return string Sanitized IP address or 'Unknown'
	 */
	private static function get_client_ip(): string {
		if ( ! empty( $_SERVER['REMOTE_ADDR'] ) ) {
			return self::anonymize_ip( $_SERVER['REMOTE_ADDR'] );
		}
		return 'Unknown';
	}

	/**
	 * Anonymize IP address for GDPR/privacy compliance
	 * Removes last octet of IPv4 or last 80 bits of IPv6
	 *
	 * @param string $ip IP address.
	 * @return string Anonymized IP address
	 */
	private static function anonymize_ip( string $ip ): string {
		// IPv4
		if ( strpos( $ip, '.' ) !== false ) {
			$parts = explode( '.', $ip );
			$parts[3] = '0'; // Replace last octet with 0
			return implode( '.', $parts );
		}

		// IPv6
		if ( strpos( $ip, ':' ) !== false ) {
			$parts = explode( ':', $ip );
			// Keep first 48 bits (3 groups of 16 bits), anonymize rest
			return implode( ':', array_slice( $parts, 0, 3 ) ) . '::';
		}

		return $ip;
	}

	/**
	 * Get formatted action label for display
	 *
	 * @param string $action Action type.
	 * @return string Human-readable action label
	 */
	public static function get_action_label( string $action ): string {
		$labels = [
			'enabled' => 'Enabled',
			'disabled' => 'Disabled',
			'updated' => 'Updated',
			'kill_switch_on' => 'Kill Switch Activated',
			'kill_switch_off' => 'Kill Switch Deactivated',
			'rollout_changed' => 'Rollout % Changed',
		];

		return $labels[ $action ] ?? ucfirst( $action );
	}

	/**
	 * Format activity entry for display
	 *
	 * @param array $entry Activity entry.
	 * @return string Formatted description
	 */
	public static function format_entry( array $entry ): string {
		$flag_display = 'Unknown';
		if ( isset( $entry['flag_name'] ) ) {
			$flag_data = FeatureManager::get_flag( $entry['flag_name'] );
			$flag_display = $flag_data['display_name'] ?? str_replace( '_', ' ', ucwords( $entry['flag_name'], '_' ) );
		}
		$action = self::get_action_label( $entry['action'] ?? 'unknown' );

		$description = sprintf(
			'%s: %s',
			$flag_display,
			$action
		);

		// Add value change details if available
		if ( isset( $entry['old_value'], $entry['new_value'] ) ) {
			if ( is_bool( $entry['old_value'] ) || is_bool( $entry['new_value'] ) ) {
				$old = $entry['old_value'] ? 'enabled' : 'disabled';
				$new = $entry['new_value'] ? 'enabled' : 'disabled';
				$description .= sprintf( ' (from %s to %s)', $old, $new );
			} elseif ( is_numeric( $entry['old_value'] ) && is_numeric( $entry['new_value'] ) ) {
				$description .= sprintf( ' (from %d%% to %d%%)', $entry['old_value'], $entry['new_value'] );
			}
		}

		return $description;
	}

	/**
	 * Get statistics about activity
	 *
	 * @return array Activity statistics
	 */
	public static function get_statistics(): array {
		// Use database if available, fallback to options
		if ( self::use_database() ) {
			$repo = new ActivityRepository();
			return $repo->get_statistics();
		}

		// Fallback to options table calculation
		$log = self::get_activity_log();

		$stats = [
			'total_changes' => count( $log ),
			'last_24h' => 0,
			'last_7d' => 0,
			'most_active_user' => 'None',
			'most_changed_flag' => 'None',
		];

		if ( empty( $log ) ) {
			return $stats;
		}

		$now = time();
		$user_counts = [];
		$flag_counts = [];

		foreach ( $log as $entry ) {
			$timestamp = $entry['timestamp'] ?? 0;
			$age = $now - $timestamp;

			// Count by time period
			if ( $age <= DAY_IN_SECONDS ) {
				$stats['last_24h']++;
			}
			if ( $age <= 7 * DAY_IN_SECONDS ) {
				$stats['last_7d']++;
			}

			// Count by user
			$user = $entry['user_login'] ?? 'Unknown';
			$user_counts[ $user ] = ( $user_counts[ $user ] ?? 0 ) + 1;

			// Count by flag
			$flag = $entry['flag_name'] ?? 'unknown';
			$flag_counts[ $flag ] = ( $flag_counts[ $flag ] ?? 0 ) + 1;
		}

		// Find most active user
		if ( ! empty( $user_counts ) ) {
			arsort( $user_counts );
			$stats['most_active_user'] = key( $user_counts );
		}

		// Find most changed flag
		if ( ! empty( $flag_counts ) ) {
			arsort( $flag_counts );
			$stats['most_changed_flag'] = key( $flag_counts );
		}

		return $stats;
	}

	/**
	 * Check if database table is available
	 *
	 * @return bool True if database table exists
	 */
	private static function use_database(): bool {
		global $wpdb;

		$table_name = $wpdb->prefix . 'flx_activity_log';

		// Check if table exists
		$table_exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table_name ) ) === $table_name;

		return $table_exists;
	}
}
