<?php
/**
 * Database Migrator
 *
 * Handles creation of custom database tables and data migration.
 * This is the first custom database table in the FlxWoo plugin.
 *
 * @package FlxWoo\Database
 * @since 2.4.0
 */

namespace FlxWoo\Database;

if (!defined('ABSPATH')) {
	exit;
}

class Migrator {
	/**
	 * Database version
	 */
	const DB_VERSION = '2.4.0';

	/**
	 * Option key for storing database version
	 */
	const VERSION_OPTION = 'flx_woo_db_version';

	/**
	 * Option key for activity log backup
	 */
	const BACKUP_OPTION = 'flx_woo_activity_log_backup';

	/**
	 * Backup expiration time (30 days)
	 */
	const BACKUP_EXPIRATION = 2592000; // 30 * 24 * 60 * 60

	/**
	 * Create activity log table
	 *
	 * @return bool True if table created successfully
	 */
	public static function create_table(): bool {
		global $wpdb;

		$table_name = $wpdb->prefix . 'flx_activity_log';
		$charset_collate = $wpdb->get_charset_collate();

		// Check if table already exists
		if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)) === $table_name) {
			return true;
		}

		$sql = "CREATE TABLE {$table_name} (
			id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
			timestamp DATETIME NOT NULL,
			flag_name VARCHAR(100) NOT NULL,
			action VARCHAR(50) NOT NULL,
			old_value TEXT DEFAULT NULL,
			new_value TEXT DEFAULT NULL,
			user_id BIGINT(20) UNSIGNED NOT NULL,
			user_login VARCHAR(60) NOT NULL,
			ip_address VARCHAR(45) NOT NULL,
			created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
			PRIMARY KEY (id),
			INDEX idx_timestamp (timestamp),
			INDEX idx_flag_name (flag_name),
			INDEX idx_user_id (user_id),
			INDEX idx_action (action),
			INDEX idx_created_at (created_at)
		) ENGINE=InnoDB {$charset_collate};";

		// Load dbDelta function if not already available (e.g., in tests)
		if (!function_exists('dbDelta')) {
			require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		}
		dbDelta($sql);

		// Verify table was created
		$table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)) === $table_name;

		if ($table_exists) {
			\update_option(self::VERSION_OPTION, self::DB_VERSION);
		}

		return $table_exists;
	}

	/**
	 * Migrate activity log data from wp_options to database table
	 *
	 * @return array Migration results with stats
	 */
	public static function migrate_from_options(): array {
		global $wpdb;

		$table_name = $wpdb->prefix . 'flx_activity_log';

		// Check if table exists
		if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)) !== $table_name) {
			return [
				'success' => false,
				'error' => 'Table does not exist',
				'migrated_count' => 0,
			];
		}

		// Check if already migrated (table has data)
		$existing_count = $wpdb->get_var("SELECT COUNT(*) FROM {$table_name}");
		if ($existing_count > 0) {
			return [
				'success' => true,
				'already_migrated' => true,
				'existing_count' => (int) $existing_count,
				'migrated_count' => 0,
			];
		}

		// Get activity log from options
		$activity_log = \get_option('flx_woo_feature_activity_log', []);

		if (empty($activity_log) || !is_array($activity_log)) {
			return [
				'success' => true,
				'migrated_count' => 0,
				'message' => 'No activity log data to migrate',
			];
		}

		// Backup old data to transient (30 days)
		\set_transient(self::BACKUP_OPTION, $activity_log, self::BACKUP_EXPIRATION);

		// Migrate each entry
		$migrated_count = 0;
		$errors = [];

		foreach ($activity_log as $entry) {
			// Validate entry has required fields
			if (!isset($entry['timestamp'], $entry['flag_name'], $entry['action'])) {
				$errors[] = 'Invalid entry: missing required fields';
				continue;
			}

			// Convert timestamp to DATETIME
			$datetime = gmdate('Y-m-d H:i:s', $entry['timestamp']);

			// Prepare values (handle missing fields)
			$old_value = isset($entry['old_value']) ? self::serialize_value($entry['old_value']) : null;
			$new_value = isset($entry['new_value']) ? self::serialize_value($entry['new_value']) : null;

			$result = $wpdb->insert(
				$table_name,
				[
					'timestamp' => $datetime,
					'flag_name' => $entry['flag_name'],
					'action' => $entry['action'],
					'old_value' => $old_value,
					'new_value' => $new_value,
					'user_id' => $entry['user_id'] ?? 0,
					'user_login' => $entry['user_login'] ?? 'system',
					'ip_address' => $entry['ip_address'] ?? '127.0.0.1',
				],
				[
					'%s', // timestamp
					'%s', // flag_name
					'%s', // action
					'%s', // old_value
					'%s', // new_value
					'%d', // user_id
					'%s', // user_login
					'%s', // ip_address
				]
			);

			if ($result === false) {
				$errors[] = $wpdb->last_error;
			} else {
				$migrated_count++;
			}
		}

		return [
			'success' => true,
			'migrated_count' => $migrated_count,
			'total_entries' => count($activity_log),
			'errors' => $errors,
			'backup_created' => true,
			'backup_expires' => self::BACKUP_EXPIRATION,
		];
	}

	/**
	 * Check if migration is needed
	 *
	 * @return bool True if migration needed
	 */
	public static function needs_migration(): bool {
		global $wpdb;

		$table_name = $wpdb->prefix . 'flx_activity_log';

		// Check if table exists
		$table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)) === $table_name;

		if (!$table_exists) {
			return true;
		}

		// Check if table has data (if yes, migration already done)
		$has_data = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table_name}") > 0;

		// Check if options table has data
		$options_data = \get_option('flx_woo_feature_activity_log', []);
		$has_options_data = !empty($options_data) && is_array($options_data);

		// Need migration if table is empty but options has data
		return !$has_data && $has_options_data;
	}

	/**
	 * Check if already migrated
	 *
	 * @return bool True if migrated
	 */
	public static function is_migrated(): bool {
		global $wpdb;

		$table_name = $wpdb->prefix . 'flx_activity_log';

		// Table exists and has data
		$table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)) === $table_name;

		if (!$table_exists) {
			return false;
		}

		// Has data or version option set
		$has_data = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table_name}") > 0;
		$version = \get_option(self::VERSION_OPTION, '');

		return $has_data || $version === self::DB_VERSION;
	}

	/**
	 * Rollback migration (restore from backup)
	 *
	 * @return bool True if rollback successful
	 */
	public static function rollback(): bool {
		global $wpdb;

		// Get backup from transient
		$backup = \get_transient(self::BACKUP_OPTION);

		if ($backup === false) {
			return false;
		}

		// Restore to options
		$restored = \update_option('flx_woo_feature_activity_log', $backup);

		if ($restored) {
			// Clear table
			$table_name = $wpdb->prefix . 'flx_activity_log';
			$wpdb->query("TRUNCATE TABLE {$table_name}");

			// Delete transient
			\delete_transient(self::BACKUP_OPTION);
		}

		return $restored;
	}

	/**
	 * Get current database version
	 *
	 * @return string Database version
	 */
	public static function get_db_version(): string {
		return \get_option(self::VERSION_OPTION, '0.0.0');
	}

	/**
	 * Get migration statistics
	 *
	 * @return array Stats about migration
	 */
	public static function get_migration_stats(): array {
		global $wpdb;

		$table_name = $wpdb->prefix . 'flx_activity_log';
		$table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)) === $table_name;

		$stats = [
			'table_exists' => $table_exists,
			'db_version' => self::get_db_version(),
			'is_migrated' => self::is_migrated(),
			'needs_migration' => self::needs_migration(),
		];

		if ($table_exists) {
			$stats['table_row_count'] = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table_name}");
		}

		$options_data = \get_option('flx_woo_feature_activity_log', []);
		$stats['options_row_count'] = is_array($options_data) ? count($options_data) : 0;

		$backup = \get_transient(self::BACKUP_OPTION);
		$stats['has_backup'] = $backup !== false;

		return $stats;
	}

	/**
	 * Serialize value for database storage
	 *
	 * @param mixed $value Value to serialize
	 * @return string Serialized value
	 */
	private static function serialize_value($value): string {
		if (is_bool($value)) {
			return $value ? 'true' : 'false';
		}

		if (is_numeric($value)) {
			return (string) $value;
		}

		if (is_array($value) || is_object($value)) {
			return wp_json_encode($value);
		}

		return (string) $value;
	}

	/**
	 * Drop the activity log table (use with caution)
	 *
	 * @return bool True if dropped successfully
	 */
	public static function drop_table(): bool {
		global $wpdb;

		$table_name = $wpdb->prefix . 'flx_activity_log';

		$result = $wpdb->query("DROP TABLE IF EXISTS {$table_name}");

		if ($result !== false) {
			\delete_option(self::VERSION_OPTION);
		}

		return $result !== false;
	}
}
