<?php
/**
 * Activity Repository
 *
 * Database access layer for activity log operations.
 * Provides CRUD operations, filtering, statistics, and pagination.
 *
 * @package FlxWoo\Database
 * @since 2.4.0
 */

namespace FlxWoo\Database;

if (!defined('ABSPATH')) {
	exit;
}

class ActivityRepository {
	/**
	 * Table name (without prefix)
	 */
	const TABLE_NAME = 'flx_activity_log';

	/**
	 * Check if the activity log table exists
	 *
	 * @return bool True if table exists
	 */
	private function table_exists(): bool {
		global $wpdb;
		$table_name = $wpdb->prefix . self::TABLE_NAME;
		$query = $wpdb->prepare('SHOW TABLES LIKE %s', $table_name);
		return $wpdb->get_var($query) === $table_name;
	}

	/**
	 * Insert a new activity log entry
	 *
	 * @param array $entry Activity log entry data
	 * @return int Insert ID or 0 on failure
	 */
	public function insert(array $entry): int {
		global $wpdb;

		$table_name = $wpdb->prefix . self::TABLE_NAME;

		// Convert timestamp to DATETIME if needed
		if (isset($entry['timestamp']) && is_numeric($entry['timestamp'])) {
			$entry['timestamp'] = gmdate('Y-m-d H:i:s', $entry['timestamp']);
		}

		// Serialize complex values
		if (isset($entry['old_value'])) {
			$entry['old_value'] = $this->serialize_value($entry['old_value']);
		}

		if (isset($entry['new_value'])) {
			$entry['new_value'] = $this->serialize_value($entry['new_value']);
		}

		$result = $wpdb->insert(
			$table_name,
			[
				'timestamp' => $entry['timestamp'] ?? gmdate('Y-m-d H:i:s'),
				'flag_name' => $entry['flag_name'],
				'action' => $entry['action'],
				'old_value' => $entry['old_value'] ?? null,
				'new_value' => $entry['new_value'] ?? null,
				'user_id' => $entry['user_id'] ?? 0,
				'user_login' => $entry['user_login'] ?? 'system',
				'ip_address' => $entry['ip_address'] ?? '127.0.0.1',
			],
			[
				'%s', // timestamp
				'%s', // flag_name
				'%s', // action
				'%s', // old_value
				'%s', // new_value
				'%d', // user_id
				'%s', // user_login
				'%s', // ip_address
			]
		);

		return $result ? (int) $wpdb->insert_id : 0;
	}

	/**
	 * Get recent activity entries
	 *
	 * @param int $limit Number of entries to retrieve
	 * @param int $offset Offset for pagination
	 * @return array Activity entries
	 */
	public function get_recent(int $limit = 10, int $offset = 0): array {
		// Return empty array if table doesn't exist yet
		if (!$this->table_exists()) {
			return [];
		}

		global $wpdb;

		$table_name = $wpdb->prefix . self::TABLE_NAME;

		$sql = $wpdb->prepare(
			"SELECT * FROM {$table_name} ORDER BY timestamp DESC, id DESC LIMIT %d OFFSET %d",
			$limit,
			$offset
		);

		$results = $wpdb->get_results($sql, ARRAY_A);

		return $this->format_results($results);
	}

	/**
	 * Get activity entries with filters
	 *
	 * @param array $filters Filter criteria
	 * @param int $limit Number of entries to retrieve
	 * @param int $offset Offset for pagination
	 * @return array Activity entries
	 */
	public function get_by_filters(array $filters, int $limit = 50, int $offset = 0): array {
		// Return empty array if table doesn't exist yet
		if (!$this->table_exists()) {
			return [];
		}

		global $wpdb;

		$table_name = $wpdb->prefix . self::TABLE_NAME;
		$where_clauses = ['1=1'];
		$params = [];

		// Date range filter
		if (!empty($filters['start_date'])) {
			$where_clauses[] = 'timestamp >= %s';
			$params[] = $filters['start_date'] . ' 00:00:00';
		}

		if (!empty($filters['end_date'])) {
			$where_clauses[] = 'timestamp <= %s';
			$params[] = $filters['end_date'] . ' 23:59:59';
		}

		// Feature flag filter (supports both singular and plural parameter names)
		$flag_filter = $filters['flag_name'] ?? $filters['flag_names'] ?? null;
		if (!empty($flag_filter)) {
			if (is_array($flag_filter)) {
				$placeholders = implode(',', array_fill(0, count($flag_filter), '%s'));
				$where_clauses[] = "flag_name IN ($placeholders)";
				$params = array_merge($params, $flag_filter);
			} else {
				$where_clauses[] = 'flag_name = %s';
				$params[] = $flag_filter;
			}
		}

		// Action filter (supports both singular and plural parameter names)
		$action_filter = $filters['action'] ?? $filters['actions'] ?? null;
		if (!empty($action_filter)) {
			if (is_array($action_filter)) {
				$placeholders = implode(',', array_fill(0, count($action_filter), '%s'));
				$where_clauses[] = "action IN ($placeholders)";
				$params = array_merge($params, $action_filter);
			} else {
				$where_clauses[] = 'action = %s';
				$params[] = $action_filter;
			}
		}

		// User filter
		if (!empty($filters['user_id'])) {
			$where_clauses[] = 'user_id = %d';
			$params[] = (int) $filters['user_id'];
		}

		if (!empty($filters['user_login'])) {
			$where_clauses[] = 'user_login LIKE %s';
			$params[] = '%' . $wpdb->esc_like($filters['user_login']) . '%';
		}

		// IP address filter
		if (!empty($filters['ip_address'])) {
			$where_clauses[] = 'ip_address LIKE %s';
			$params[] = '%' . $wpdb->esc_like($filters['ip_address']) . '%';
		}

		$where_sql = implode(' AND ', $where_clauses);

		// Add limit and offset
		$params[] = $limit;
		$params[] = $offset;

		$sql = $wpdb->prepare(
			"SELECT * FROM {$table_name} WHERE {$where_sql} ORDER BY timestamp DESC, id DESC LIMIT %d OFFSET %d",
			$params
		);

		$results = $wpdb->get_results($sql, ARRAY_A);

		return $this->format_results($results);
	}

	/**
	 * Count entries matching filters
	 *
	 * @param array $filters Filter criteria
	 * @return int Count of matching entries
	 */
	public function count_by_filters(array $filters): int {
		// Return 0 if table doesn't exist yet
		if (!$this->table_exists()) {
			return 0;
		}

		global $wpdb;

		$table_name = $wpdb->prefix . self::TABLE_NAME;
		$where_clauses = ['1=1'];
		$params = [];

		// Same filter logic as get_by_filters (without limit/offset)
		if (!empty($filters['start_date'])) {
			$where_clauses[] = 'timestamp >= %s';
			$params[] = $filters['start_date'] . ' 00:00:00';
		}

		if (!empty($filters['end_date'])) {
			$where_clauses[] = 'timestamp <= %s';
			$params[] = $filters['end_date'] . ' 23:59:59';
		}

		// Feature flag filter (supports both singular and plural parameter names)
		$flag_filter = $filters['flag_name'] ?? $filters['flag_names'] ?? null;
		if (!empty($flag_filter)) {
			if (is_array($flag_filter)) {
				$placeholders = implode(',', array_fill(0, count($flag_filter), '%s'));
				$where_clauses[] = "flag_name IN ($placeholders)";
				$params = array_merge($params, $flag_filter);
			} else {
				$where_clauses[] = 'flag_name = %s';
				$params[] = $flag_filter;
			}
		}

		// Action filter (supports both singular and plural parameter names)
		$action_filter = $filters['action'] ?? $filters['actions'] ?? null;
		if (!empty($action_filter)) {
			if (is_array($action_filter)) {
				$placeholders = implode(',', array_fill(0, count($action_filter), '%s'));
				$where_clauses[] = "action IN ($placeholders)";
				$params = array_merge($params, $action_filter);
			} else {
				$where_clauses[] = 'action = %s';
				$params[] = $action_filter;
			}
		}

		if (!empty($filters['user_id'])) {
			$where_clauses[] = 'user_id = %d';
			$params[] = (int) $filters['user_id'];
		}

		if (!empty($filters['user_login'])) {
			$where_clauses[] = 'user_login LIKE %s';
			$params[] = '%' . $wpdb->esc_like($filters['user_login']) . '%';
		}

		if (!empty($filters['ip_address'])) {
			$where_clauses[] = 'ip_address LIKE %s';
			$params[] = '%' . $wpdb->esc_like($filters['ip_address']) . '%';
		}

		$where_sql = implode(' AND ', $where_clauses);

		$sql = "SELECT COUNT(*) FROM {$table_name} WHERE {$where_sql}";

		// Only prepare if we have parameters
		if (!empty($params)) {
			$sql = $wpdb->prepare($sql, $params);
		}

		return (int) $wpdb->get_var($sql);
	}

	/**
	 * Get activity statistics
	 *
	 * @param array $filters Optional filters
	 * @return array Statistics
	 */
	public function get_statistics(array $filters = []): array {
		global $wpdb;

		$table_name = $wpdb->prefix . self::TABLE_NAME;

		// Total changes
		$total_changes = $this->count_by_filters($filters);

		// Last 24 hours
		$filters_24h = array_merge($filters, [
			'start_date' => gmdate('Y-m-d', strtotime('-1 day')),
		]);
		$last_24h = $this->count_by_filters($filters_24h);

		// Last 7 days
		$filters_7d = array_merge($filters, [
			'start_date' => gmdate('Y-m-d', strtotime('-7 days')),
		]);
		$last_7d = $this->count_by_filters($filters_7d);

		// Most active user
		$most_active_user = 'N/A';
		$user_sql = "SELECT user_login, COUNT(*) as count FROM {$table_name}";
		$where_clauses = ['1=1'];
		$params = [];

		// Build WHERE clauses only if filters exist
		if (!empty($filters)) {
			if (!empty($filters['start_date'])) {
				$where_clauses[] = 'timestamp >= %s';
				$params[] = $filters['start_date'] . ' 00:00:00';
			}

			if (!empty($filters['end_date'])) {
				$where_clauses[] = 'timestamp <= %s';
				$params[] = $filters['end_date'] . ' 23:59:59';
			}
		}

		// Build COMPLETE query BEFORE preparing
		$where_sql = implode(' AND ', $where_clauses);
		$user_sql .= " WHERE {$where_sql}";
		$user_sql .= " GROUP BY user_login ORDER BY count DESC LIMIT 1";

		// Only prepare if we have parameters
		if (!empty($params)) {
			$user_sql = $wpdb->prepare($user_sql, $params);
		}

		$top_user = $wpdb->get_row($user_sql);

		if ($top_user) {
			$most_active_user = $top_user->user_login;
		}

		// Most changed flag
		$most_changed_flag = 'N/A';
		$flag_sql = "SELECT flag_name, COUNT(*) as count FROM {$table_name}";
		$where_clauses = ['1=1'];
		$params = [];

		// Build WHERE clauses only if filters exist
		if (!empty($filters)) {
			if (!empty($filters['start_date'])) {
				$where_clauses[] = 'timestamp >= %s';
				$params[] = $filters['start_date'] . ' 00:00:00';
			}

			if (!empty($filters['end_date'])) {
				$where_clauses[] = 'timestamp <= %s';
				$params[] = $filters['end_date'] . ' 23:59:59';
			}
		}

		// Build COMPLETE query BEFORE preparing
		$where_sql = implode(' AND ', $where_clauses);
		$flag_sql .= " WHERE {$where_sql}";
		$flag_sql .= " GROUP BY flag_name ORDER BY count DESC LIMIT 1";

		// Only prepare if we have parameters
		if (!empty($params)) {
			$flag_sql = $wpdb->prepare($flag_sql, $params);
		}

		$top_flag = $wpdb->get_row($flag_sql);

		if ($top_flag) {
			$most_changed_flag = $top_flag->flag_name;
		}

		return [
			'total_changes' => $total_changes,
			'last_24h' => $last_24h,
			'last_7d' => $last_7d,
			'most_active_user' => $most_active_user,
			'most_changed_flag' => $most_changed_flag,
		];
	}

	/**
	 * Delete entries before a specific date
	 *
	 * @param string $date Date in Y-m-d format
	 * @return int Number of deleted rows
	 */
	public function delete_before_date(string $date): int {
		global $wpdb;

		$table_name = $wpdb->prefix . self::TABLE_NAME;

		$sql = $wpdb->prepare(
			"DELETE FROM {$table_name} WHERE timestamp < %s",
			$date . ' 00:00:00'
		);

		$result = $wpdb->query($sql);

		return $result !== false ? (int) $result : 0;
	}

	/**
	 * Get date range of activity log
	 *
	 * @return array Min and max dates
	 */
	public function get_date_range(): array {
		global $wpdb;

		$table_name = $wpdb->prefix . self::TABLE_NAME;

		$sql = "SELECT MIN(timestamp) as min_date, MAX(timestamp) as max_date FROM {$table_name}";
		$result = $wpdb->get_row($sql, ARRAY_A);

		return [
			'min_date' => $result['min_date'] ?? null,
			'max_date' => $result['max_date'] ?? null,
		];
	}

	/**
	 * Get unique feature flags in activity log
	 *
	 * @return array List of unique flag names
	 */
	public function get_unique_flags(): array {
		// Return empty array if table doesn't exist yet
		if (!$this->table_exists()) {
			return [];
		}

		global $wpdb;

		$table_name = $wpdb->prefix . self::TABLE_NAME;

		$sql = "SELECT DISTINCT flag_name FROM {$table_name} ORDER BY flag_name";
		$results = $wpdb->get_col($sql);

		return $results ?: [];
	}

	/**
	 * Get unique actions in activity log
	 *
	 * @return array List of unique actions
	 */
	public function get_unique_actions(): array {
		// Return empty array if table doesn't exist yet
		if (!$this->table_exists()) {
			return [];
		}

		global $wpdb;

		$table_name = $wpdb->prefix . self::TABLE_NAME;

		$sql = "SELECT DISTINCT action FROM {$table_name} ORDER BY action";
		$results = $wpdb->get_col($sql);

		return $results ?: [];
	}

	/**
	 * Get unique users in activity log
	 *
	 * @return array List of unique users
	 */
	public function get_unique_users(): array {
		// Return empty array if table doesn't exist yet
		if (!$this->table_exists()) {
			return [];
		}

		global $wpdb;

		$table_name = $wpdb->prefix . self::TABLE_NAME;

		$sql = "SELECT DISTINCT user_id, user_login FROM {$table_name} ORDER BY user_login";
		$results = $wpdb->get_results($sql, ARRAY_A);

		return $results ?: [];
	}

	/**
	 * Format database results to match ActivityLogger format
	 *
	 * @param array $results Database results
	 * @return array Formatted results
	 */
	private function format_results(array $results): array {
		if (empty($results)) {
			return [];
		}

		return array_map(function($row) {
			// Convert DATETIME to Unix timestamp
			$row['timestamp'] = strtotime($row['timestamp']);

			// Deserialize values
			$row['old_value'] = $this->deserialize_value($row['old_value']);
			$row['new_value'] = $this->deserialize_value($row['new_value']);

			// Convert numeric strings to integers
			$row['id'] = (int) $row['id'];
			$row['user_id'] = (int) $row['user_id'];

			return $row;
		}, $results);
	}

	/**
	 * Serialize value for database storage
	 *
	 * @param mixed $value Value to serialize
	 * @return string Serialized value
	 */
	private function serialize_value($value): string {
		if (is_bool($value)) {
			return $value ? 'true' : 'false';
		}

		if (is_numeric($value)) {
			return (string) $value;
		}

		if (is_array($value) || is_object($value)) {
			return wp_json_encode($value);
		}

		return (string) $value;
	}

	/**
	 * Deserialize value from database
	 *
	 * @param string|null $value Serialized value
	 * @return mixed Deserialized value
	 */
	private function deserialize_value($value) {
		if ($value === null || $value === '') {
			return null;
		}

		// Boolean
		if ($value === 'true') {
			return true;
		}
		if ($value === 'false') {
			return false;
		}

		// Numeric
		if (is_numeric($value)) {
			return strpos($value, '.') !== false ? (float) $value : (int) $value;
		}

		// JSON
		if (substr($value, 0, 1) === '{' || substr($value, 0, 1) === '[') {
			$decoded = json_decode($value, true);
			if (json_last_error() === JSON_ERROR_NONE) {
				return $decoded;
			}
		}

		return $value;
	}
}
