<?php
namespace FlxWoo\Data;

use FlxWoo\Utils\Logger;

if (!defined('ABSPATH')) exit;

/**
 * User context coordinator
 * Delegates to specialized data classes and coordinates context for routes
 */
class UserContext {

  /**
   * Specialized data class instances
   */
  private $cart_data;
  private $checkout_data;
  private $order_data;

  /**
   * Constructor - Initialize data class instances
   */
  public function __construct() {
    $this->cart_data = new CartData();
    $this->checkout_data = new CheckoutData();
    $this->order_data = new OrderData();
  }

  /**
   * Get cart data from WooCommerce
   * Delegates to CartData
   *
   * @return array|null Cart data or null if unavailable
   */
  public function get_cart_data() {
    return $this->cart_data->get_cart_data();
  }

  /**
   * Get checkout data from WooCommerce
   * Delegates to CheckoutData
   *
   * @return array|null Checkout data or null if unavailable
   */
  public function get_checkout_data() {
    return $this->checkout_data->get_checkout_data();
  }

  /**
   * Get logged-in user data
   *
   * Note: User billing/shipping address data is now automatically loaded
   * in CheckoutData::get_checkout_data() for logged-in users.
   *
   * This method is reserved for future user-specific features such as:
   * - Wishlist data
   * - User preferences
   * - Order history
   * - Saved payment methods
   *
   * @return array|null User data or null if not logged in
   */
  public function get_user_data() {
    // Not currently needed - CheckoutData handles user addresses
    // Placeholder for future user-specific features
    return null;
  }

  /**
   * Get order data for the thank you page
   * Delegates to OrderData
   *
   * @param int $order_id The order ID
   * @param string $order_key The order key for validation
   * @return array|null Order data or null if order not found/invalid
   */
  public function get_order_data($order_id, $order_key = '') {
    return $this->order_data->get_order_data($order_id, $order_key);
  }

  /**
   * Get all context data needed for a specific route
   *
   * @param string $route The route being rendered (e.g., '/cart', '/checkout')
   * @return array User context data for the route
   */
  public function get_context_for_route($route) {
    $context = [];

    // Cart data for cart and checkout pages
    if ($route === '/cart' || $route === '/checkout') {
      $cart_data = $this->get_cart_data();
      if ($cart_data !== null) {
        $context['cart'] = $cart_data;
      } else {
        Logger::debug('WooCommerce cart data unavailable when building user context.');
      }
    }

    // Checkout-specific data
    if ($route === '/checkout') {
      $checkout_data = $this->get_checkout_data();
      if ($checkout_data !== null) {
        $context['checkout'] = $checkout_data;
      }
    }

    // Thank you page - order data
    if ($route === '/thank-you') {
      // Get order ID and key from query parameters
      // WooCommerce uses 'order-received' as endpoint, but the ID is in query var
      $order_id = 0;

      // Try to get from WooCommerce query var (used with pretty permalinks)
      if (function_exists('get_query_var')) {
        $order_id = intval(get_query_var('order-received'));
      }

      // Fallback to $_GET parameter
      // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only parameter access for displaying thank-you page. No privileged operations performed. Order key validation in OrderData provides security.
      if ($order_id === 0 && isset($_GET['order-received'])) {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only parameter access for displaying thank-you page. No privileged operations performed. Order key validation in OrderData provides security.
        $order_id = intval($_GET['order-received']);
      }

      // Another fallback to 'order_id' for compatibility
      // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only parameter access for displaying thank-you page. No privileged operations performed. Order key validation in OrderData provides security.
      if ($order_id === 0 && isset($_GET['order_id'])) {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only parameter access for displaying thank-you page. No privileged operations performed. Order key validation in OrderData provides security.
        $order_id = intval($_GET['order_id']);
      }

      // phpcs:ignore WordPress.Security.NonceVerification.Recommended,WordPress.Security.ValidatedSanitizedInput.MissingUnslash -- Read-only parameter access for displaying thank-you page. No privileged operations performed. Order key validation in OrderData provides security.
      $order_key = isset($_GET['key']) ? sanitize_text_field(wp_unslash($_GET['key'])) : '';

      if ($order_id > 0) {
        $order_data = $this->get_order_data($order_id, $order_key);
        if ($order_data !== null) {
          $context['order'] = $order_data;

          // CRITICAL: Clear persistent cart for logged-in users on thank-you page
          // This prevents old cart items from reappearing after checkout
          $this->clear_persistent_cart_for_user($order_id);
        } elseif (defined('WP_DEBUG') && WP_DEBUG) {
          Logger::error('Failed to get order data for order ID: ' . $order_id);
        }
      } else {
        Logger::debug('No valid order ID provided for thank you page');
      }
    }

    // Always include user data if logged in
    $user_data = $this->get_user_data();
    if ($user_data !== null) {
      $context['user'] = $user_data;
    }

    return $context;
  }

  /**
   * Clear cart on thank-you page for both logged-in and guest users
   *
   * WooCommerce stores cart in different places depending on user type:
   *
   * LOGGED-IN USERS:
   * 1. Session storage (temporary, in wp_woocommerce_sessions table)
   * 2. Persistent cart (permanent, in wp_usermeta as _woocommerce_persistent_cart)
   *
   * GUEST USERS:
   * 1. Session storage only (in wp_woocommerce_sessions table, keyed by session cookie)
   *
   * After successful checkout, ALL cart data must be cleared to prevent old items
   * from reappearing when the user returns to the cart page.
   *
   * BUG FIX: Previously only cleared cart for logged-in users, causing guest cart
   * items to reappear after checkout. Now handles both user types.
   *
   * @param int $order_id The order ID (for logging only)
   */
  private function clear_persistent_cart_for_user($order_id) {
    // Verify order exists
    $order = \wc_get_order($order_id);
    if (!$order) {
      return;
    }

    // STEP 1: Empty the active cart in memory and session (for ALL users)
    if (\function_exists('WC') && \WC()->cart) {
      // Clear cart contents (includes persistent cart by default for logged-in users)
      \WC()->cart->empty_cart(true);

      // CRITICAL: Force save empty cart to session
      // This is essential for GUEST users whose cart is stored only in session
      // Without this, the empty state might not persist to database session
      if (\WC()->session) {
        \WC()->session->set('cart', array());
        \WC()->session->set('cart_totals', null);
        \WC()->session->set('applied_coupons', array());
        \WC()->session->set('coupon_discount_totals', array());
        \WC()->session->set('coupon_discount_tax_totals', array());
        \WC()->session->set('removed_cart_contents', array());

        // Force save session data to database
        // This ensures guest cart session is destroyed in wp_woocommerce_sessions table
        \WC()->session->save_data();

        // For extra safety, destroy the session completely
        \WC()->session->destroy_session();
      }
    }

    // STEP 2: Additional cleanup for logged-in users only
    // Guest users don't have persistent cart in user meta
    if (\is_user_logged_in()) {
      $customer_id = \get_current_user_id();

      // Delete all persistent cart data from user meta
      \delete_user_meta($customer_id, '_woocommerce_persistent_cart');
      \delete_user_meta($customer_id, '_woocommerce_persistent_cart_' . \get_current_blog_id());

      // Also clear any session-specific persistent cart keys
      global $wpdb;
      // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query required for LIKE pattern matching. delete_user_meta() doesn't support wildcards. This is a write operation after checkout, so caching doesn't apply.
      $wpdb->query(
        $wpdb->prepare(
          "DELETE FROM {$wpdb->usermeta} WHERE user_id = %d AND meta_key LIKE %s",
          $customer_id,
          '_woocommerce_persistent_cart%'
        )
      );

      if (defined('WP_DEBUG') && WP_DEBUG) {
        Logger::debug("Cleared cart for logged-in user {$customer_id} after order #{$order_id}");
      }
    } else {
      // Guest user
      if (defined('WP_DEBUG') && WP_DEBUG) {
        Logger::debug("Cleared cart for guest user after order #{$order_id}");
      }
    }
  }
}
