<?php
namespace FlxWoo\Data\Traits;

if (!defined('ABSPATH')) exit;

/**
 * Price formatting utilities
 * Provides consistent price normalization and formatting across all data classes
 *
 * @since 2.0.0
 */
trait PriceFormatter {

  /**
   * Normalize WooCommerce monetary amounts to floats with standard decimals
   *
   * Ensures consistent decimal precision across all monetary calculations
   * Uses WooCommerce's configured decimal places for the store
   *
   * @param string|float $amount The amount to normalize
   * @return float Normalized amount with consistent decimal precision
   * @since 2.0.0
   */
  protected function normalize_amount($amount) {
    return (float) wc_format_decimal($amount, wc_get_price_decimals());
  }

  /**
   * Format price with WooCommerce formatting and decode HTML entities
   *
   * Applies store's currency formatting (symbol, position, separators)
   * and ensures clean output without HTML tags or entities
   *
   * @param float $amount The amount to format
   * @return string Formatted price without HTML tags (e.g., "$10.00")
   * @since 2.0.0
   */
  protected function format_price($amount) {
    return html_entity_decode(wp_strip_all_tags(wc_price($amount)), ENT_QUOTES | ENT_HTML5, 'UTF-8');
  }
}
