<?php
namespace FlxWoo\Data;

if (!defined('ABSPATH')) exit;

use FlxWoo\Data\Traits\PriceFormatter;
use FlxWoo\Data\Traits\ImageHelper;
use FlxWoo\Data\Traits\AddressHelper;
use FlxWoo\Data\Traits\WooCommerceValidator;
use FlxWoo\Utils\Logger;

/**
 * Order data extraction and formatting
 * Handles all order-related data operations
 *
 * @since 2.0.0
 */
class OrderData {

  use PriceFormatter;
  use ImageHelper;
  use AddressHelper;
  use WooCommerceValidator;

  /**
   * Get order data for the thank you page
   *
   * Retrieves complete order data including items, addresses, totals, and payment details
   * Validates order key for security before returning data
   *
   * @param int $order_id The order ID
   * @param string $order_key The order key for validation
   * @return array|null Order data or null if order not found/invalid
   * @since 2.0.0
   */
  public function get_order_data($order_id, $order_key = '') {
    // Check if WooCommerce is active
    if (!$this->is_woocommerce_active()) {
      return null;
    }

    // Validate order ID
    if (empty($order_id) || !is_numeric($order_id)) {
      Logger::debug('Invalid order ID provided to get_order_data');
      return null;
    }

    // Get the order
    $order = wc_get_order($order_id);
    if (!$order) {
      if (defined('WP_DEBUG') && WP_DEBUG) {
        Logger::error('Order not found: ' . $order_id);
      }
      return null;
    }

    // Validate order key for security
    if (!empty($order_key) && $order->get_order_key() !== $order_key) {
      if (defined('WP_DEBUG') && WP_DEBUG) {
        Logger::error('Invalid order key for order: ' . $order_id);
      }
      return null;
    }

    // Get order items
    $order_items = [];
    foreach ($order->get_items() as $item_id => $item) {
      $product = $item->get_product();

      // Calculate unit price from total / quantity
      $quantity = $item->get_quantity();
      $unit_price = $quantity > 0 ? $this->normalize_amount($item->get_total() / $quantity) : 0.0;
      $subtotal = $this->normalize_amount($item->get_subtotal());
      $total = $this->normalize_amount($item->get_total());
      $tax = $this->normalize_amount($item->get_total_tax());

      // Get variation attributes for variable products
      $variation_attributes = (object)[]; // Empty object for JSON encoding
      if ($item->get_variation_id() > 0 && $product instanceof \WC_Product_Variation) {
        $attrs = $product->get_variation_attributes();
        $variation_attributes = !empty($attrs) ? $attrs : (object)[];
      }

      $order_items[] = [
        'product_id' => $item->get_product_id(),
        'variation_id' => $item->get_variation_id(),
        'name' => $item->get_name(),
        'quantity' => $quantity,
        'price' => $unit_price,
        'price_formatted' => $this->format_price($unit_price),
        'subtotal' => $subtotal,
        'subtotal_formatted' => $this->format_price($subtotal),
        'total' => $total,
        'total_formatted' => $this->format_price($total),
        'tax' => $tax,
        'tax_formatted' => $this->format_price($tax),
        'image' => $this->get_product_image_data($product ? $product->get_image_id() : 0),
        'permalink' => $product ? $product->get_permalink() : '',
        'variation_attributes' => $variation_attributes,
      ];
    }

    // Get billing and shipping addresses using helper method
    $billing_address = $this->get_address_data($order, 'billing');
    $shipping_address = $this->get_address_data($order, 'shipping');

    // Get payment method details
    $payment_method = $order->get_payment_method();
    $payment_method_title = $order->get_payment_method_title();

    // Get shipping method details
    $shipping_method = '';
    $shipping_method_title = '';
    $shipping_methods = $order->get_shipping_methods();
    if (!empty($shipping_methods)) {
      $first_shipping_method = reset($shipping_methods);
      $shipping_method = $first_shipping_method->get_method_id();
      $shipping_method_title = $first_shipping_method->get_name();
    }

    // Normalize all monetary values
    $subtotal = $this->normalize_amount($order->get_subtotal());
    $shipping_total = $this->normalize_amount($order->get_shipping_total());
    $tax_total = $this->normalize_amount($order->get_total_tax());
    $discount_total = $this->normalize_amount($order->get_discount_total());
    $total = $this->normalize_amount($order->get_total());

    // Return comprehensive order data
    return [
      'order_id' => $order->get_id(),
      'order_number' => $order->get_order_number(),
      'order_key' => $order->get_order_key(),
      'status' => $order->get_status(),
      'date_created' => $order->get_date_created()->date('Y-m-d H:i:s'),
      'date_created_formatted' => $order->get_date_created()->date_i18n(wc_date_format() . ' ' . wc_time_format()),
      'customer_note' => $order->get_customer_note(),
      'billing_address' => $billing_address,
      'shipping_address' => $shipping_address,
      'payment_method' => $payment_method,
      'payment_method_title' => $payment_method_title,
      'shipping_method' => $shipping_method,
      'shipping_method_title' => $shipping_method_title,
      'items' => $order_items,
      'subtotal' => $subtotal,
      'subtotal_formatted' => $this->format_price($subtotal),
      'shipping_total' => $shipping_total,
      'shipping_total_formatted' => $this->format_price($shipping_total),
      'tax_total' => $tax_total,
      'tax_total_formatted' => $this->format_price($tax_total),
      'discount_total' => $discount_total,
      'discount_total_formatted' => $this->format_price($discount_total),
      'total' => $total,
      'total_formatted' => $this->format_price($total),
      'currency' => $order->get_currency(),
      'currency_symbol' => html_entity_decode(get_woocommerce_currency_symbol($order->get_currency()), ENT_QUOTES | ENT_HTML5, 'UTF-8'),
    ];
  }
}
