<?php
namespace FlxWoo\Data;

if (!defined('ABSPATH')) exit;

use FlxWoo\Data\Traits\PriceFormatter;
use FlxWoo\Data\Traits\AddressHelper;
use FlxWoo\Data\Traits\WooCommerceValidator;
use FlxWoo\Utils\Logger;

/**
 * Checkout data extraction and formatting
 * Handles all checkout-related data operations
 *
 * @since 2.0.0
 */
class CheckoutData {

  use PriceFormatter;
  use AddressHelper;
  use WooCommerceValidator;

  /**
   * Get enqueued scripts and styles for payment gateways
   *
   * Captures script and style data after payment_scripts() is called
   * Returns data needed to render script/link tags in Next.js template
   *
   * @return array{scripts: array, styles: array} Enqueued assets
   * @since 2.1.0
   */
  protected function get_enqueued_assets() {
    global $wp_scripts, $wp_styles;

    $scripts = [];
    $styles = [];

    // Get enqueued scripts
    if (isset($wp_scripts->queue) && !empty($wp_scripts->queue)) {
      foreach ($wp_scripts->queue as $handle) {
        if (!isset($wp_scripts->registered[$handle])) {
          continue;
        }

        $script = $wp_scripts->registered[$handle];
        $scripts[$handle] = [
          'handle' => $handle,
          'src' => $script->src,
          'deps' => $script->deps,
          'ver' => $script->ver,
          'in_footer' => isset($script->extra['group']) && $script->extra['group'] === 1,
        ];

        // Capture inline scripts (like Stripe API keys)
        if (isset($script->extra['data'])) {
          $scripts[$handle]['data'] = $script->extra['data'];
        }
        if (isset($script->extra['before'])) {
          $scripts[$handle]['before'] = $script->extra['before'];
        }
        if (isset($script->extra['after'])) {
          $scripts[$handle]['after'] = $script->extra['after'];
        }
      }
    }

    // Get enqueued styles
    if (isset($wp_styles->queue) && !empty($wp_styles->queue)) {
      foreach ($wp_styles->queue as $handle) {
        if (!isset($wp_styles->registered[$handle])) {
          continue;
        }

        $style = $wp_styles->registered[$handle];
        $styles[$handle] = [
          'handle' => $handle,
          'src' => $style->src,
          'deps' => $style->deps,
          'ver' => $style->ver,
          'media' => $style->args ?? 'all',
        ];
      }
    }

    return [
      'scripts' => $scripts,
      'styles' => $styles,
    ];
  }

  /**
   * Get WooCommerce notices (errors, success, info)
   *
   * Captures all active WooCommerce notices and clears them
   * Notices will be displayed in the Next.js rendered page
   *
   * @return array{error: array, success: array, notice: array} Array of notices by type
   * @since 2.0.0
   */
  protected function get_woocommerce_notices() {
    if (!function_exists('wc_get_notices')) {
      return [
        'error' => [],
        'success' => [],
        'notice' => [],
      ];
    }

    $notices = [
      'error' => wc_get_notices('error') ?: [],
      'success' => wc_get_notices('success') ?: [],
      'notice' => wc_get_notices('notice') ?: [],
    ];

    // Clear notices after capturing them
    // They will be displayed in Next.js rendered page
    wc_clear_notices();

    return $notices;
  }

  /**
   * Get minimum order amount data
   *
   * Checks if store has minimum order amount requirement
   * and whether current cart meets the requirement
   *
   * @param \WC_Cart $cart WooCommerce cart instance
   * @return array{enabled: bool, amount: float, amount_formatted: string, meets_minimum: bool, amount_remaining: float, amount_remaining_formatted: string} Minimum order data
   * @since 2.0.0
   */
  protected function get_minimum_order_data($cart) {
    $min_amount = $this->normalize_amount(get_option('woocommerce_min_amount', 0));
    $cart_total = $this->normalize_amount($cart->get_total('edit'));
    $has_minimum_amount = $min_amount > 0;
    $meets_minimum = !$has_minimum_amount || $cart_total >= $min_amount;
    $amount_needed = $has_minimum_amount && !$meets_minimum ? $min_amount - $cart_total : 0;

    return [
      'enabled' => $has_minimum_amount,
      'amount' => $min_amount,
      'amount_formatted' => $this->format_price($min_amount),
      'meets_minimum' => $meets_minimum,
      'amount_remaining' => $amount_needed,
      'amount_remaining_formatted' => $this->format_price($amount_needed),
    ];
  }

  /**
   * Normalize checkout field for consistent structure
   *
   * Ensures consistent field structure for frontend form rendering
   * Converts empty options arrays to objects for proper JSON encoding
   *
   * @param array $field WooCommerce checkout field definition
   * @return array{label: string, placeholder: string, required: bool, autocomplete: string, priority: int, type: string, options: array|object} Normalized field data
   * @since 2.0.0
   */
  protected function normalize_checkout_field($field) {
    // Get options, ensuring they're objects not arrays when empty
    $options = $field['options'] ?? [];

    // Convert empty arrays to empty objects for JSON encoding
    if (empty($options)) {
      $options = (object)[];
    }

    return [
      'label' => $field['label'] ?? '',
      'placeholder' => $field['placeholder'] ?? '',
      'required' => isset($field['required']) && $field['required'],
      'autocomplete' => $field['autocomplete'] ?? '',
      'priority' => $field['priority'] ?? 0,
      'type' => $field['type'] ?? 'text',
      'options' => $options,
    ];
  }

  /**
   * Get checkout data from WooCommerce
   *
   * Retrieves complete checkout data including addresses, payment methods,
   * shipping methods, checkout fields, and validation requirements
   *
   * @return array|null Checkout data or null if unavailable
   * @since 2.0.0
   */
  public function get_checkout_data() {
    // Check if WooCommerce is active
    if (!$this->is_woocommerce_active()) {
      return null;
    }

    // CRITICAL: Initialize WooCommerce session if not already done
    // This is especially important in production/cross-domain environments
    // where session cookies may take longer to propagate
    if (is_null(WC()->session)) {
      WC()->initialize_session();
    }

    // Force session to load from cookie (required for REST API and headless contexts)
    WC()->session->set_customer_session_cookie(true);

    // Initialize cart if needed
    if (is_null(WC()->cart)) {
      wc_load_cart();
    }

    // CRITICAL: Force cart to populate from session
    // The session may have the data, but cart_contents won't be populated
    // unless we explicitly tell the cart to load from session
    WC()->cart->get_cart_from_session();

    // Check if cart exists and is not empty
    $cart = WC()->cart;
    if (!$cart || $cart->is_empty()) {
      // Cart is empty - cannot retrieve checkout data
      // This is normal after order completion or if user hasn't added items yet
      return null;
    }

    // Initialize checkout if needed
    $checkout = WC()->checkout();
    if (!$checkout) {
      return null;
    }

    // Get customer object
    $customer = WC()->customer;
    if (!$customer) {
      return null;
    }

    // For logged-in users, merge saved customer data from database (user meta)
    // This ensures billing/shipping addresses from previous orders are auto-filled
    if (is_user_logged_in()) {
      $user_id = get_current_user_id();

      // Load saved customer data from database
      $saved_customer = new \WC_Customer($user_id);

      // Only merge if saved customer has billing data
      if (!empty($saved_customer->get_billing_first_name())) {
        // Merge billing data from saved customer
        $customer->set_billing_first_name($saved_customer->get_billing_first_name());
        $customer->set_billing_last_name($saved_customer->get_billing_last_name());
        $customer->set_billing_company($saved_customer->get_billing_company());
        $customer->set_billing_address_1($saved_customer->get_billing_address_1());
        $customer->set_billing_address_2($saved_customer->get_billing_address_2());
        $customer->set_billing_city($saved_customer->get_billing_city());
        $customer->set_billing_state($saved_customer->get_billing_state());
        $customer->set_billing_postcode($saved_customer->get_billing_postcode());
        $customer->set_billing_country($saved_customer->get_billing_country());
        $customer->set_billing_phone($saved_customer->get_billing_phone());

        // Merge shipping data from saved customer
        $customer->set_shipping_first_name($saved_customer->get_shipping_first_name());
        $customer->set_shipping_last_name($saved_customer->get_shipping_last_name());
        $customer->set_shipping_company($saved_customer->get_shipping_company());
        $customer->set_shipping_address_1($saved_customer->get_shipping_address_1());
        $customer->set_shipping_address_2($saved_customer->get_shipping_address_2());
        $customer->set_shipping_city($saved_customer->get_shipping_city());
        $customer->set_shipping_state($saved_customer->get_shipping_state());
        $customer->set_shipping_postcode($saved_customer->get_shipping_postcode());
        $customer->set_shipping_country($saved_customer->get_shipping_country());

        // Debug logging if WP_DEBUG is enabled
        if (defined('WP_DEBUG') && WP_DEBUG) {
          Logger::debug('Loaded saved customer data for user ' . $user_id);
          Logger::debug('Billing first name: ' . $saved_customer->get_billing_first_name());
          Logger::debug('Billing address: ' . $saved_customer->get_billing_address_1());
        }
      } else {
        // Debug logging if WP_DEBUG is enabled
        if (defined('WP_DEBUG') && WP_DEBUG) {
          Logger::debug('No saved customer data found for user ' . $user_id);
        }
      }
    }

    // Get billing and shipping addresses using helper method
    $billing_address = $this->get_address_data($customer, 'billing');
    $shipping_address = $this->get_address_data($customer, 'shipping');

    // Get available payment gateways
    $available_gateways = WC()->payment_gateways->get_available_payment_gateways();
    $payment_gateways = [];

    // First, enqueue all payment scripts for all gateways
    // This ensures all scripts are registered before we capture them
    // Note: Global compatibility filter in CompatibilityHooks prevents plugin conflicts
    foreach ($available_gateways as $gateway) {
      // Only call payment_scripts() if the method exists
      // Some gateways may not implement this method
      if (method_exists($gateway, 'payment_scripts')) {
        try {
          $gateway->payment_scripts();
        } catch (\Throwable $e) {
          // Log error but don't break checkout
          // Some gateways may have compatibility issues when called outside normal checkout context
          if (defined('WP_DEBUG') && WP_DEBUG) {
            Logger::debug('Error calling payment_scripts() for gateway ' . $gateway->id . ': ' . $e->getMessage());
          }
        }
      }
    }

    // Now capture payment gateway data including HTML fields
    foreach ($available_gateways as $gateway) {
      // Capture payment gateway HTML output (credit card fields, etc.)
      // Only call payment_fields() if the method exists
      $payment_html = '';
      if (method_exists($gateway, 'payment_fields')) {
        try {
          ob_start();
          $gateway->payment_fields();
          $payment_html = ob_get_clean();
        } catch (\Throwable $e) {
          // Clean output buffer in case of error
          if (ob_get_level() > 0) {
            ob_end_clean();
          }
          // Log error but don't break checkout
          if (defined('WP_DEBUG') && WP_DEBUG) {
            Logger::debug('Error calling payment_fields() for gateway ' . $gateway->id . ': ' . $e->getMessage());
          }
        }
      }

      // Debug: Log if payment HTML was captured
      if (defined('WP_DEBUG') && WP_DEBUG && $gateway->has_fields()) {
        Logger::debug("Payment HTML for {$gateway->id} - Length: " . strlen($payment_html) . " bytes");
        if (empty($payment_html)) {
          Logger::debug("WARNING - {$gateway->id} has_fields=true but payment_html is empty!");
        }
      }

      $payment_gateways[] = [
        'id' => $gateway->id,
        'title' => $gateway->get_title(),
        'description' => $gateway->get_description(),
        'payment_html' => $payment_html, // HTML for payment fields (credit card input, etc.)
        'icon' => $gateway->get_icon(), // Payment gateway icon/logo
        'has_fields' => $gateway->has_fields(), // Whether gateway has custom payment fields
      ];
    }

    // Capture all enqueued scripts and styles after all gateways have registered theirs
    $payment_assets = $this->get_enqueued_assets();

    // Debug: Log captured scripts for payment gateways
    if (defined('WP_DEBUG') && WP_DEBUG) {
      Logger::debug('Captured ' . count($payment_assets['scripts']) . ' payment scripts');
      Logger::debug('Captured ' . count($payment_assets['styles']) . ' payment styles');
      foreach ($payment_assets['scripts'] as $handle => $script) {
        Logger::debug("Script '$handle' - src: " . substr($script['src'], 0, 80));
      }
    }

    // Get available shipping methods
    $shipping_methods = [];
    $packages = WC()->shipping()->get_packages();

    foreach ($packages as $package_key => $package) {
      if (isset($package['rates']) && is_array($package['rates'])) {
        foreach ($package['rates'] as $rate) {
          $shipping_methods[] = [
            'id' => $rate->get_id(),
            'label' => $rate->get_label(),
            'cost_formatted' => $this->format_price($rate->get_cost()),
          ];
        }
      }
    }

    // Get checkout fields
    $billing_fields = [];
    $shipping_fields = [];

    $checkout_fields = $checkout->get_checkout_fields();

    if (isset($checkout_fields['billing'])) {
      foreach ($checkout_fields['billing'] as $key => $field) {
        $billing_fields[$key] = $this->normalize_checkout_field($field);
      }
    }

    if (isset($checkout_fields['shipping'])) {
      foreach ($checkout_fields['shipping'] as $key => $field) {
        $shipping_fields[$key] = $this->normalize_checkout_field($field);
      }
    }

    // Get terms and privacy policy
    $terms_url = wc_terms_and_conditions_page_id() ? get_permalink(wc_terms_and_conditions_page_id()) : '';
    $privacy_url = wc_privacy_policy_page_id() ? get_permalink(wc_privacy_policy_page_id()) : '';

    // Get selected payment method
    $selected_payment_method = WC()->session->get('chosen_payment_method', '');
    if (empty($selected_payment_method) && !empty($available_gateways)) {
      $selected_payment_method = key($available_gateways);
    }

    // Get selected shipping method
    $chosen_methods = WC()->session->get('chosen_shipping_methods', []);
    $selected_shipping_method = !empty($chosen_methods) ? $chosen_methods[0] : '';

    // Note: $cart variable already retrieved and validated at beginning of function
    $order_total = $this->normalize_amount($cart->get_total('edit'));

    // Generate nonce for WooCommerce Store API checkout
    $store_api_nonce = wp_create_nonce('wc_store_api');

    return [
      'customer_email' => $customer->get_billing_email(),
      'customer_note' => '',
      'billing_address' => $billing_address,
      'shipping_address' => $shipping_address,
      'ship_to_different_address' => !empty($shipping_address['address_1']),
      'payment_gateways' => $payment_gateways,
      'selected_payment_method' => $selected_payment_method,
      'shipping_methods' => $shipping_methods,
      'selected_shipping_method' => $selected_shipping_method,
      'needs_shipping' => $cart->needs_shipping(),
      'billing_fields' => $billing_fields,
      'shipping_fields' => $shipping_fields,
      'terms_url' => $terms_url,
      'privacy_policy_url' => $privacy_url,
      'order_total' => $order_total,
      'order_total_formatted' => $this->format_price($order_total),
      'notices' => $this->get_woocommerce_notices(),
      'minimum_order' => $this->get_minimum_order_data($cart),
      'store_api_nonce' => $store_api_nonce, // Required for Store API checkout
      'payment_assets' => $payment_assets, // Scripts and styles for payment gateways
    ];
  }
}
