<?php
namespace FlxWoo\Data;

if (!defined('ABSPATH')) exit;

use FlxWoo\Data\Traits\PriceFormatter;
use FlxWoo\Data\Traits\ImageHelper;
use FlxWoo\Data\Traits\WooCommerceValidator;

/**
 * Cart data extraction and formatting
 * Handles all cart-related data operations
 *
 * @since 2.0.0
 */
class CartData {

  use PriceFormatter;
  use ImageHelper;
  use WooCommerceValidator;

  /**
   * Configuration constants
   */
  const CROSS_SELL_LIMIT = 4;
  const DEFAULT_LOW_STOCK_THRESHOLD = 2;

  /**
   * Get cart data from WooCommerce
   *
   * Retrieves complete cart data including items, totals, coupons, fees, and cross-sells
   * Returns null if WooCommerce is not available
   *
   * @return array|null Cart data or null if unavailable
   * @since 2.0.0
   */
  public function get_cart_data() {
    // Check if WooCommerce is active
    if (!$this->is_woocommerce_active()) {
      return null;
    }

    // CRITICAL: Initialize WooCommerce session if not already done
    // This is especially important in production/cross-domain environments
    // where session cookies may take longer to propagate
    if (is_null(WC()->session)) {
      WC()->initialize_session();
    }

    // Force session to load from cookie (required for REST API and headless contexts)
    WC()->session->set_customer_session_cookie(true);

    // Initialize cart if needed
    // wc_load_cart() already calls get_cart_from_session() internally
    if (is_null(WC()->cart)) {
      wc_load_cart();
    }

    $cart = WC()->cart;

    // Cache configuration options to avoid repeated DB queries
    $low_stock_threshold = (int) get_option('woocommerce_notify_low_stock_amount', self::DEFAULT_LOW_STOCK_THRESHOLD);

    // Get cart items
    $cart_items = [];
    foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
      $product = $cart_item['data'];

      // Normalize monetary values for calculations
      $product_price = $this->normalize_amount($product->get_price());
      $line_subtotal = $this->normalize_amount($cart_item['line_subtotal']);

      // Get variation attributes for variable products
      $variation_attributes = (object)[]; // Empty object for JSON encoding
      if ($cart_item['variation_id'] > 0 && $product instanceof \WC_Product_Variation) {
        $attrs = $product->get_variation_attributes();
        $variation_attributes = !empty($attrs) ? $attrs : (object)[];
      }

      // Get stock status
      $stock_quantity = $product->get_stock_quantity();
      $is_low_stock = $product->managing_stock() && $stock_quantity !== null && $stock_quantity <= $low_stock_threshold;

      $cart_items[] = [
        'key' => $cart_item_key,
        'product_id' => $cart_item['product_id'],
        'variation_id' => $cart_item['variation_id'],
        'quantity' => $cart_item['quantity'],
        'name' => $product->get_name(),
        'price' => $product_price,
        'subtotal' => $line_subtotal,
        'price_formatted' => $this->format_price($product_price),
        'subtotal_formatted' => $this->format_price($line_subtotal),
        'image' => $this->get_product_image_data($product->get_image_id()),
        'permalink' => $product->get_permalink(),
        'variation_attributes' => $variation_attributes,
        'stock_status' => $product->get_stock_status(), // 'instock', 'outofstock', 'onbackorder'
        'stock_quantity' => $stock_quantity,
        'is_low_stock' => $is_low_stock,
        'backorders_allowed' => $product->backorders_allowed(),
      ];
    }

    $cart_contents_tax = $this->normalize_amount($cart->get_cart_contents_tax());
    $shipping_tax = method_exists($cart, 'get_shipping_tax_total')
      ? $this->normalize_amount($cart->get_shipping_tax_total())
      : 0.0;
    $tax_total = $cart_contents_tax + $shipping_tax;
    $subtotal = $this->normalize_amount($cart->get_cart_contents_total());
    $total = $this->normalize_amount($cart->get_total('edit'));
    $shipping_total = $this->normalize_amount($cart->get_shipping_total());
    $discount_total = $this->normalize_amount($cart->get_discount_total());

    // Generate nonce for WooCommerce Store API cart operations
    $store_api_nonce = wp_create_nonce('wc_store_api');

    // Return cart data with normalized monetary values
    return [
      'items' => $cart_items,
      'item_count' => $cart->get_cart_contents_count(),
      'subtotal' => $subtotal,
      'subtotal_formatted' => $this->format_price($subtotal),
      'total' => $total,
      'total_formatted' => $this->format_price($total),
      'tax_total' => $tax_total,
      'tax_total_formatted' => $this->format_price($tax_total),
      'shipping_total' => $shipping_total,
      'shipping_total_formatted' => $this->format_price($shipping_total),
      'discount_total' => $discount_total,
      'discount_total_formatted' => $this->format_price($discount_total),
      'applied_coupons' => $this->get_applied_coupons($cart),
      'fees' => $this->get_cart_fees($cart),
      'cross_sells' => $this->get_cross_sells($cart),
      'is_empty' => $cart->is_empty(),
      'needs_shipping' => $cart->needs_shipping(),
      'currency' => get_woocommerce_currency(),
      'currency_symbol' => html_entity_decode(get_woocommerce_currency_symbol(), ENT_QUOTES | ENT_HTML5, 'UTF-8'),
      'store_api_nonce' => $store_api_nonce, // Required for Store API cart operations
    ];
  }

  /**
   * Get applied coupons from cart
   *
   * @param \WC_Cart $cart WooCommerce cart instance
   * @return array Array of applied coupon data
   * @since 2.0.0
   */
  protected function get_applied_coupons($cart) {
    $applied_coupons = [];

    foreach ($cart->get_applied_coupons() as $coupon_code) {
      $coupon = new \WC_Coupon($coupon_code);
      $discount_amount = $this->normalize_amount($cart->get_coupon_discount_amount($coupon_code));

      $applied_coupons[] = [
        'code' => $coupon_code,
        'amount' => $discount_amount,
        'amount_formatted' => $this->format_price($discount_amount),
        'discount_type' => $coupon->get_discount_type(), // 'percent', 'fixed_cart', 'fixed_product'
      ];
    }

    return $applied_coupons;
  }

  /**
   * Get cart fees
   *
   * @param \WC_Cart $cart WooCommerce cart instance
   * @return array Array of fee data
   * @since 2.0.0
   */
  protected function get_cart_fees($cart) {
    $fees = [];

    foreach ($cart->get_fees() as $fee) {
      $amount = $this->normalize_amount($fee->amount);
      $tax = $this->normalize_amount($fee->tax);

      $fees[] = [
        'id' => $fee->id,
        'name' => $fee->name,
        'amount' => $amount,
        'amount_formatted' => $this->format_price($amount),
        'taxable' => $fee->taxable,
        'tax' => $tax,
        'tax_formatted' => $this->format_price($tax),
      ];
    }

    return $fees;
  }

  /**
   * Get cross-sell products from cart
   *
   * @param \WC_Cart $cart WooCommerce cart instance
   * @return array Array of cross-sell product data
   * @since 2.0.0
   */
  protected function get_cross_sells($cart) {
    $cross_sells = [];

    // Get cross-sell product IDs from cart items
    $cross_sell_ids = array_filter(array_map('wc_get_product', $cart->get_cross_sells()));

    // Limit to configured number of cross-sells for performance
    $cross_sell_products = array_slice($cross_sell_ids, 0, self::CROSS_SELL_LIMIT);

    foreach ($cross_sell_products as $product) {
      if (!$product) continue;

      $price = $this->normalize_amount($product->get_price());

      $cross_sells[] = [
        'product_id' => $product->get_id(),
        'name' => $product->get_name(),
        'price' => $price,
        'price_formatted' => $this->format_price($price),
        'permalink' => $product->get_permalink(),
        'image' => $this->get_product_image_data($product->get_image_id()),
      ];
    }

    return $cross_sells;
  }
}
