<?php
namespace FlxWoo\Cors;

if (!defined('ABSPATH')) exit;

/**
 * CORS Handler
 *
 * Determines which origins are allowed to make cross-origin requests.
 * Strategy: FLX_WOO_RENDERER_URL is the single source of truth.
 */
class CorsHandler {

  /**
   * Get allowed origin based on configuration
   *
   * Priority:
   * 1. FLX_WOO_RENDERER_URL constant - The configured Next.js renderer (primary)
   * 2. WP_DEBUG mode - Auto-allows localhost, .local for development
   * 3. WordPress filter - For advanced multi-tenant scenarios only
   *
   * @return string The allowed origin URL, or empty string if not allowed
   */
  public function get_allowed_origin() {
    if (!isset($_SERVER['HTTP_ORIGIN'])) {
      return '';
    }

    $origin = sanitize_text_field(wp_unslash($_SERVER['HTTP_ORIGIN']));

    // Primary method: Check if origin matches FLX_WOO_RENDERER_URL
    if (defined('FLX_WOO_RENDERER_URL')) {
      $renderer_url = FLX_WOO_RENDERER_URL;
      $renderer_host = wp_parse_url($renderer_url, PHP_URL_HOST);
      $origin_host = wp_parse_url($origin, PHP_URL_HOST);

      if ($renderer_host && $origin_host && $renderer_host === $origin_host) {
        return $origin;
      }
    }

    // Development mode: auto-allow localhost and .local domains
    // Security: Use exact host matching to prevent bypass attacks
    if (defined('WP_DEBUG') && WP_DEBUG) {
      $origin_host = wp_parse_url($origin, PHP_URL_HOST);
      if (!$origin_host) {
        return '';
      }

      // Exact match for localhost and loopback IPs
      $dev_hosts = [
        'localhost',
        '127.0.0.1',
        '::1'
      ];

      if (in_array($origin_host, $dev_hosts, true)) {
        return $origin;
      }

      // Check if host ends with .local (for local development domains)
      // Must end with .local to prevent bypass via localhost.evil.com
      if (substr($origin_host, -6) === '.local') {
        return $origin;
      }
    }

    // Advanced filter for edge cases (multi-tenant, staging, etc.)
    // Most users will never need this
    $filtered_origin = apply_filters('flx_woo_allowed_origin', '', $origin);
    if (!empty($filtered_origin)) {
      // Security: Validate filtered origin is a valid URL
      $parsed = wp_parse_url($filtered_origin);
      if ($parsed && isset($parsed['scheme']) && isset($parsed['host'])) {
        return $filtered_origin;
      }
      if (defined('WP_DEBUG') && WP_DEBUG) {
        Logger::error('Invalid origin from flx_woo_allowed_origin filter: ' . $filtered_origin);
      }
    }

    return '';
  }

  /**
   * Get CORS headers array
   *
   * @param string $origin The allowed origin URL
   * @return array Associative array of CORS headers
   */
  public function get_cors_headers($origin) {
    return [
      'Access-Control-Allow-Origin' => $origin,
      'Access-Control-Allow-Credentials' => 'true',
      'Access-Control-Allow-Methods' => 'GET, POST, PUT, DELETE, OPTIONS',
      'Access-Control-Allow-Headers' => 'Content-Type, Authorization, X-WP-Nonce',
      'Access-Control-Expose-Headers' => 'X-WP-Total, X-WP-TotalPages',
      'Access-Control-Max-Age' => '3600',
    ];
  }

  /**
   * Send CORS headers
   *
   * @param string $origin The allowed origin URL
   */
  public function send_cors_headers($origin) {
    $headers = $this->get_cors_headers($origin);
    foreach ($headers as $key => $value) {
      header("$key: $value");
    }
  }
}
