<?php
/**
 * Plugin Compatibility Reporter
 *
 * Automatically detects installed plugins and reports compatibility to Next.js API.
 *
 * @package FlxWoo
 * @version 2.4.0
 */

namespace FlxWoo\Compatibility;

use FlxWoo\Constants\Constants;

/**
 * Class Reporter
 *
 * Handles automatic plugin compatibility detection and reporting.
 */
class Reporter {
    /**
     * Get store ID (SHA-256 hash of home URL)
     *
     * @return string 16-character hex string
     */
    private function get_store_id(): string {
        $home_url = get_home_url();
        $hash     = hash( 'sha256', $home_url );
        return substr( $hash, 0, 16 );
    }

    /**
     * Get all active plugins
     *
     * @return array Array of plugin info
     */
    public function get_active_plugins(): array {
        if ( ! function_exists( 'get_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $all_plugins    = get_plugins();
        $active_plugins = get_option( 'active_plugins', [] );
        $plugins        = [];

        foreach ( $active_plugins as $plugin_file ) {
            if ( isset( $all_plugins[ $plugin_file ] ) ) {
                $plugin_data = $all_plugins[ $plugin_file ];
                $plugin_slug = dirname( $plugin_file );

                // Handle single-file plugins
                if ( $plugin_slug === '.' ) {
                    $plugin_slug = basename( $plugin_file, '.php' );
                }

                $plugins[] = [
                    'slug'           => $plugin_slug,
                    'name'           => $plugin_data['Name'],
                    'version'        => $plugin_data['Version'],
                    'active'         => true,
                    'author'         => $plugin_data['Author'] ?? null,
                    'description'    => $plugin_data['Description'] ?? null,
                    'network_active' => is_multisite() && is_plugin_active_for_network( $plugin_file ),
                ];
            }
        }

        return $plugins;
    }

    /**
     * Test FlxWoo rendering on a page
     *
     * @param string $page_type Page type: 'cart', 'checkout', 'thank-you'
     * @return array Test result
     */
    private function test_page_rendering( string $page_type ): array {
        $start_time = microtime( true );

        try {
            // Simple test: Check if renderer URL is accessible
            $renderer_url = FLX_WOO_RENDERER_URL;
            $timeout      = FLX_WOO_RENDERER_TIMEOUT;

            // Test basic connectivity
            $response = wp_remote_get(
                $renderer_url . '/api/health',
                [
                    'timeout' => $timeout,
                    'headers' => [
                        'User-Agent' => 'FlxWoo-Compatibility/' . Constants::VERSION,
                    ],
                ]
            );

            if ( is_wp_error( $response ) ) {
                return [
                    'success'       => false,
                    'error_message' => $response->get_error_message(),
                    'render_time_ms' => round( ( microtime( true ) - $start_time ) * 1000 ),
                ];
            }

            $status_code = wp_remote_retrieve_response_code( $response );

            $result = [
                'success'        => $status_code === 200,
                'render_time_ms' => round( ( microtime( true ) - $start_time ) * 1000 ),
            ];

            // Only include error_message if there was an error
            if ( $status_code !== 200 ) {
                $result['error_message'] = "HTTP $status_code";
            }

            return $result;
        } catch ( \Exception $e ) {
            return [
                'success'       => false,
                'error_message' => $e->getMessage(),
                'render_time_ms' => round( ( microtime( true ) - $start_time ) * 1000 ),
            ];
        }
    }

    /**
     * Run compatibility tests
     *
     * @return array Test results for all pages
     */
    public function run_compatibility_tests(): array {
        return [
            'cart_page'      => $this->test_page_rendering( 'cart' ),
            'checkout_page'  => $this->test_page_rendering( 'checkout' ),
            'thank_you_page' => $this->test_page_rendering( 'thank-you' ),
            'rest_api'       => $this->test_rest_api(),
        ];
    }

    /**
     * Test REST API endpoints
     *
     * @return array Test result
     */
    private function test_rest_api(): array {
        $start_time = microtime( true );

        try {
            // Test site-info endpoint
            $response = rest_do_request( '/flx-woo/v1/site-info' );

            if ( is_wp_error( $response ) ) {
                return [
                    'success'       => false,
                    'error_message' => $response->get_error_message(),
                    'render_time_ms' => round( ( microtime( true ) - $start_time ) * 1000 ),
                ];
            }

            $result = [
                'success'        => $response->get_status() === 200,
                'render_time_ms' => round( ( microtime( true ) - $start_time ) * 1000 ),
            ];

            // Only include error_message if there was an error
            if ( $response->get_status() !== 200 ) {
                $result['error_message'] = 'REST API error';
            }

            return $result;
        } catch ( \Exception $e ) {
            return [
                'success'       => false,
                'error_message' => $e->getMessage(),
                'render_time_ms' => round( ( microtime( true ) - $start_time ) * 1000 ),
            ];
        }
    }

    /**
     * Submit compatibility report to Next.js API
     *
     * @return array|WP_Error Response or error
     */
    public function submit_compatibility_report() {
        // Get active plugins
        $plugins = $this->get_active_plugins();

        if ( empty( $plugins ) ) {
            return new \WP_Error( 'no_plugins', 'No active plugins found' );
        }

        // Run compatibility tests
        $base_test_results = $this->run_compatibility_tests();

        // Build test results per plugin
        // For now, we assume the same test results for all plugins
        // In a more advanced implementation, we could test with each plugin individually
        $test_results = [];
        foreach ( $plugins as $plugin ) {
            $test_results[ $plugin['slug'] ] = $base_test_results;
        }

        // Build request payload
        $payload = [
            'store_id'     => $this->get_store_id(),
            'plugins'      => $plugins,
            'test_results' => $test_results,
        ];

        // Submit to Next.js API
        $renderer_url = FLX_WOO_RENDERER_URL;
        $api_url      = $renderer_url . '/api/v1/compatibility/report';
        $timeout      = FLX_WOO_RENDERER_TIMEOUT;

        $response = wp_remote_post(
            $api_url,
            [
                'timeout' => $timeout,
                'headers' => [
                    'Content-Type' => 'application/json',
                    'User-Agent'   => 'FlxWoo-Compatibility/' . Constants::VERSION,
                ],
                'body'    => wp_json_encode( $payload ),
            ]
        );

        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code( $response );
        $body        = wp_remote_retrieve_body( $response );
        $data        = json_decode( $body, true );

        if ( $status_code !== 200 ) {
            return new \WP_Error(
                'api_error',
                sprintf( 'API returned status code %d', $status_code ),
                $data
            );
        }

        return $data;
    }

    /**
     * Get compatibility info for installed plugins from API
     *
     * @return array Array of plugin compatibility info
     */
    public function get_installed_plugins_compatibility(): array {
        $plugins      = $this->get_active_plugins();
        $renderer_url = FLX_WOO_RENDERER_URL;
        $timeout      = FLX_WOO_RENDERER_TIMEOUT;
        $results      = [];

        foreach ( $plugins as $plugin ) {
            $api_url = $renderer_url . '/api/v1/compatibility/' . urlencode( $plugin['slug'] );

            $response = wp_remote_get(
                $api_url,
                [
                    'timeout' => $timeout,
                    'headers' => [
                        'User-Agent' => 'FlxWoo-Compatibility/' . Constants::VERSION,
                    ],
                ]
            );

            if ( is_wp_error( $response ) ) {
                $results[] = [
                    'plugin'       => $plugin,
                    'compatibility' => null,
                    'error'        => $response->get_error_message(),
                ];
                continue;
            }

            $status_code = wp_remote_retrieve_response_code( $response );
            $body        = wp_remote_retrieve_body( $response );
            $data        = json_decode( $body, true );

            if ( $status_code === 200 ) {
                $results[] = [
                    'plugin'        => $plugin,
                    'compatibility' => $data,
                    'error'         => null,
                ];
            } else {
                $results[] = [
                    'plugin'        => $plugin,
                    'compatibility' => null,
                    'error'         => $data['error'] ?? 'Unknown error',
                ];
            }
        }

        return $results;
    }

    /**
     * Schedule automatic compatibility reporting
     *
     * Run once per week to keep compatibility database updated
     */
    public function schedule_auto_reporting(): void {
        if ( ! wp_next_scheduled( 'flx_woo_compatibility_report' ) ) {
            wp_schedule_event( time(), 'weekly', 'flx_woo_compatibility_report' );
        }
    }

    /**
     * Unschedule automatic reporting
     */
    public function unschedule_auto_reporting(): void {
        $timestamp = wp_next_scheduled( 'flx_woo_compatibility_report' );
        if ( $timestamp ) {
            wp_unschedule_event( $timestamp, 'flx_woo_compatibility_report' );
        }
    }
}
