<?php
namespace FlxWoo;

if (!defined('ABSPATH')) exit;

require_once __DIR__ . '/Constants/Constants.php';
require_once __DIR__ . '/Utils/Logger.php';
require_once __DIR__ . '/Utils/RateLimiter.php';
require_once __DIR__ . '/Data/Traits/PriceFormatter.php';
require_once __DIR__ . '/Data/Traits/ImageHelper.php';
require_once __DIR__ . '/Data/Traits/AddressHelper.php';
require_once __DIR__ . '/Data/Traits/WooCommerceValidator.php';
require_once __DIR__ . '/Data/CartData.php';
require_once __DIR__ . '/Data/CheckoutData.php';
require_once __DIR__ . '/Data/OrderData.php';
require_once __DIR__ . '/Data/UserContext.php';
require_once __DIR__ . '/Hooks/RenderHooks.php';
require_once __DIR__ . '/Hooks/RestHooks.php';
require_once __DIR__ . '/Hooks/RateLimitHooks.php';
require_once __DIR__ . '/Hooks/CorsHooks.php';
require_once __DIR__ . '/Hooks/CompatibilityHooks.php';
require_once __DIR__ . '/Hooks/StripeCompatibilityHooks.php';
require_once __DIR__ . '/Hooks/AnalyticsHooks.php';
require_once __DIR__ . '/Analytics/EventTracker.php';
require_once __DIR__ . '/Analytics/AggregationScheduler.php';
require_once __DIR__ . '/Renderer/HeadlessRender.php';
require_once __DIR__ . '/Rest/Traits/ResponseFormatter.php';
require_once __DIR__ . '/Rest/Endpoints/SiteEndpoints.php';
require_once __DIR__ . '/Rest/RestEndpoints.php';
require_once __DIR__ . '/Cors/CorsHandler.php';

// Feature flags (loaded before admin classes that depend on it)
require_once __DIR__ . '/FeatureFlags/FeatureManager.php';
require_once __DIR__ . '/FeatureFlags/ActivityLogger.php';
require_once __DIR__ . '/FeatureFlags/RetentionManager.php';

// Database classes (loaded before classes that depend on it)
require_once __DIR__ . '/Database/Migrator.php';
require_once __DIR__ . '/Database/ActivityRepository.php';

// Compatibility classes (loaded before admin classes that depend on it)
require_once __DIR__ . '/Compatibility/Reporter.php';

// Admin classes (only loaded in admin)
if (is_admin()) {
  require_once __DIR__ . '/Admin/SettingsManager.php';
  require_once __DIR__ . '/Admin/PerformanceDashboard.php';
  require_once __DIR__ . '/Admin/FeatureFlagsPage.php';
  require_once __DIR__ . '/Admin/ActivityAnalyticsPage.php';
  require_once __DIR__ . '/Admin/BenchmarkingPage.php';
  require_once __DIR__ . '/Admin/ABTestingPage.php';
  require_once __DIR__ . '/Admin/CompatibilityPage.php';
  require_once __DIR__ . '/Admin/AdminHooks.php';
}

use FlxWoo\Hooks\RenderHooks;
use FlxWoo\Hooks\RestHooks;
use FlxWoo\Hooks\RateLimitHooks;
use FlxWoo\Hooks\CorsHooks;
use FlxWoo\Hooks\CompatibilityHooks;
use FlxWoo\Hooks\StripeCompatibilityHooks;
use FlxWoo\Hooks\AnalyticsHooks;
use FlxWoo\Analytics\AggregationScheduler;
use FlxWoo\Admin\AdminHooks;
use FlxWoo\Database\Migrator;

class Bootstrap {
  public function init() {
    (new RenderHooks())->init();
    (new RestHooks())->init();
    (new RateLimitHooks())->register();
    (new CorsHooks())->init();
    (new CompatibilityHooks())->init();
    (new StripeCompatibilityHooks())->init();
    (new AnalyticsHooks())->register();

    // Initialize aggregation scheduler (v2.3.0)
    (new AggregationScheduler())->init();

    // Initialize admin features
    if (is_admin()) {
      // Ensure database tables exist (auto-create if missing)
      $this->ensure_database_tables();

      (new AdminHooks())->init();
    }
  }

  /**
   * Ensure database tables exist (auto-create if missing)
   *
   * This method runs on admin_init to automatically create the activity log
   * table if it doesn't exist. This handles upgrade scenarios where sites
   * updated from older versions without reactivating the plugin.
   *
   * Uses transient caching to avoid repeated database checks (24 hours).
   *
   * @since 2.4.1
   * @return void
   */
  private function ensure_database_tables(): void {
    // Check transient cache first (24 hours)
    $tables_verified = \get_transient('flx_woo_db_tables_verified');

    if ($tables_verified === 'yes') {
      return; // Already verified recently
    }

    // Check if table exists and needs migration
    if (!Migrator::is_migrated()) {
      // Create table
      $created = Migrator::create_table();

      if ($created) {
        // Migrate data from options table if any exists
        $migration_result = Migrator::migrate_from_options();

        // Log success
        if ($migration_result['success']) {
          $migrated_count = $migration_result['migrated_count'] ?? 0;
          \error_log(sprintf(
            'FlxWoo: Activity log table created successfully. Migrated %d entries from options table.',
            $migrated_count
          ));
        } else {
          \error_log('FlxWoo: Activity log table created but migration encountered issues: ' .
                    ($migration_result['error'] ?? 'Unknown error'));
        }

        // Cache verification for 24 hours on success
        \set_transient('flx_woo_db_tables_verified', 'yes', DAY_IN_SECONDS);
      } else {
        // Log error - don't cache failure so it retries
        \error_log('FlxWoo: Failed to create activity log table. Will retry on next admin page load.');
      }
    } else {
      // Table exists and is migrated - cache verification
      \set_transient('flx_woo_db_tables_verified', 'yes', DAY_IN_SECONDS);
    }
  }
}
