<?php
/**
 * Site Registration
 *
 * Handles automatic site registration with FlxWoo SaaS.
 * Generates unique API key and registers site on plugin activation.
 *
 * @package FlxWoo
 * @version 2.4.0
 */

namespace FlxWoo\Analytics;

use FlxWoo\Constants\Constants;

/**
 * SiteRegistration class
 *
 * Auto-generates and manages analytics API keys for SaaS multi-tenant model.
 */
class SiteRegistration {
	/**
	 * Database option name for API key
	 */
	const API_KEY_OPTION = 'flxwoo_analytics_api_key';

	/**
	 * Database option name for registration status
	 */
	const REGISTRATION_STATUS_OPTION = 'flxwoo_site_registration_status';

	/**
	 * Register site on plugin activation
	 * Auto-generates API key and registers with FlxWoo SaaS
	 */
	public static function register_on_activation(): void {
		// Check if API key already exists
		$existing_key = get_option( self::API_KEY_OPTION, '' );

		if ( ! empty( $existing_key ) ) {
			// API key already exists, no need to generate a new one
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( '[FlxWoo] API key already exists. Skipping auto-generation.' );
			}
			return;
		}

		// Generate unique API key
		$api_key = self::generate_api_key();

		// Store API key in database
		update_option( self::API_KEY_OPTION, $api_key );

		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( '[FlxWoo] Generated new API key: ' . substr( $api_key, 0, 16 ) . '...' );
		}

		// Register with SaaS (async, non-blocking)
		self::register_with_saas( $api_key );
	}

	/**
	 * Generate secure random API key
	 *
	 * @return string 64-character hexadecimal API key
	 */
	private static function generate_api_key(): string {
		// Generate 32 random bytes (256 bits)
		$random_bytes = random_bytes( 32 );

		// Convert to hexadecimal string (64 characters)
		return bin2hex( $random_bytes );
	}

	/**
	 * Get site ID (SHA-256 hash of site URL)
	 *
	 * @return string 16-character hexadecimal site ID
	 */
	public static function get_site_id(): string {
		$home_url = home_url();

		// Generate SHA-256 hash of site URL
		$hash = hash( 'sha256', $home_url );

		// Return first 16 characters (64 bits, sufficient for millions of sites)
		return substr( $hash, 0, 16 );
	}

	/**
	 * Register site with FlxWoo SaaS
	 *
	 * @param string $api_key Generated API key
	 */
	private static function register_with_saas( string $api_key ): void {
		// Get renderer URL
		$renderer_url = Constants::get( 'FLX_WOO_RENDERER_URL' );

		if ( empty( $renderer_url ) ) {
			// No renderer URL configured, skip registration
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( '[FlxWoo] Renderer URL not configured. Skipping site registration.' );
			}
			return;
		}

		// Prepare registration data
		$site_id      = self::get_site_id();
		$home_url     = home_url();
		$wp_version   = get_bloginfo( 'version' );
		$wc_version   = defined( 'WC_VERSION' ) ? WC_VERSION : null;

		// Get plugin version from main file
		$plugin_data    = get_file_data(
			dirname( dirname( __DIR__ ) ) . '/flx-woo.php',
			array( 'Version' => 'Version' )
		);
		$plugin_version = $plugin_data['Version'] ?? '2.4.0';

		$registration_data = array(
			'site_id'        => $site_id,
			'site_url'       => $home_url,
			'api_key'        => $api_key,
			'wp_version'     => $wp_version,
			'wc_version'     => $wc_version,
			'plugin_version' => $plugin_version,
		);

		// Register with SaaS API
		$response = wp_remote_post(
			$renderer_url . '/api/v1/sites/register',
			array(
				'timeout'  => 10,
				'blocking' => true, // Wait for response
				'headers'  => array(
					'Content-Type' => 'application/json',
				),
				'body'     => wp_json_encode( $registration_data ),
			)
		);

		// Handle response
		if ( is_wp_error( $response ) ) {
			// Registration failed (network error)
			$error_message = $response->get_error_message();

			error_log(
				sprintf(
					'[FlxWoo] Site registration failed: %s',
					$error_message
				)
			);

			update_option(
				self::REGISTRATION_STATUS_OPTION,
				array(
					'status'     => 'failed',
					'error'      => $error_message,
					'timestamp'  => time(),
					'retry_count' => (int) get_option( self::REGISTRATION_STATUS_OPTION )['retry_count'] ?? 0 + 1,
				)
			);

			return;
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		$body        = wp_remote_retrieve_body( $response );
		$result      = json_decode( $body, true );

		if ( $status_code === 200 && ! empty( $result['success'] ) ) {
			// Registration successful
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log(
					sprintf(
						'[FlxWoo] Site registered successfully: %s (ID: %s)',
						$home_url,
						$site_id
					)
				);
			}

			update_option(
				self::REGISTRATION_STATUS_OPTION,
				array(
					'status'        => 'registered',
					'site_id'       => $site_id,
					'registered_at' => time(),
					'message'       => $result['message'] ?? 'Site registered successfully',
				)
			);
		} else {
			// Registration failed (API error)
			error_log(
				sprintf(
					'[FlxWoo] Site registration failed: HTTP %d - %s',
					$status_code,
					$result['error'] ?? 'Unknown error'
				)
			);

			update_option(
				self::REGISTRATION_STATUS_OPTION,
				array(
					'status'     => 'failed',
					'error'      => $result['error'] ?? 'Unknown error',
					'http_code'  => $status_code,
					'timestamp'  => time(),
				)
			);
		}
	}

	/**
	 * Get registration status
	 *
	 * @return array|null Registration status or null if not attempted
	 */
	public static function get_registration_status(): ?array {
		$status = get_option( self::REGISTRATION_STATUS_OPTION, null );
		return $status ? $status : null;
	}

	/**
	 * Check if site is registered
	 *
	 * @return bool True if registered
	 */
	public static function is_registered(): bool {
		$status = self::get_registration_status();
		return $status && $status['status'] === 'registered';
	}

	/**
	 * Get current API key
	 *
	 * @return string|null API key or null if not set
	 */
	public static function get_api_key(): ?string {
		$key = get_option( self::API_KEY_OPTION, '' );
		return ! empty( $key ) ? $key : null;
	}

	/**
	 * Regenerate API key
	 * Generates new key and re-registers with SaaS
	 *
	 * @return string New API key
	 */
	public static function regenerate_api_key(): string {
		// Generate new API key
		$api_key = self::generate_api_key();

		// Update database
		update_option( self::API_KEY_OPTION, $api_key );

		// Re-register with SaaS
		self::register_with_saas( $api_key );

		return $api_key;
	}
}
