<?php
/**
 * Event Tracker
 *
 * Core analytics tracking functionality for FlxWoo.
 * Sends anonymous conversion events to Next.js analytics API.
 *
 * @package FlxWoo
 * @version 2.3.0
 */

namespace FlxWoo\Analytics;

use FlxWoo\Constants\Constants;

/**
 * EventTracker class
 *
 * Handles conversion event tracking with privacy-by-design.
 */
class EventTracker {

	/**
	 * Analytics API endpoint
	 *
	 * @var string
	 */
	private $analytics_endpoint;

	/**
	 * Store identifier (SHA-256 hash)
	 *
	 * @var string
	 */
	private $store_id;

	/**
	 * Request timeout in seconds
	 *
	 * @var int
	 */
	private $timeout;

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->analytics_endpoint = Constants::get( 'FLX_WOO_RENDERER_URL' ) . '/api/v1/analytics/track';
		$this->store_id           = $this->get_store_id();
		$this->timeout            = 2; // 2 second timeout (non-blocking)
	}

	/**
	 * Get store identifier (SHA-256 hash of home URL)
	 *
	 * Privacy: Irreversible hash - cannot determine original URL from hash.
	 *
	 * @return string 16-character hex hash
	 */
	public function get_store_id(): string {
		$home_url = home_url();
		$hash     = hash( 'sha256', $home_url );

		// Return first 16 characters (sufficient for uniqueness)
		return substr( $hash, 0, 16 );
	}

	/**
	 * Track checkout started event
	 *
	 * @param array $args Event arguments.
	 * @return bool Success status
	 */
	public function track_checkout_started( array $args = array() ): bool {
		$event = array(
			'store_id'         => $this->store_id,
			'event_type'       => 'checkout_started',
			'event_timestamp'  => time(),
			'cart_value_cents' => $this->get_cart_value_cents(),
			'item_count'       => $this->get_cart_item_count(),
			'page_load_ms'     => isset( $args['page_load_ms'] ) ? intval( $args['page_load_ms'] ) : null,
			'metadata'         => $this->get_safe_metadata( $args ),
		);

		return $this->send_event( $event );
	}

	/**
	 * Track checkout completed event
	 *
	 * @param int   $order_id WooCommerce order ID.
	 * @param array $args Event arguments.
	 * @return bool Success status
	 */
	public function track_checkout_completed( int $order_id, array $args = array() ): bool {
		$order = wc_get_order( $order_id );

		if ( ! $order ) {
			return false;
		}

		$event = array(
			'store_id'                => $this->store_id,
			'event_type'              => 'checkout_completed',
			'event_timestamp'         => time(),
			'cart_value_cents'        => intval( $order->get_total() * 100 ),
			'item_count'              => $order->get_item_count(),
			'payment_method_category' => $this->get_payment_category( $order->get_payment_method() ),
			'page_load_ms'            => isset( $args['page_load_ms'] ) ? intval( $args['page_load_ms'] ) : null,
			'metadata'                => $this->get_safe_metadata( $args ),
		);

		return $this->send_event( $event );
	}

	/**
	 * Track checkout abandoned event
	 *
	 * @param string $abandonment_step Where checkout was abandoned.
	 * @param array  $args Event arguments.
	 * @return bool Success status
	 */
	public function track_checkout_abandoned( string $abandonment_step = 'unknown', array $args = array() ): bool {
		$event = array(
			'store_id'         => $this->store_id,
			'event_type'       => 'checkout_abandoned',
			'event_timestamp'  => time(),
			'cart_value_cents' => $this->get_cart_value_cents(),
			'item_count'       => $this->get_cart_item_count(),
			'abandonment_step' => $this->sanitize_abandonment_step( $abandonment_step ),
			'page_load_ms'     => isset( $args['page_load_ms'] ) ? intval( $args['page_load_ms'] ) : null,
			'metadata'         => $this->get_safe_metadata( $args ),
		);

		return $this->send_event( $event );
	}

	/**
	 * Send event to Next.js analytics API
	 *
	 * @param array $event Event data.
	 * @return bool Success status
	 */
	private function send_event( array $event ): bool {
		// Filter out null values
		$event = array_filter(
			$event,
			function ( $value ) {
				return $value !== null;
			}
		);

		$response = wp_remote_post(
			$this->analytics_endpoint,
			array(
				'timeout'     => $this->timeout,
				'blocking'    => false, // Non-blocking request
				'headers'     => array(
					'Content-Type'      => 'application/json',
					'X-FlxWoo-Site-ID'  => $this->store_id,
					'X-FlxWoo-Version'  => Constants::VERSION,
				),
				'body'        => wp_json_encode( $event ),
				'data_format' => 'body',
			)
		);

		// Non-blocking requests don't return errors
		// We consider it successful if the request was sent
		return ! is_wp_error( $response );
	}

	/**
	 * Get cart value in cents
	 *
	 * @return int|null Cart total in cents
	 */
	private function get_cart_value_cents(): ?int {
		if ( ! WC()->cart ) {
			return null;
		}

		$total = WC()->cart->get_total( 'edit' );
		return $total > 0 ? intval( $total * 100 ) : null;
	}

	/**
	 * Get cart item count
	 *
	 * @return int|null Number of items in cart
	 */
	private function get_cart_item_count(): ?int {
		if ( ! WC()->cart ) {
			return null;
		}

		return WC()->cart->get_cart_contents_count();
	}

	/**
	 * Get payment method category (generic only, no specific details)
	 *
	 * Privacy: Only returns generic categories ('card', 'paypal', 'cod', 'other')
	 * Never returns specific card types or account IDs.
	 *
	 * @param string $payment_method WooCommerce payment method ID.
	 * @return string Generic payment category
	 */
	private function get_payment_category( string $payment_method ): string {
		// Map payment methods to generic categories
		$category_map = array(
			'stripe'            => 'card',
			'stripe_cc'         => 'card',
			'stripe_upe'        => 'card',
			'square'            => 'card',
			'authorize_net'     => 'card',
			'braintree'         => 'card',
			'paypal'            => 'paypal',
			'paypal_express'    => 'paypal',
			'ppcp-gateway'      => 'paypal',
			'cod'               => 'cod',
			'bacs'              => 'other',
			'cheque'            => 'other',
		);

		return $category_map[ $payment_method ] ?? 'other';
	}

	/**
	 * Sanitize abandonment step
	 *
	 * @param string $step Abandonment step.
	 * @return string Sanitized step
	 */
	private function sanitize_abandonment_step( string $step ): string {
		$valid_steps = array( 'billing', 'shipping', 'payment', 'review', 'unknown' );

		return in_array( $step, $valid_steps, true ) ? $step : 'unknown';
	}

	/**
	 * Get safe metadata (no PII)
	 *
	 * Filters metadata to ensure no PII is included.
	 *
	 * @param array $args Raw arguments.
	 * @return string|null JSON-encoded safe metadata
	 */
	private function get_safe_metadata( array $args ): ?string {
		// Allowed metadata keys (extensibility)
		$allowed_keys = array(
			'source',           // e.g., 'checkout_page', 'cart_page'
			'device_type',      // e.g., 'mobile', 'desktop'
			'browser_category', // e.g., 'chrome', 'safari' (generic only)
			'referrer_type',    // e.g., 'organic', 'direct', 'social'
		);

		$safe_metadata = array();

		foreach ( $allowed_keys as $key ) {
			if ( isset( $args[ $key ] ) ) {
				$safe_metadata[ $key ] = $args[ $key ];
			}
		}

		if ( empty( $safe_metadata ) ) {
			return null;
		}

		return wp_json_encode( $safe_metadata );
	}

	/**
	 * Batch send multiple events
	 *
	 * More efficient than sending events individually.
	 *
	 * @param array $events Array of events.
	 * @return bool Success status
	 */
	public function send_batch( array $events ): bool {
		if ( empty( $events ) ) {
			return false;
		}

		$payload = array(
			'events' => $events,
		);

		$response = wp_remote_post(
			$this->analytics_endpoint,
			array(
				'timeout'     => $this->timeout,
				'blocking'    => false,
				'headers'     => array(
					'Content-Type'      => 'application/json',
					'X-FlxWoo-Site-ID'  => $this->store_id,
					'X-FlxWoo-Version'  => Constants::VERSION,
				),
				'body'        => wp_json_encode( $payload ),
				'data_format' => 'body',
			)
		);

		return ! is_wp_error( $response );
	}
}
