<?php
/**
 * Aggregation Scheduler
 *
 * Schedules daily aggregation of analytics events via WordPress cron.
 * Calls Next.js aggregation API to process raw events into daily statistics.
 *
 * @package FlxWoo
 * @version 2.3.0
 */

namespace FlxWoo\Analytics;

use FlxWoo\Constants\Constants;

/**
 * AggregationScheduler class
 *
 * Manages WordPress cron schedule for daily aggregation.
 */
class AggregationScheduler {
	/**
	 * Cron hook name
	 */
	const CRON_HOOK = 'flxwoo_daily_aggregation';

	/**
	 * Cron schedule name
	 */
	const CRON_SCHEDULE = 'daily';

	/**
	 * Default API key for development/testing
	 * WARNING: Do NOT use this in production!
	 * Set FLX_WOO_ANALYTICS_API_KEY in wp-config.php for production
	 */
	const DEFAULT_DEV_KEY = 'f67ec07f3f9e9dc1ef37065b24ff18a5145d133e4cc921adf2b23427c528b938';

	/**
	 * Initialize scheduler
	 * Registers cron hook and schedules daily event
	 */
	public function init(): void {
		// Register cron hook
		add_action( self::CRON_HOOK, array( $this, 'run_aggregation' ) );

		// Schedule daily aggregation if not already scheduled
		$this->schedule_event();
	}

	/**
	 * Schedule daily aggregation event
	 * Runs at 2:00 AM daily (server time)
	 */
	public function schedule_event(): void {
		// Check if event is already scheduled
		if ( wp_next_scheduled( self::CRON_HOOK ) ) {
			return;
		}

		// Schedule event at 2:00 AM tomorrow
		$timestamp = strtotime( 'tomorrow 02:00:00' );

		wp_schedule_event( $timestamp, self::CRON_SCHEDULE, self::CRON_HOOK );

		// Log scheduling
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log(
				sprintf(
					'[FlxWoo] Aggregation cron scheduled for %s',
					wp_date( 'Y-m-d H:i:s', $timestamp )
				)
			);
		}
	}

	/**
	 * Get API key for aggregation endpoint
	 *
	 * Priority order:
	 * 1. FLX_WOO_ANALYTICS_API_KEY constant (wp-config.php)
	 * 2. flxwoo_analytics_api_key option (database)
	 * 3. DEFAULT_DEV_KEY (development only)
	 *
	 * @return string API key
	 */
	private function get_api_key(): string {
		// Priority 1: wp-config.php constant (recommended for production)
		if ( defined( 'FLX_WOO_ANALYTICS_API_KEY' ) ) {
			$key = FLX_WOO_ANALYTICS_API_KEY;

			// Validate key format (should be 64-character hex)
			if ( preg_match( '/^[a-f0-9]{64}$/', $key ) ) {
				return $key;
			}

			// Log warning for invalid key format
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log(
					'[FlxWoo] Invalid FLX_WOO_ANALYTICS_API_KEY format. ' .
					'Expected 64-character hex string. Using fallback.'
				);
			}
		}

		// Priority 2: Database option (set via admin UI - future feature)
		$option_key = get_option( 'flxwoo_analytics_api_key', '' );
		if ( ! empty( $option_key ) && preg_match( '/^[a-f0-9]{64}$/', $option_key ) ) {
			return $option_key;
		}

		// Priority 3: Default development key
		// Check if we're in production environment
		$is_production = ( defined( 'WP_ENV' ) && WP_ENV === 'production' )
			|| ( defined( 'WP_ENVIRONMENT_TYPE' ) && WP_ENVIRONMENT_TYPE === 'production' );

		if ( $is_production ) {
			// Log critical warning in production
			error_log(
				'[FlxWoo] SECURITY WARNING: Using default analytics API key in production! ' .
				'Set FLX_WOO_ANALYTICS_API_KEY in wp-config.php immediately.'
			);

			// Optionally disable aggregation in production if no key is set
			// Uncomment the following line to enforce explicit configuration:
			// return '';
		}

		// Return default key (development/testing)
		return self::DEFAULT_DEV_KEY;
	}

	/**
	 * Unschedule aggregation event
	 * Called on plugin deactivation
	 */
	public function unschedule_event(): void {
		$timestamp = wp_next_scheduled( self::CRON_HOOK );

		if ( $timestamp ) {
			wp_unschedule_event( $timestamp, self::CRON_HOOK );

			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( '[FlxWoo] Aggregation cron unscheduled' );
			}
		}
	}

	/**
	 * Run aggregation
	 * Calls Next.js aggregation API to process yesterday's events
	 */
	public function run_aggregation(): void {
		// Get aggregation API URL
		$api_url = Constants::get( 'FLX_WOO_RENDERER_URL' ) . '/api/v1/analytics/aggregate';

		// Get yesterday's date (default behavior)
		$yesterday = wp_date( 'Y-m-d', strtotime( '-1 day' ) );

		// Prepare request payload
		$payload = array(
			'date'           => $yesterday,
			'include_global' => true,
		);

		// Make API request
		$response = wp_remote_post(
			$api_url,
			array(
				'timeout'     => 30,
				'blocking'    => true,
				'headers'     => array(
					'Content-Type' => 'application/json',
					'X-API-Key'    => $this->get_api_key(),
				),
				'body'        => wp_json_encode( $payload ),
				'data_format' => 'body',
			)
		);

		// Handle response
		if ( is_wp_error( $response ) ) {
			// Log error
			$this->log_aggregation_error(
				'HTTP request failed: ' . $response->get_error_message(),
				$yesterday
			);
			return;
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		$body        = wp_remote_retrieve_body( $response );
		$result      = json_decode( $body, true );

		// Check response status
		if ( $status_code !== 200 ) {
			$this->log_aggregation_error(
				sprintf(
					'API returned error %d: %s',
					$status_code,
					$result['error'] ?? 'Unknown error'
				),
				$yesterday
			);
			return;
		}

		// Verify success
		if ( empty( $result['success'] ) ) {
			$this->log_aggregation_error(
				'API returned success=false: ' . ( $result['error'] ?? 'Unknown error' ),
				$yesterday
			);
			return;
		}

		// Log success
		$this->log_aggregation_success( $yesterday, $result );

		// Update last aggregation timestamp
		update_option( 'flxwoo_last_aggregation', time() );
	}

	/**
	 * Log aggregation success
	 *
	 * @param string $date Date aggregated.
	 * @param array  $result Aggregation result from API.
	 */
	private function log_aggregation_success( string $date, array $result ): void {
		$message = sprintf(
			'[FlxWoo] Aggregation successful: %s (Stores: %d, Records: %d)',
			$date,
			$result['stores_processed'] ?? 0,
			$result['records_created'] ?? 0
		);

		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( $message );
		}

		// Store success in transient (for admin dashboard display)
		set_transient(
			'flxwoo_aggregation_status',
			array(
				'status'            => 'success',
				'date'              => $date,
				'stores_processed'  => $result['stores_processed'] ?? 0,
				'records_created'   => $result['records_created'] ?? 0,
				'timestamp'         => time(),
			),
			DAY_IN_SECONDS
		);
	}

	/**
	 * Log aggregation error
	 *
	 * @param string $error_message Error message.
	 * @param string $date Date attempted.
	 */
	private function log_aggregation_error( string $error_message, string $date ): void {
		$message = sprintf(
			'[FlxWoo] Aggregation failed: %s (%s)',
			$date,
			$error_message
		);

		error_log( $message );

		// Store error in transient
		set_transient(
			'flxwoo_aggregation_status',
			array(
				'status'    => 'error',
				'date'      => $date,
				'error'     => $error_message,
				'timestamp' => time(),
			),
			DAY_IN_SECONDS
		);
	}

	/**
	 * Manually trigger aggregation (for admin testing)
	 *
	 * @param string|null $date Optional specific date (YYYY-MM-DD).
	 * @return array Aggregation result
	 */
	public function run_manual_aggregation( ?string $date = null ): array {
		$api_url = Constants::get( 'FLX_WOO_RENDERER_URL' ) . '/api/v1/analytics/aggregate';

		$payload = array(
			'date'           => $date ?? wp_date( 'Y-m-d', strtotime( '-1 day' ) ),
			'include_global' => true,
		);

		$response = wp_remote_post(
			$api_url,
			array(
				'timeout'     => 30,
				'blocking'    => true,
				'headers'     => array(
					'Content-Type' => 'application/json',
					'X-API-Key'    => $this->get_api_key(),
				),
				'body'        => wp_json_encode( $payload ),
				'data_format' => 'body',
			)
		);

		if ( is_wp_error( $response ) ) {
			return array(
				'success' => false,
				'error'   => $response->get_error_message(),
			);
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		$body        = wp_remote_retrieve_body( $response );
		$result      = json_decode( $body, true );

		if ( $status_code !== 200 || empty( $result['success'] ) ) {
			return array(
				'success' => false,
				'error'   => $result['error'] ?? 'Unknown error',
				'code'    => $status_code,
			);
		}

		return $result;
	}

	/**
	 * Get last aggregation status
	 *
	 * @return array|null Aggregation status or null if none
	 */
	public function get_last_aggregation_status(): ?array {
		$status = get_transient( 'flxwoo_aggregation_status' );
		return $status ? $status : null;
	}

	/**
	 * Get next scheduled aggregation time
	 *
	 * @return int|null Unix timestamp or null if not scheduled
	 */
	public function get_next_aggregation_time(): ?int {
		$timestamp = wp_next_scheduled( self::CRON_HOOK );
		return $timestamp ? $timestamp : null;
	}

	/**
	 * Get API key configuration status
	 *
	 * Useful for admin dashboard to show warnings if default key is used
	 *
	 * @return array Configuration status
	 */
	public function get_api_key_status(): array {
		$is_production = ( defined( 'WP_ENV' ) && WP_ENV === 'production' )
			|| ( defined( 'WP_ENVIRONMENT_TYPE' ) && WP_ENVIRONMENT_TYPE === 'production' );

		// Check which source is providing the key
		$source = 'default';
		$is_custom = false;

		if ( defined( 'FLX_WOO_ANALYTICS_API_KEY' ) ) {
			$key    = FLX_WOO_ANALYTICS_API_KEY;
			$source = 'wp-config.php (FLX_WOO_ANALYTICS_API_KEY)';

			// Check if it's a custom key (not the default)
			if ( preg_match( '/^[a-f0-9]{64}$/', $key ) && $key !== self::DEFAULT_DEV_KEY ) {
				$is_custom = true;
			}
		} elseif ( ! empty( get_option( 'flxwoo_analytics_api_key', '' ) ) ) {
			$key    = get_option( 'flxwoo_analytics_api_key', '' );
			$source = 'Database option (flxwoo_analytics_api_key)';

			if ( preg_match( '/^[a-f0-9]{64}$/', $key ) && $key !== self::DEFAULT_DEV_KEY ) {
				$is_custom = true;
			}
		}

		// Determine status
		$status  = 'ok';
		$message = 'Custom API key configured';

		if ( ! $is_custom ) {
			if ( $is_production ) {
				$status  = 'critical';
				$message = 'SECURITY RISK: Using default API key in production! Set FLX_WOO_ANALYTICS_API_KEY in wp-config.php.';
			} else {
				$status  = 'warning';
				$message = 'Using default development key. Set custom key in wp-config.php for production.';
			}
			$source = 'Default (hardcoded)';
		}

		return array(
			'status'      => $status,        // 'ok', 'warning', 'critical'
			'message'     => $message,
			'source'      => $source,
			'is_custom'   => $is_custom,
			'is_default'  => ! $is_custom,
			'environment' => $is_production ? 'production' : 'development',
		);
	}
}
