<?php
/**
 * Feature Configuration Form Section Partial
 *
 * Displays main feature flag configuration form with cards for each feature.
 *
 * @package FlxWoo\Admin\Views\Partials
 * @version 2.4.0
 *
 * @var FeatureFlagsPage $this Controller instance (for get_dependent_features method)
 * @var array $flags Feature flag configurations
 * @var array $enabled_features Currently enabled feature names
 * @var bool $kill_switch_active Whether global kill switch is active
 * @var string $privacy_url Privacy policy URL
 */

if (!defined('ABSPATH')) {
	exit;
}

// Validate required variables
if (!isset($flags) || !isset($enabled_features) || !isset($kill_switch_active) || !isset($privacy_url)) {
	return;
}
?>

<!-- Section 2: Feature Configuration -->
<div class="flx-dashboard-section flx-feature-flags">
	<?php
	$section_title = 'Feature Configuration';
	$section_icon = 'dashicons-admin-settings';
	$section_icon_color = '';
	$section_description = 'Enable or disable individual features. Features with dependencies require their parent features to be enabled first.';
	include __DIR__ . '/../_shared/section-header.php';
	?>

	<form method="post">
		<?php wp_nonce_field('flx_woo_feature_flags_nonce'); ?>

		<div class="flx-feature-cards">
			<?php foreach ($flags as $flag_name => $flag_config) : ?>
				<?php
				$is_enabled = in_array($flag_name, $enabled_features, true);
				$has_dependencies = !empty($flag_config['dependencies']);
				$dependent_features = $this->get_dependent_features($flag_name);
				?>

				<div class="flx-feature-card <?php echo $is_enabled ? 'enabled' : 'disabled'; ?>">
					<!-- Card Header -->
					<div class="flx-feature-card-header">
						<h3 class="flx-feature-card-title">
							<?php echo esc_html($flag_config['display_name']); ?>
						</h3>
						<span class="flx-feature-badge <?php echo $is_enabled ? 'enabled' : 'disabled'; ?>">
							<?php echo $is_enabled ? __('Enabled', 'flx-woo') : __('Disabled', 'flx-woo'); ?>
						</span>
					</div>

					<!-- Description -->
					<p class="flx-feature-card-description">
						<?php echo esc_html($flag_config['description']); ?>
					</p>

					<!-- Controls -->
					<div class="flx-feature-card-controls">
						<label>
							<input
								type="checkbox"
								id="flag_<?php echo esc_attr($flag_name); ?>_enabled"
								name="flag_<?php echo esc_attr($flag_name); ?>_enabled"
								value="1"
								<?php checked($flag_config['enabled_globally'], true); ?>
								<?php disabled($kill_switch_active, true); ?>
							/>
							<?php _e('Enable globally', 'flx-woo'); ?>
						</label>

						<!-- Rollout Slider (shown when NOT globally enabled) -->
						<div class="flx-rollout-slider" style="<?php echo $flag_config['enabled_globally'] ? 'display:none;' : ''; ?>">
							<div class="flx-rollout-slider-header">
								<strong><?php _e('Rollout Percentage', 'flx-woo'); ?></strong>
								<span class="flx-rollout-value"><?php echo esc_html($flag_config['rollout_percentage']); ?>%</span>
							</div>
							<input
								type="range"
								id="flag_<?php echo esc_attr($flag_name); ?>_rollout"
								name="flag_<?php echo esc_attr($flag_name); ?>_rollout"
								min="0"
								max="100"
								step="10"
								value="<?php echo esc_attr($flag_config['rollout_percentage']); ?>"
								<?php disabled($kill_switch_active, true); ?>
							/>
							<p class="description">
								<?php _e('When "Enable globally" is OFF, this percentage determines how many stores receive this feature via hash-based rollout (0-100%).', 'flx-woo'); ?>
							</p>
						</div>
					</div>

					<!-- Dependency Warning -->
					<?php if ($has_dependencies) : ?>
						<div class="flx-feature-dependency-warning">
							<strong><?php _e('⚠️ Requires:', 'flx-woo'); ?></strong>
							<?php
							$dep_names = array_map(function($dep) use ($flags) {
								return $flags[$dep]['display_name'] ?? $dep;
							}, $flag_config['dependencies']);
							echo esc_html(implode(', ', $dep_names));
							?>
						</div>
					<?php endif; ?>

					<!-- Dependents Warning (what depends on this feature) -->
					<?php if (!empty($dependent_features)) : ?>
						<div class="flx-feature-dependency-info">
							<strong><?php _e('Used by:', 'flx-woo'); ?></strong>
							<?php
							$dependent_names = array_map(function($dep) use ($flags) {
								return $flags[$dep]['display_name'] ?? $dep;
							}, $dependent_features);
							echo esc_html(implode(', ', $dependent_names));
							?>
							<?php _e('(will be disabled if you disable this feature)', 'flx-woo'); ?>
						</div>
					<?php endif; ?>

					<!-- Privacy Notice (Analytics only) -->
					<?php if ($flag_name === \FlxWoo\FeatureFlags\FeatureManager::ANALYTICS_TRACKING) : ?>
						<div class="flx-feature-privacy-notice">
							<strong><?php _e('🔒 Privacy Notice:', 'flx-woo'); ?></strong>
							<?php _e('Analytics tracking is 100% anonymous and GDPR/CCPA compliant. No personally identifiable information (PII) is collected or stored.', 'flx-woo'); ?>
							<a href="<?php echo esc_url($privacy_url); ?>" target="_blank"><?php _e('Learn more', 'flx-woo'); ?></a>
						</div>
					<?php endif; ?>
				</div>
			<?php endforeach; ?>
		</div>

		<p class="submit">
			<button type="submit" name="flx_woo_save_feature_flags" class="button button-primary">
				<?php _e('Save Feature Flags', 'flx-woo'); ?>
			</button>
		</p>
	</form>
</div>
