<?php
/**
 * Feature Card Component
 *
 * Reusable component for individual feature flag cards with controls.
 * Displays feature status, configuration controls, and dependency information.
 *
 * @package FlxWoo\Admin\Views\Components
 * @version 2.4.0
 *
 * @var string $flag_name Feature flag identifier
 * @var array $flag_config Feature flag configuration array containing:
 *   - 'display_name' (string): Human-readable feature name
 *   - 'description' (string): Feature description
 *   - 'enabled_globally' (bool): Whether enabled for all stores
 *   - 'rollout_percentage' (int): Percentage for hash-based rollout
 *   - 'dependencies' (array): Required feature dependencies
 * @var bool $is_enabled Whether feature is currently enabled
 * @var bool $has_dependencies Whether feature has required dependencies
 * @var array $dependent_features Features that depend on this feature
 * @var bool $kill_switch_active Whether global kill switch is active
 * @var string $privacy_url URL to privacy policy page
 */

if (!defined('ABSPATH')) {
	exit;
}

// Validate required variables
if (!isset($flag_name) || !isset($flag_config)) {
	return;
}

// Set defaults for optional variables
$is_enabled = $is_enabled ?? false;
$has_dependencies = $has_dependencies ?? false;
$dependent_features = $dependent_features ?? [];
$kill_switch_active = $kill_switch_active ?? false;
$privacy_url = $privacy_url ?? '';
?>

<div class="flx-feature-card <?php echo $is_enabled ? 'enabled' : 'disabled'; ?>">
	<!-- Card Header -->
	<div class="flx-feature-card-header">
		<h3 class="flx-feature-card-title">
			<?php echo esc_html($flag_config['display_name']); ?>
		</h3>
		<span class="flx-feature-badge <?php echo $is_enabled ? 'enabled' : 'disabled'; ?>">
			<?php echo $is_enabled ? __('Enabled', 'flx-woo') : __('Disabled', 'flx-woo'); ?>
		</span>
	</div>

	<!-- Description -->
	<p class="flx-feature-card-description">
		<?php echo esc_html($flag_config['description']); ?>
	</p>

	<!-- Controls -->
	<div class="flx-feature-card-controls">
		<label>
			<input
				type="checkbox"
				id="flag_<?php echo esc_attr($flag_name); ?>_enabled"
				name="flag_<?php echo esc_attr($flag_name); ?>_enabled"
				value="1"
				<?php checked($flag_config['enabled_globally'], true); ?>
				<?php disabled($kill_switch_active, true); ?>
			/>
			<?php _e('Enable globally', 'flx-woo'); ?>
		</label>

		<!-- Rollout Slider (shown when NOT globally enabled) -->
		<div class="flx-rollout-slider" style="<?php echo $flag_config['enabled_globally'] ? 'display:none;' : ''; ?>">
			<div class="flx-rollout-slider-header">
				<strong><?php _e('Rollout Percentage', 'flx-woo'); ?></strong>
				<span class="flx-rollout-value"><?php echo esc_html($flag_config['rollout_percentage']); ?>%</span>
			</div>
			<input
				type="range"
				id="flag_<?php echo esc_attr($flag_name); ?>_rollout"
				name="flag_<?php echo esc_attr($flag_name); ?>_rollout"
				min="0"
				max="100"
				step="10"
				value="<?php echo esc_attr($flag_config['rollout_percentage']); ?>"
				<?php disabled($kill_switch_active, true); ?>
			/>
			<p class="description">
				<?php _e('When "Enable globally" is OFF, this percentage determines how many stores receive this feature via hash-based rollout (0-100%).', 'flx-woo'); ?>
			</p>
		</div>
	</div>

	<!-- Dependency Warning -->
	<?php if ($has_dependencies) : ?>
		<div class="flx-feature-dependency-warning">
			<strong><?php _e('⚠️ Requires:', 'flx-woo'); ?></strong>
			<?php
			// Access $flags from parent scope
			global $flags;
			$dep_names = array_map(function($dep) use ($flags) {
				return $flags[$dep]['display_name'] ?? $dep;
			}, $flag_config['dependencies']);
			echo esc_html(implode(', ', $dep_names));
			?>
		</div>
	<?php endif; ?>

	<!-- Dependents Warning (what depends on this feature) -->
	<?php if (!empty($dependent_features)) : ?>
		<div class="flx-feature-dependency-info">
			<strong><?php _e('Used by:', 'flx-woo'); ?></strong>
			<?php
			// Access $flags from parent scope
			global $flags;
			$dependent_names = array_map(function($dep) use ($flags) {
				return $flags[$dep]['display_name'] ?? $dep;
			}, $dependent_features);
			echo esc_html(implode(', ', $dependent_names));
			?>
			<?php _e('(will be disabled if you disable this feature)', 'flx-woo'); ?>
		</div>
	<?php endif; ?>

	<!-- Privacy Notice (Analytics only) -->
	<?php if ($flag_name === \FlxWoo\FeatureFlags\FeatureManager::ANALYTICS_TRACKING) : ?>
		<div class="flx-feature-privacy-notice">
			<strong><?php _e('🔒 Privacy Notice:', 'flx-woo'); ?></strong>
			<?php _e('Analytics tracking is 100% anonymous and GDPR/CCPA compliant. No personally identifiable information (PII) is collected or stored.', 'flx-woo'); ?>
			<a href="<?php echo esc_url($privacy_url); ?>" target="_blank"><?php _e('Learn more', 'flx-woo'); ?></a>
		</div>
	<?php endif; ?>
</div>
